/*
 * Copyright (c) 2015 Silicon Craft Technology Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.sic.module.nfc.tech.chips.registers;

import com.sic.module.nfc.tech.chips.SIC43xx;
import com.sic.module.nfc.tech.interfaces.IDevice;
import com.sic.module.utils.SICLog;

/**
 * @author Tanawat Hongthai - http://www.sic.co.th/
 * @version 1.0.1
 * @since 25/Nov/2015
 */
public abstract class IGPIO extends IDevice {

    private static final String TAG = IGPIO.class.getName();
    protected static byte ADDRESS_GPIO_MODE_0;
    protected static byte ADDRESS_GPIO_MODE_1;
    protected static byte ADDRESS_GPIO_MODE_2;
    protected static byte ADDRESS_GPIO_OUT;
    protected static byte ADDRESS_GPIO_IN;

    public static final int Address_Mode_0 = 0;
    public static final int Address_Mode_1 = 1;
    public static final int Address_Mode_2 = 2;

    public static final int Pin_0 = BIT0;
    public static final int Pin_1 = BIT1;
    public static final int Pin_2 = BIT2;
    public static final int Pin_3 = BIT3;
    public static final int Pin_4 = BIT4;
    public static final int Pin_5 = BIT5;
    public static final int Pin_6 = BIT6;
    public static final int Pin_7 = BIT7;
    public static final int Pin_LNib = BIT0 | BIT1 | BIT2 | BIT3;
    public static final int Pin_HNib = BIT4 | BIT5 | BIT6 | BIT7;
    public static final int Pin_All = Pin_LNib | Pin_HNib;

    public static int Pin_Power_Ready;
    public static int Pin_RF_Busy;
    public static int Pin_RF_Detected;

    private int[] ADDRESS_GPIO_MODE;

    protected IGPIO(SIC43xx sic43xx) {
        super(sic43xx.getRegister());
        ADDRESS_GPIO_MODE = new int[3];
        ADDRESS_GPIO_MODE[0] = ADDRESS_GPIO_MODE_0;
        ADDRESS_GPIO_MODE[1] = ADDRESS_GPIO_MODE_1;
        ADDRESS_GPIO_MODE[2] = ADDRESS_GPIO_MODE_2;
    }

    /**
     * Gets the GPIO mode according to the specified parameters.
     *
     * @param  GPIO_Address_Mode: Select the address mode
     *           This parameter can be one of the following values:
     *            {@link #Address_Mode_0}: Address mode 0
     *            {@link #Address_Mode_1}: Address mode 1
     *            {@link #Address_Mode_2}: Address mode 2
     *
     * @return  byte GPIO mode data port value.
     */
    public Byte getMode(int GPIO_Address_Mode) {
        return mRegister.readBuffer(ADDRESS_GPIO_MODE[GPIO_Address_Mode]);
    }

    /**
     * Sets the GPIO mode according to the specified parameters.
     *
     * @param  GPIO_Address_Mode: Select the address mode
     *           This parameter can be one of the following values:
     *            {@link #Address_Mode_0}: Address mode 0
     *            {@link #Address_Mode_1}: Address mode 1
     *            {@link #Address_Mode_2}: Address mode 2
     *
     * @param  GPIO_Pin_Mode: Specifies the value to be set to the gpio mode register.
     */
    public void setMode(int GPIO_Address_Mode, int GPIO_Pin_Mode) {
        mRegister.write(ADDRESS_GPIO_MODE[GPIO_Address_Mode], GPIO_Pin_Mode);
    }

    /**
     * Sets the GPIO mode bit according to the specified parameters.
     *
     * @param  GPIO_Address_Mode: Select the address mode
     *           This parameter can be one of the following values:
     *            {@link #Address_Mode_0}: Address mode 0
     *            {@link #Address_Mode_1}: Address mode 1
     *            {@link #Address_Mode_2}: Address mode 2
     *
     * @param  GPIO_Pin: Specifies the port bit to be set.
     *          This parameter can be any combination of the following values:
     *            {@link #Pin_0}: Pin 0
     *            {@link #Pin_1}: Pin 1
     *            {@link #Pin_2}: Pin 2
     *            {@link #Pin_3}: Pin 3
     *            {@link #Pin_4}: Pin 4
     *            {@link #Pin_5}: Pin 5
     *            {@link #Pin_6}: Pin 6
     *            {@link #Pin_7}: Pin 7
     *            {@link #Pin_LNib}: Low nibble pins
     *            {@link #Pin_HNib}: High nibble pins
     *            {@link #Pin_All}: All pins
     *
     * @param  GPIO_Pin_Mode: Specifies the value to be set to the gpio mode register.
     *          This parameter can be {@link #High_Level} or {@link #Low_Level}
     */
    public void setModeBit(int GPIO_Address_Mode, int GPIO_Pin, int GPIO_Pin_Mode) {
        byte mode = mRegister.readBuffer(ADDRESS_GPIO_MODE[GPIO_Address_Mode]);
        if (GPIO_Pin_Mode == Low_Level) {
            mode &= ~GPIO_Pin;
        } else {
            mode |= GPIO_Pin;
        }
        mRegister.write(ADDRESS_GPIO_MODE[GPIO_Address_Mode], mode);
    }

    /**
     * gets the specified GPIO output data port.
     * The port must be configured in output mode.
     *
     * @return  byte GPIO output data port value.
     */
    public byte getOutput() {
        return mRegister.readBuffer(ADDRESS_GPIO_OUT);
    }

    /**
     * Sets data to the specified GPIO data port.
     * The port must be configured in output mode.
     *
     * @param  GPIO_PortVal: Specifies the value to be set to the port output data register.
     */
    public void setOutput(int GPIO_PortVal) {
        mRegister.write(ADDRESS_GPIO_OUT, GPIO_PortVal);
    }

    /**
     * Sets or clears the selected data port bit.
     * @param  GPIO_Pin: Specifies the port bit to be set.
     *          This parameter can be any combination of the following values:
     *            {@link #Pin_0}: Pin 0
     *            {@link #Pin_1}: Pin 1
     *            {@link #Pin_2}: Pin 2
     *            {@link #Pin_3}: Pin 3
     *            {@link #Pin_4}: Pin 4
     *            {@link #Pin_5}: Pin 5
     *            {@link #Pin_6}: Pin 6
     *            {@link #Pin_7}: Pin 7
     *            {@link #Pin_LNib}: Low nibble pins
     *            {@link #Pin_HNib}: High nibble pins
     *            {@link #Pin_All}: All pins
     *
     * @param  GPIO_BitVal: specifies the desired status to be set.
     *          This parameter can be {@link #High_Level} or {@link #Low_Level}
     */
    public void setOutputBit(int GPIO_Pin, int GPIO_BitVal) {
        byte output = mRegister.readBuffer(ADDRESS_GPIO_OUT);
        if (GPIO_BitVal == Low_Level) {
            output &= ~GPIO_Pin;
        } else {
            output |= GPIO_Pin;
        }
        mRegister.write(ADDRESS_GPIO_OUT, output);
    }

    /**
     * Sets the specified GPIO input data port.
     * The port must be configured in input mode.
     *
     * @return  byte GPIO input data port value.
     */
    public Byte getInput() {
        Byte recv = mRegister.read(ADDRESS_GPIO_IN);

        if (recv == null) {
            SICLog.e(TAG, "Can not get GPIO input.");
        }
        return recv;
    }

    /**
     * Gets the specified GPIO input data pin.
     *  The pin must be configured in input mode.
     *
     * @param  GPIO_Pin: This parameter contains the pin number
     *          This parameter can be any combination of the following values:
     *            {@link #Pin_0}: Pin 0
     *            {@link #Pin_1}: Pin 1
     *            {@link #Pin_2}: Pin 2
     *            {@link #Pin_3}: Pin 3
     *            {@link #Pin_4}: Pin 4
     *            {@link #Pin_5}: Pin 5
     *            {@link #Pin_6}: Pin 6
     *            {@link #Pin_7}: Pin 7
     *            {@link #Pin_LNib}: Low nibble pins
     *            {@link #Pin_HNib}: High nibble pins
     *            {@link #Pin_All}: All pins
     *
     * @return  BitStatus GPIO input pin status.
     */
    public Byte getInputBit(int GPIO_Pin) {
        Byte recv = mRegister.read(ADDRESS_GPIO_IN);
        if (recv == null) {
            SICLog.e(TAG, "Can not get GPIO input.");
            return null;
        }

        if ((recv & GPIO_Pin) == GPIO_Pin) {
            return High_Level;
        } else {
            return Low_Level;
        }
    }

    /**
     * Set high level to the specified GPIO pins.
     * The pin must be configured in output mode.
     *
     * @param  GPIO_Pin: Specifies the pins to be turned high.
     *          This parameter can be any combination of the following values:
     *            {@link #Pin_0}: Pin 0
     *            {@link #Pin_1}: Pin 1
     *            {@link #Pin_2}: Pin 2
     *            {@link #Pin_3}: Pin 3
     *            {@link #Pin_4}: Pin 4
     *            {@link #Pin_5}: Pin 5
     *            {@link #Pin_6}: Pin 6
     *            {@link #Pin_7}: Pin 7
     *            {@link #Pin_LNib}: Low nibble pins
     *            {@link #Pin_HNib}: High nibble pins
     *            {@link #Pin_All}: All pins
     */
    public void setHighOutput(int GPIO_Pin) {
        setOutputBit(GPIO_Pin, High_Level);
    }

    /**
     * Set low level to the specified GPIO pins.
     * The pin must be configured in output mode.
     *
     * @param  GPIO_Pin: Specifies the pins to be turned low.
     *          This parameter can be any combination of the following values:
     *            {@link #Pin_0}: Pin 0
     *            {@link #Pin_1}: Pin 1
     *            {@link #Pin_2}: Pin 2
     *            {@link #Pin_3}: Pin 3
     *            {@link #Pin_4}: Pin 4
     *            {@link #Pin_5}: Pin 5
     *            {@link #Pin_6}: Pin 6
     *            {@link #Pin_7}: Pin 7
     *            {@link #Pin_LNib}: Low nibble pins
     *            {@link #Pin_HNib}: High nibble pins
     *            {@link #Pin_All}: All pins
     */
    public void setLowOutput(int GPIO_Pin) {
        setOutputBit(GPIO_Pin, Low_Level);
    }

    /**
     * Toggles the specified GPIO pins.
     * The pin must be configured in output mode.
     *
     * @param  GPIO_Pin: Specifies the pins to be toggled data register.
     *          This parameter can be any combination of the following values:
     *            {@link #Pin_0}: Pin 0
     *            {@link #Pin_1}: Pin 1
     *            {@link #Pin_2}: Pin 2
     *            {@link #Pin_3}: Pin 3
     *            {@link #Pin_4}: Pin 4
     *            {@link #Pin_5}: Pin 5
     *            {@link #Pin_6}: Pin 6
     *            {@link #Pin_7}: Pin 7
     *            {@link #Pin_LNib}: Low nibble pins
     *            {@link #Pin_HNib}: High nibble pins
     *            {@link #Pin_All}: All pins
     */
    public void setToggleOutput(int GPIO_Pin) {
        byte output = getOutput();
        byte sel = (byte) (GPIO_Pin & ~output);
        byte uns = (byte) (~GPIO_Pin & output);
        setOutput((byte) (sel | uns));
    }
}
