/*
 * Copyright (c) 2015 Silicon Craft Technology Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.sic.module.nfc;

import android.app.Activity;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.nfc.NfcAdapter;
import android.nfc.Tag;
import android.os.Build;
import android.provider.Settings;
import android.widget.Toast;

import com.sic.module.utils.Boast;
import com.sic.module.utils.SICLog;

/**
 * @author Tanawat Hongthai - http://www.sic.co.th/
 * @version 1.0.2
 * @since 16/Oct/2015
 */
public abstract class Nfc {

    private static final String TAG = Nfc.class.getName();

    private final IntentFilter[] mIntentFilters;
    private final String[][] mTechLists;

    private final Context mContext;
    private final NfcAdapter mNfcAdapter;

    private Tag tag;
    private volatile NFC_STATE mState = NFC_STATE.PAUSED;
    private volatile NFC_MODE mInterfaceMode = NFC_MODE.PASSTHROUGH;

    protected Nfc(Context context, IntentFilter[] intentFilters, String[][] techLists) {
        mContext = context;
        mNfcAdapter = NfcWrapper.getInstance().getAdapter(mContext);
        mIntentFilters = intentFilters;
        mTechLists = techLists;

        checkNfcAdapter();
    }

    /**
     * Call this method in your Activity's onResume() method body.
     *
     * @param activity locale activity
     */
    public void onResume(Activity activity) {
        mState = NFC_STATE.RESUMING;

        if (mNfcAdapter != null) {
            synchronized (this) {
                installNfcHandler(activity);
            }
        }
        mState = NFC_STATE.RESUMED;
    }

    /**
     * Call this method in your Activity's onPause() method body.
     *
     * @param activity locale activity
     */
    public void onPause(Activity activity) {
        mState = NFC_STATE.PAUSING;

        if (mNfcAdapter != null) {
            synchronized (this) {
                mNfcAdapter.disableForegroundDispatch(activity);
            }
        }
        mState = NFC_STATE.PAUSED;
    }

    /**
     * Call this method in your activity's onNewIntent(Intent) method body.
     *
     * @param intent locale intent
     * @return true if NFC tag is found.
     */
    public boolean onTagFinding(Intent intent) {
        // Check to see if the intent is ours to handle:
        mInterfaceMode = NFC_MODE.TAG_FINDING;
        new TagHandlerThread(intent).start();
        try {
            while (mInterfaceMode == NFC_MODE.TAG_FINDING) {
                Thread.sleep(1);
            }
        } catch (InterruptedException ignored) {
        }
        return mInterfaceMode == NFC_MODE.EXCHANGE;
    }

    /**
     * Put the interface in mode {NFC_MODE.PASSTHROUGH}.
     */
    public void disable() {
        mInterfaceMode = NFC_MODE.PASSTHROUGH;

        synchronized (this) {
            try {
                if (mState == NFC_STATE.PAUSING || mState == NFC_STATE.PAUSED) {
                    return;
                }
                mNfcAdapter.disableForegroundDispatch((Activity) mContext);
            } catch (IllegalStateException ignored) {

            } catch (NullPointerException ignored) {
                Boast.showToast((Activity) mContext, "Your phone is not support a NFC Technology.");
            }
        }

    }

    /**
     * Check if NFC tag is found.
     *
     * @return true, if NFC tag is found.
     */
    public boolean isNfcTag() {
        try {
            while (mInterfaceMode == NFC_MODE.TAG_FINDING) {
                Thread.sleep(1);
            }
        } catch (InterruptedException ignored) {
        }
        return tag != null;
    }

    private void checkNfcAdapter() {
        if (mNfcAdapter == null) {
            SICLog.i(TAG, "Nfc implementation not available.");
        } else if (!(mNfcAdapter.isEnabled())) {
            if (mContext != null) {
                Toast.makeText(mContext, "Please activate NFC and press Back to return to the application!", Toast.LENGTH_LONG).show();
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
                    Intent intent = new Intent(Settings.ACTION_NFC_SETTINGS);
                    intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                    mContext.startActivity(intent);
                } else {
                    Intent intent = new Intent(Settings.ACTION_WIRELESS_SETTINGS);
                    intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                    mContext.startActivity(intent);
                }
            }

        }
    }

    /**
     * Requests any foreground NFC activity. This method must be called from the
     * main thread.
     *
     * @param activity locale activity
     */
    private void installNfcHandler(Activity activity) {
        Intent activityIntent = new Intent(mContext, activity.getClass());
        activityIntent.addFlags(Intent.FLAG_ACTIVITY_SINGLE_TOP);

        PendingIntent intent = PendingIntent.getActivity(mContext, 0,
                activityIntent, PendingIntent.FLAG_CANCEL_CURRENT);
        mNfcAdapter.enableForegroundDispatch(activity, intent,
                mIntentFilters, mTechLists);
    }

    public Tag getTag() {
        return tag;
    }

    /**
     * Get the Tag Identifier (if it has one).
     *
     * @return UID as byte array, never null
     */
    public byte[] getUID() {
        if (tag == null) {
            return null;
        }
        return tag.getId();
    }

    private enum NFC_STATE {
        PAUSED, PAUSING, RESUMING, RESUMED
    }

    private enum NFC_MODE {
        // Nfc interface mode in which Nfc interaction is disabled for this class.
        PASSTHROUGH,
        // Nfc interface mode for finding a tag
        TAG_FINDING,
        // Nfc interface mode for exchanging information with another active device.
        EXCHANGE
    }

    private class TagHandlerThread extends Thread {
        final Intent mIntent;

        TagHandlerThread(Intent intent) {
            mIntent = intent;
        }

        @Override
        public void run() {
            tag = mIntent.getParcelableExtra(NfcAdapter.EXTRA_TAG);
            if (tag != null) {
                mInterfaceMode = NFC_MODE.EXCHANGE;
            } else {
                disable();
            }
        }
    }
}
