/*
 * Copyright (c) 2015 Silicon Craft Technology Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.sic.module.nfc.tech.mandatories;

import android.content.Context;
import android.content.Intent;
import android.nfc.tech.NfcA;

import com.sic.module.nfc.Flag;
import com.sic.module.nfc.Nfc;
import com.sic.module.nfc.tech.interfaces.ICommand;
import com.sic.module.utils.SICLog;
import com.sic.module.utils.Utils;

import java.io.IOException;

/**
 * @author Tanawat Hongthai - http://www.sic.co.th/
 * @version 1.0.1
 * @since 10/16/2015
 */
public class NfcTypeA extends Nfc {

    private static final String TAG = NfcTypeA.class.getName();

    private static final String[] TECH_LISTS = new String[]{NfcA.class.getName()};

    private static NfcTypeA instance;
    private final Context mContext;
    private NfcA nfcA;
    private boolean criticalState = false;
    private boolean e2CriticalState = false;
    private boolean commandCriticalState = false;
    private int timeout = 500;
    private byte responseFlag;
    private byte[] receiveData;

    private NfcTypeA(Context context) {
        super(context, null, new String[][]{TECH_LISTS});
        mContext = context;
    }

    public static NfcTypeA getInstance(Context context) {
        if (instance == null)
            instance = new NfcTypeA(context);
        return instance;
    }

    public static String getFlagName(byte flag) {
        switch (flag) {
            case Flag.NFC_ACK:
                return "NFC ACK";
            case Flag.NFC_NAK:
                return "NFC NAK";
            case Flag.EXCEPTION:
                return "Exception";
            case Flag.DISAPPEAR:
                return "Tag lost";
            case Flag.UNKNOWN:
            default:
                return " - ";
        }
    }

    public Context getContext() {
        return mContext;
    }

    public boolean isConnected() {
        try {
            nfcA.isConnected();
            return true;
        } catch (Exception e) {
            return false;
        }
    }

    @Override
    public boolean onTagFinding(Intent intent) {
        if (super.onTagFinding(intent)) {
            nfcA = NfcA.get(getTag());
            return true;
        }
        return false;
    }

    public synchronized byte[] transceive(byte[] data) throws IOException {
        return nfcA.transceive(data);
    }

    /**
     * NFC connection software service.
     *
     * @throws IOException occurs if tag is not found.
     */
    public void connect() throws IOException {
        nfcA.close();
        nfcA.connect();
        nfcA.setTimeout(timeout);
    }

    /**
     * NFC disconnects the software service.
     *
     * @throws IOException occurs if tag is not found.
     */
    public void close() throws IOException {
        nfcA.close();
    }

    /**
     * NFC transceive function connects and closes NFC software service
     * automatically.
     *
     * @param send_data raw data sent from device.
     * @return data received from NFC.
     */
    public synchronized byte[] autoTransceive(byte[] send_data) {
        try {
            while (criticalState) {
                Thread.sleep(1);
            }
        } catch (InterruptedException ignored) {
        }
        criticalState = true;
        byte[] receive_data = null;
        try {
            connect();
            receive_data = nfcA.transceive(send_data);
        } catch (Exception e) {
            if (e.getMessage() != null) {
                SICLog.e(TAG, e.getMessage());
                if (e.getMessage().contains("Transceive failed")) {
                    responseFlag = Flag.EXCEPTION;
                } else {
                    responseFlag = Flag.DISAPPEAR;
                }
            } else {
                responseFlag = Flag.DISAPPEAR;
            }
        }

        criticalState = false;
        return receive_data;
    }

    public synchronized byte[][] autoTransceive(byte[]... multi_data) {
        try {
            while (criticalState) {
                Thread.sleep(1);
            }
        } catch (InterruptedException ignored) {
        }
        criticalState = true;
        int index = 0;
        byte[][] receive_data = new byte[multi_data.length][];
        try {
            connect();
            for (byte[] data : multi_data) {
                receive_data[index++] = nfcA.transceive(data);
            }
        } catch (Exception e) {
            if (e.getMessage() != null) {
                SICLog.e(TAG, e.getMessage());
                if (e.getMessage().contains("Transceive failed")) {
                    responseFlag = Flag.EXCEPTION;
                } else {
                    responseFlag = Flag.DISAPPEAR;
                }
            } else {
                responseFlag = Flag.DISAPPEAR;
            }
        }
        criticalState = false;
        return receive_data;
    }

    /**
     * This function is used for general commands and raw data. WARNING!! the
     * commands related to NFC states can not be used in Android device.
     *
     * @param command general commands for NFC Type A
     * @param data    raw data sent from device.
     * @return data received from NFC.
     */
    public byte[] commandTransceive(ICommand command, byte[] data) {
        byte[][] recv = commandTransceive(command, new byte[][]{data});
        if (recv == null || recv.length == 0) {
            return null;
        } else {
            return recv[0];
        }
    }

    /**
     * This function is used for general commands and raw data. WARNING!! the
     * commands related to NFC states can not be used in Android device.
     *
     * @param command    general commands for NFC Type A
     * @param multi_data raw data sent from device.
     * @return data received from NFC.
     */
    public byte[][] commandTransceive(ICommand command, byte[]... multi_data) {
        byte[][] receive = null;
        byte[][] buffer = new byte[multi_data.length][];
        int index;

        if (!isConnected() || !isNfcTag()) {
            SICLog.e(TAG, "Block data transmission");
            commandCriticalState = false;
            return null;
        }

        if (buffer.length == 0) {
            SICLog.i(TAG, "Please check data input.");
            commandCriticalState = false;
            return null;
        }
        try {
            while (commandCriticalState) {
                Thread.sleep(1);
            }
        } catch (InterruptedException ignored) {
        }
        commandCriticalState = true;

        index = 0;
        for (byte[] data : multi_data) {
            buffer[index] = command.getBuffer(data);
            if (buffer[index] == null) {
                SICLog.i(TAG, "Buffer[-]: ");

            } else {
                SICLog.i(TAG, "Buffer[" + buffer[index].length + "]: " + Utils.parseByteArrayToHexString(buffer[index]));

            }
            index++;
        }

        resetFlag();
        e2CriticalState = false;

        try {
            if (command.equals(MFCommand.COMPATIBILITY_WRITE_EEPROM)) {
                e2CriticalState = true;
                byte[] buffer_address = MFCommand.COMPATIBILITY_WRITE_EEPROM_ADDRESS
                        .getBuffer(multi_data[0]);
                byte[] buffer_data = MFCommand.COMPATIBILITY_WRITE_EEPROM_DATA
                        .getBuffer(multi_data[1]);
                receive = autoTransceive(buffer_address, buffer_data);

            } else if (command.equals(MFCommand.WRITE_EEPROM)) {
                e2CriticalState = true;
                receive = autoTransceive(buffer);

            } else if (command.equals(MFCommand.READ_EEPROM)) {
                e2CriticalState = true;
                receive = autoTransceive(buffer);

            } else {
                receive = autoTransceive(buffer);
            }

            for (byte[] data : receive) {
                if (data != null && data.length > 0) {
                    SICLog.i(TAG, "Receive[" + data.length + "]: " + Utils.parseByteArrayToHexString(data));
                } else {
                    SICLog.i(TAG, "Receive[" + 0 + "]: NULL");
                }
            }
            checkLatestReceiver(receive);
        } catch (Exception e) {
            // e.printStackTrace();
            responseFlag = Flag.DISAPPEAR;
            SICLog.e(TAG, "NFC communicate exception: " + e.getLocalizedMessage());
        }

        commandCriticalState = false;
        return receive;
    }

    private void checkLatestReceiver(byte[]... buffer) {
        int lastData = buffer.length - 1;

        if (lastData == -1) {
            SICLog.e(TAG, "Data receiver is null pointer exception.");
            responseFlag = Flag.EXCEPTION;
            receiveData = new byte[0];
            return;
        }

        receiveData = new byte[buffer[lastData].length];
        if (e2CriticalState) {
            responseFlag = Flag.UNKNOWN;

            if (receiveData.length == 1) {
                responseFlag = buffer[lastData][0];
                receiveData[0] = buffer[lastData][0];
            } else if (receiveData.length > 1) {
                System.arraycopy(buffer[lastData], 0, receiveData, 0, receiveData.length);
            }
            e2CriticalState = false;

        } else if (receiveData.length > 1) {
            responseFlag = buffer[lastData][0];
            System.arraycopy(buffer[lastData], 0, receiveData, 0, receiveData.length);

        } else {
            responseFlag = Flag.DISAPPEAR;

        }

//        if (responseFlag == Flag.RESET) {
//            responseFlag = Flag.EXCEPTION;
//            receiveData = new byte[0];
//            return;
//        }
    }

    /**
     * Clean NFC flag.
     */
    private void resetFlag() {
        responseFlag = Flag.RESET;
    }

    /**
     * Check if the last transmission is successful.
     *
     * @return true if flag is not NAK.
     */
    public boolean isSendCompleted() {
        return responseFlag != Flag.NFC_NAK && responseFlag != Flag.DISAPPEAR && responseFlag != Flag.EXCEPTION && isConnected();
    }

    protected boolean checkCriticalState() {
        return criticalState | commandCriticalState;
    }

    public int getTimeout() {
        return timeout;
    }

    public void setTimeout(int timeout) {
        this.timeout = timeout;
        if (nfcA != null) {
            nfcA.setTimeout(timeout);
        }
    }

    public byte getResponseFlag() {
        return responseFlag;
    }

    public byte[] getReceiverData() {
        return receiveData;
    }

    /**
     * Get ATQA of tag.
     *
     * @return byte array of ATQA.
     */
    public byte[] getAtqa() {
        return nfcA.getAtqa();
    }

    /**
     * Get max length transceiver of Tag.
     *
     * @return max length transceiver.
     */
    public int getMaxTransceiveLength() {
        return nfcA.getMaxTransceiveLength();
    }

    /**
     * Get SAK of tag.
     *
     * @return SAK.
     */
    public short getSak() {
        return nfcA.getSak();
    }

    public NfcA getRawNfcA() {
        return nfcA;
    }
}
