/*
 * Copyright (c) 2015 Silicon Craft Technology Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.sic.module.nfc.tech.mandatories;

import com.sic.module.nfc.tech.interfaces.ICommand;

/**
 * @author Tanawat Hongthai - http://www.sic.co.th/
 * @version 1.0.0
 * @since 3/27/2015
 */
public enum MFCommand implements ICommand {

    /**
     * This command has 2 transactions between reader and ASIC. Authentication
     * command
     */
    AUTHENTICATION(4110) {
        @Override
        public byte[] getBuffer(byte[] unused) {
            return new byte[]{0x00};
        }
    },

    /**
     * Authentication command format 1 (0x60)
     */
    AUTHENTICATION_ACCESS(4111) {
        @Override
        public byte[] getBuffer(byte[] data) {
            if (checkDataBuffer(data, 1, data.length)) {
                byte[] buffer = new byte[2];
                buffer[0] = (byte) 0x60;
                buffer[1] = data[0];
                return buffer;
            }
            return null;
        }
    },

    /**
     * Authentication command format 1 (0x61)
     */
    AUTHENTICATION_ACCESS_OPTIONAL(4112) {
        @Override
        public byte[] getBuffer(byte[] data) {
            if (checkDataBuffer(data, 1, data.length)) {
                byte[] buffer = new byte[2];
                buffer[0] = (byte) 0x61;
                buffer[1] = data[0];
                return buffer;
            }
            return null;
        }
    },

    /**
     * Authentication command format 2
     */
    AUTHENTICATION_ENCRYPTED_DATA(4113) {
        @Override
        public byte[] getBuffer(byte[] data) {
            if (checkDataBuffer(data, 8, data.length)) {
                byte[] buffer = new byte[8];
                System.arraycopy(data, 0, buffer, 0, 8);
                return buffer;
            }
            return null;
        }
    },

    /**
     * The purpose of ReadE2 command is to read EEPROM content. The ReadE2
     * command contains a page address with a valid CRC.
     */
    READ_EEPROM(4120) {
        @Override
        public byte[] getBuffer(byte[] data) {
            if (checkDataBuffer(data, 1, data.length)) {
                byte[] buffer = new byte[2];
                buffer[0] = (byte) 0x30;
                buffer[1] = data[0];
                return buffer;
            }
            return null;
        }
    },

    /**
     * The purpose of WriteE2 command is to write the EEPROM, program lock bits,
     * set bits in OTP byte and preset initial register value.
     */
    WRITE_EEPROM(4130) {
        @Override
        public byte[] getBuffer(byte[] data) {
            if (checkDataBuffer(data, 5, data.length)) {
                byte[] buffer = new byte[6];
                buffer[0] = (byte) 0xA2;
                System.arraycopy(data, 0, buffer, 1, 5);
                return buffer;
            }
            return null;
        }
    },

    /**
     * The purpose of Compatible WriteE2 command is to make programming process
     * backward compatible with MIFARE classic system. The command contains page
     * address with a CRC.
     */
    COMPATIBILITY_WRITE_EEPROM(4140) {
        @Override
        public byte[] getBuffer(byte[] unused) {
            return new byte[]{0x00};
        }
    },

    /**
     * a valid address of compatible WriteE2 command.
     */
    COMPATIBILITY_WRITE_EEPROM_ADDRESS(4141) {
        @Override
        public byte[] getBuffer(byte[] data) {
            if (checkDataBuffer(data, 1, data.length)) {
                byte[] buffer = new byte[2];
                buffer[0] = (byte) 0xA0;
                buffer[1] = data[0];
                return buffer;
            }
            return null;
        }
    },

    /**
     * 16-byte data but only the first 4 bytes is written into the memory.
     */
    COMPATIBILITY_WRITE_EEPROM_DATA(4142) {
        @Override
        public byte[] getBuffer(byte[] data) {
            if (checkDataBuffer(data, 16, data.length)) {
                byte[] buffer = new byte[16];
                System.arraycopy(data, 0, buffer, 0, 16);
                return buffer;
            }
            return null;

        }
    },

    /**
     * This command requires data format as value block to operate. The data in
     * buffer is written to EEPROM when Transfer command is executed.
     */
    RESTORE(4150) {
        @Override
        public byte[] getBuffer(byte[] unused) {
            return new byte[]{0x00};
        }
    },

    /**
     * a valid address of restore command.
     */
    RESTORE_ADDRESS(4151) {
        @Override
        public byte[] getBuffer(byte[] data) {
            if (checkDataBuffer(data, 1, data.length)) {
                byte[] buffer = new byte[2];
                buffer[0] = (byte) 0xC2;
                buffer[1] = data[0];
                return buffer;
            }
            return null;
        }
    },

    /**
     * 4-byte data is written into the memory.
     */
    RESTORE_DATA(4152) {
        @Override
        public byte[] getBuffer(byte[] data) {
            if (checkDataBuffer(data, 4, data.length)) {
                byte[] buffer = new byte[4];
                System.arraycopy(data, 0, buffer, 0, 4);
                return buffer;
            }
            return null;

        }
    },

    /**
     * This command requires data format as value block to operate. The data in
     * buffer is written to EEPROM when Transfer command is executed.
     */
    INCREMENT(4160) {
        @Override
        public byte[] getBuffer(byte[] unused) {
            return new byte[]{0x00};
        }
    },

    /**
     * a valid address of restore command.
     */
    INCREMENT_ADDRESS(4161) {
        @Override
        public byte[] getBuffer(byte[] data) {
            if (checkDataBuffer(data, 1, data.length)) {
                byte[] buffer = new byte[2];
                buffer[0] = (byte) 0xC1;
                buffer[1] = data[0];
                return buffer;
            }
            return null;
        }
    },

    /**
     * 4-byte data is written into the memory.
     */
    INCREMENT_DATA(4162) {
        @Override
        public byte[] getBuffer(byte[] data) {
            if (checkDataBuffer(data, 4, data.length)) {
                byte[] buffer = new byte[4];
                System.arraycopy(data, 0, buffer, 0, 4);
                return buffer;
            }
            return null;

        }
    },

    /**
     * This command requires data format as value block to operate. The data in
     * buffer is written to EEPROM when Transfer command is executed.
     */
    DECREMENT(4170) {
        @Override
        public byte[] getBuffer(byte[] unused) {
            return new byte[]{0x00};
        }
    },

    /**
     * a valid address of restore command.
     */
    DECREMENT_ADDRESS(4171) {
        @Override
        public byte[] getBuffer(byte[] data) {
            if (checkDataBuffer(data, 1, data.length)) {
                byte[] buffer = new byte[2];
                buffer[0] = (byte) 0xC0;
                buffer[1] = data[0];
                return buffer;
            }
            return null;
        }
    },

    /**
     * 4-byte data is written into the memory.
     */
    DECREMENT_DATA(4172) {
        @Override
        public byte[] getBuffer(byte[] data) {
            if (checkDataBuffer(data, 4, data.length)) {
                byte[] buffer = new byte[4];
                System.arraycopy(data, 0, buffer, 0, 4);
                return buffer;
            }
            return null;

        }
    },
    /**
     * Transfer command
     */
    TRANSFER(4180) {
        @Override
        public byte[] getBuffer(byte[] data) {
            if (checkDataBuffer(data, 1, data.length)) {
                byte[] buffer = new byte[2];
                buffer[0] = (byte) 0x30;
                buffer[1] = data[0];
                return buffer;
            }
            return null;
        }
    };

    private final int commandID;

    /**
     * the function prototype to create function.
     *
     * @param commandID ID of each command
     */
    MFCommand(int commandID) {
        this.commandID = commandID;
    }

    /**
     * get ID of each command
     *
     * @return ID
     */
    public int getID() {
        return commandID;
    }

    public abstract byte[] getBuffer(byte[] data);


    public boolean checkDataBuffer(byte[] data, int min_size, int max_size) {
        return data != null && !(min_size < 0 || data.length < min_size || data.length > max_size);
    }

}