/*
 * Copyright (c) 2015 Silicon Craft Technology Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.sic.module.nfc.tech.chips.s431x;

import android.util.Log;

import com.sic.module.nfc.tech.interfaces.IRegister;
import com.sic.module.utils.Buffer;

/**
 * @author Tanawat Hongthai - http://www.sic.co.th/
 * @version 1.0.2
 * @since 7/14/2015
 */
public abstract class xRegister implements IRegister {

    // Response Flag (RF - UART, RF - RegPage)
    public static final byte FLAG_B_ACK = 0x1A;
    public static final byte FLAG_B_NAK = (byte) 0x80;
    // Data Bit Clear Flag
    public static final byte FLAG_C_FLUSH_TX_FIFO = 0x01;
    public static final byte FLAG_C_FLUSH_RX_FIFO = 0x02;
    public static final byte FLAG_C_CLEAR_PWR_LOW_FLAG = 0x04;
    public static final byte FLAG_C_CLEAR_XVDD_DROP_FLAG = 0x08;
    public static final byte FLAG_C_CLEAR_OSC_STOP_FLAG = 0x10;
    // Response NACK
    public static final byte FLAG_B_NAK_DL_FIFO_OVF = 0x01;
    public static final byte FLAG_B_NAK_UL_FIFO_EMP = 0x02;
    public static final byte FLAG_B_NAK_UL_FIFO_OVF = 0x04;
    public static final byte FLAG_B_NAK_RSPW_LOW = 0x08;
    public static final byte FLAG_B_NAK_XVDD_DROP = 0x10;
    public static final byte FLAG_B_NAK_UART_FAIL = 0x20;
    private static final String TAG = xRegister.class.getSimpleName();
    private final SIC431x mSIC431x;
    private final Buffer mRegBuffer;

    public xRegister() {
        mSIC431x = SIC431x.getInstance();
        mRegBuffer = Buffer.newInstance(getRegisterPage());
    }

    protected abstract int getRegisterPage();

    public boolean sync() {
        int length = getRegisterPage();
        byte[] address = new byte[length];
        for (int i = 0; i < length; ++i) {
            address[i] = (byte) i;
        }

        return read(address) != null;
    }


    public boolean isSync() {
        return mRegBuffer.isSyncAll();
    }

    /**
     * Configure register function of SIC431x.
     *
     * @param addr address of register.
     * @return data of register.
     */
    public Byte read(byte addr) {
        byte[] recv = mSIC431x.commandsTransceive(xCommand.READ_REGISTER_PAGE, new byte[]{addr});
        if (recv == null || recv.length == 0) {
            return null;
        } else {
            mRegBuffer.set(addr, recv[1]);
            return recv[1];
        }
    }

    public byte[] read(byte... addr) {
        int length = addr.length;
        byte[] recv = new byte[length];
        byte[][] transeiver = new byte[length][];
        for (int i = 0; i < length; ++i) {
            transeiver[i] = new byte[]{addr[i]};
        }

        //Log.wtf(TAG, "mRegBuffer read");
        transeiver = mSIC431x.commandsTransceive(xCommand.READ_REGISTER_PAGE, transeiver);
        if (mSIC431x.isSendCompleted()) {
            for (int i = 0; i < length; ++i) {
                recv[i] = transeiver[i][1];
            }
            mRegBuffer.set(addr, recv);
            return recv;
        }
        return null;
    }

    /**
     * Configure register function of SIC431x.
     *
     * @param addr address of register.
     * @return data of register.
     */
    public Byte readBuffer(byte addr) {
        return mRegBuffer.get(addr);
    }

    public Byte[] readBuffer(byte... addr) {
        int length = addr.length;
        Byte[] recv = new Byte[length];
        for (int i = 0; i < length; ++i) {
            recv[i] = mRegBuffer.get(addr[i]);
        }
        return recv;
    }

    /**
     * Configure register function of SIC431x.
     *
     * @param addr address of register.
     * @param data data to be configured.
     */
    public void write(byte addr, byte data) {
        mSIC431x.commandsTransceive(xCommand.WRITE_REGISTER_PAGE, new byte[]{addr, data});
        if (mSIC431x.isSendCompleted()) {
            mRegBuffer.set(addr, data);
        }
    }

    public void write(byte[]... multi_data) {
        mSIC431x.commandsTransceive(xCommand.WRITE_REGISTER_PAGE, multi_data);
        if (mSIC431x.isSendCompleted()) {
            for (byte[] data : multi_data) {
                mRegBuffer.set(data[0], data[1]);
            }
        }
    }

    public void writeParams(byte addr, int params, int raw) {
        Byte buff = mRegBuffer.get(addr);
        if (buff == null) {
            Log.w(TAG, "It cannot write, register buffer is empty.");
            return;
        }
        byte data = (byte) ((raw & params) | (buff & ~params));
        write(addr, data);
    }
}
