/*
 * Copyright (c) 2015 Silicon Craft Technology Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.sic.module.nfc.tech.chips.s431x;

import com.sic.module.nfc.tech.chips.s4310.SIC4310;
import com.sic.module.nfc.tech.interfaces.ICommand;

/**
 * @author Tanawat Hongthai - http://www.sic.co.th/
 * @version 1.0.0
 * @since 3/27/2015
 */
public enum xCommand implements ICommand {

    // RF - UART Commands

    /**
     * The purpose of TxRU command is to convey data in the payload from
     * NFC/RFID reader device to the UART. The data in the payload must be less
     * than 64 bytes.
     */
    TXRU(8010) {
        @Override
        public byte[] getBuffer(byte[] data) {
            if (checkDataBuffer(data, 1, SIC4310.MAX_SIZE_PACKET)) {
                int datlen = data.length;
                byte[] buffer = new byte[datlen + 1];
                buffer[0] = (byte) 0xB1;
                System.arraycopy(data, 0, buffer, 1, datlen);
                return buffer;
            }
            return null;
        }
    },

    /**
     * The purpose of command RxUR is to retrieve remaining data in uplink FIFO.
     */
    RXUR(8020) {
        @Override
        public byte[] getBuffer(byte[] data) {
            if (checkDataBuffer(data, 1, data.length)) {
                byte[] buffer = new byte[2];
                buffer[0] = (byte) 0xB2;
                buffer[1] = data[0];
                return buffer;
            }
            return null;
        }
    },

    /**
     * The purpose of command TRxRU is to transmit data from NFC device/RFID
     * reader and retrieve data back from UART end device in a single command.
     * Depending on design system or application, wait time for response packet
     * from UART end device, can be called TRxRU_Resp_Time, must be
     * appropriately set via register TRxRU_Time to prevent system stuck.
     */
    TRXRU(8030) {
        @Override
        public byte[] getBuffer(byte[] data) {
            if (checkDataBuffer(data, 1, SIC4310.MAX_SIZE_PACKET)) {
                int datlen = data.length;
                byte[] buffer = new byte[datlen + 1];
                buffer[0] = (byte) 0xB3;
                System.arraycopy(data, 0, buffer, 1, datlen);
                return buffer;
            }
            return null;
        }
    },

    /**
     * NFC/RF device sends Clear_Flag command to clear the error flags
     * presenting in last B_NAK to make communication process going on.
     */
    CLEAR_FLAGS(8040) {
        @Override
        public byte[] getBuffer(byte[] data) {
            if (checkDataBuffer(data, 1, data.length)) {
                byte[] buffer = new byte[2];
                buffer[0] = (byte) 0xB4;
                buffer[1] = data[0];
                return buffer;
            }
            return null;

        }
    },

    // RF - RegPage Commands

    /**
     * The purpose of ReadReg command is to read register value. The response
     * frame consists of B_ACK or B_NAK with the current accessed register
     * value.
     */
    READ_REGISTER_PAGE(8050) {
        @Override
        public byte[] getBuffer(byte[] data) {
            if (checkDataBuffer(data, 1, data.length)) {
                byte[] buffer = new byte[2];
                buffer[0] = (byte) 0xB5;
                buffer[1] = data[0];
                return buffer;
            }
            return null;

        }
    },

    /**
     * The purpose of WriteReg command is to write register value. The response
     * frame can be B_ACK or B_NAK depending input power level.
     */
    WRITE_REGISTER_PAGE(8060) {
        @Override
        public byte[] getBuffer(byte[] data) {
            if (checkDataBuffer(data, 2, data.length)) {
                byte[] buffer = new byte[3];
                buffer[0] = (byte) 0xB6;
                System.arraycopy(data, 0, buffer, 1, 2);
                return buffer;
            }
            return null;

        }
    };

    private final int commandID;

    /**
     * This function is used to create a command.
     *
     * @param commandID ID of each command
     */
    xCommand(int commandID) {
        this.commandID = commandID;
    }

    /**
     * get ID of each command
     *
     * @return ID
     */
    public int getID() {
        return commandID;
    }

    public abstract byte[] getBuffer(byte[] data);

    public boolean checkDataBuffer(byte[] data, int min_size, int max_size) {
        return data != null && !(min_size < 0 || data.length < min_size || data.length > max_size);
    }
}
