/*
 * Copyright (c) 2015 Silicon Craft Technology Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.sic.module.nfc.tech.chips.s431x;

import android.content.Context;

import com.sic.module.nfc.Flag;
import com.sic.module.nfc.tech.chips.SIC43xx;
import com.sic.module.nfc.tech.chips.extension.ExtDevice;
import com.sic.module.nfc.tech.mandatories.NfcTypeA;
import com.sic.module.utils.AppContext;
import com.sic.module.utils.Utils;

/**
 * @author Tanawat Hongthai - http://www.sic.co.th/
 * @version 1.0.1
 * @since 11/16/2014
 */
public class SIC431x extends SIC43xx {

    public static final int MAX_SIZE_PACKET = 64;
    private static final String TAG = SIC431x.class.getName();
    private static SIC431x instance;
    protected ExtDevice exInstance;

    protected SIC431x(Context context) {
        super(context);
    }

    public static SIC431x getInstance() {
        return getInstance(AppContext.getInstance().getContext());
    }

    public static SIC431x getInstance(Context context) {
        if (instance == null) {
            instance = new SIC431x(context);
        }
        return instance;
    }

    public static String getFlagName(byte flag) {
        switch (flag) {
            case xRegister.FLAG_B_ACK:
                return "B_ACK";

            default:
                String rtFlag = "";
                byte cFlag = xRegister.FLAG_B_NAK;

                if ((flag & cFlag) == cFlag) {
                    cFlag = xRegister.FLAG_B_NAK_DL_FIFO_OVF;
                    if ((flag & cFlag) == cFlag) {
                        if (!rtFlag.isEmpty()) {
                            rtFlag += "/";
                        }
                        rtFlag += "DL_FIFO_OVF";
                    }

                    cFlag = xRegister.FLAG_B_NAK_UL_FIFO_EMP;
                    if ((flag & cFlag) == cFlag) {
                        if (!rtFlag.isEmpty()) {
                            rtFlag += "/";
                        }
                        rtFlag += "UL_FIFO_EMP";
                    }

                    cFlag = xRegister.FLAG_B_NAK_UL_FIFO_OVF;
                    if ((flag & cFlag) == cFlag) {
                        if (!rtFlag.isEmpty()) {
                            rtFlag += "/";
                        }
                        rtFlag += "UL_FIFO_OVF";
                    }

                    cFlag = xRegister.FLAG_B_NAK_RSPW_LOW;
                    if ((flag & cFlag) == cFlag) {
                        if (!rtFlag.isEmpty()) {
                            rtFlag += "/";
                        }
                        rtFlag += "PWR_LOW";
                    }

                    cFlag = xRegister.FLAG_B_NAK_XVDD_DROP;
                    if ((flag & cFlag) == cFlag) {
                        if (!rtFlag.isEmpty()) {
                            rtFlag += "/";
                        }
                        rtFlag += "XVDD_DROP";
                    }

                    cFlag = xRegister.FLAG_B_NAK_UART_FAIL;
                    if ((flag & cFlag) == cFlag) {
                        if (!rtFlag.isEmpty()) {
                            rtFlag += "/";
                        }
                        rtFlag += "UART_FAIL";
                    }

                    return rtFlag;
                } else {
                    return NfcTypeA.getFlagName(flag);
                }
        }
    }

    public ExtDevice getExtension() {
        return exInstance;
    }

    public void setExtension(ExtDevice instance) {
        exInstance = instance;
    }

    /**
     * Clear flag register. This function must be called at the first place.
     */
    public void clearFlagsRegister() {
        commandsTransceive(xCommand.CLEAR_FLAGS,
                Utils.parseIntegerToByteArray(0xff));
    }

    /**
     * Retrieve data from uplink FIFO and uplink to NFC device/RFID reader.
     * @return data receiver
     */
    public byte[] receiveDataFromUART() {
        return commandsTransceive(xCommand.RXUR,
                Utils.parseIntegerToByteArray(0x00));
    }

    public void clearFIFO() {
        autoTransceive(new byte[]{(byte) 0xB2, 0x00});
    }

    /**
     * Transmit data from NFC device/RFID reader to downlink FIFO.
     * @param data data transmission
     * @return data receiver
     */
    public byte[] sendDataViaUART(byte[] data) {
        return commandsTransceive(xCommand.TXRU, data);
    }

    public byte[][] sendDataViaUART(byte[]... data) {
        return commandsTransceive(xCommand.TXRU, data);
    }

    /**
     * Get last data receiver of all NFC communication commands without any
     * flags.
     *
     * @return data receiver from NFC without flags.
     */
    public byte[] getReceiveDataWithoutFlag() {
        byte[] data;
        byte[] receiveData = mNfcA.getReceiverData();
        if (receiveData == null || receiveData.length == 0) {
            return new byte[0];
        }
        if (receiveData.length == 2 && (isUARTFail() || isUplinkFIFOOverflow() || isDownlinkFIFOOverflow() || isUplinkFIFOEmpty())) {
            return new byte[0];
        }
        data = new byte[receiveData.length - 1];
        System.arraycopy(receiveData, 1, data, 0, data.length);
        return data;
    }


    public Byte getReceiveFlag() {
        byte[] receiveData = mNfcA.getReceiverData();
        if (receiveData == null || receiveData.length == 0) {
            return Flag.UNKNOWN;
        }
        return receiveData[0];
    }

    /**
     * get power measurement of chip
     *
     * @return power status.
     */
    public byte getPowerMeasurementStatus() {
        byte[] pwr_status = autoTransceive(new byte[]{(byte) 0xB5, 0x01});
        if (isSendCompleted()) {
            if (pwr_status != null) {
                if (pwr_status.length > 1) {
                    return pwr_status[1];
                }
            }

        }
        return 0x00;
    }

    public boolean isDownlinkFIFOOverflow() {
        byte flag = (xRegister.FLAG_B_NAK | xRegister.FLAG_B_NAK_DL_FIFO_OVF);
        return (mNfcA.getResponseFlag() & flag) == flag;
    }

    public boolean isUplinkFIFOEmpty() {
        byte flag = (xRegister.FLAG_B_NAK | xRegister.FLAG_B_NAK_UL_FIFO_EMP);
        return (mNfcA.getResponseFlag() & flag) == flag;
    }

    public boolean isUplinkFIFOOverflow() {
        byte flag = (xRegister.FLAG_B_NAK | xRegister.FLAG_B_NAK_UL_FIFO_OVF);
        return (mNfcA.getResponseFlag() & flag) == flag;
    }

    public boolean isInsufficientPowerToSourceLoad() {
        byte flag = (xRegister.FLAG_B_NAK | xRegister.FLAG_B_NAK_RSPW_LOW);
        return (mNfcA.getResponseFlag() & flag) == flag;
    }

    public boolean isVoltageOnXVddDrop() {
        byte flag = (xRegister.FLAG_B_NAK | xRegister.FLAG_B_NAK_XVDD_DROP);
        return (mNfcA.getResponseFlag() & flag) == flag;
    }

    public boolean isUARTFail() {
        byte flag = (xRegister.FLAG_B_NAK | xRegister.FLAG_B_NAK_UART_FAIL);
        return (mNfcA.getResponseFlag() & flag) == flag;
    }
}
