/*
 * Copyright (c) 2015 Silicon Craft Technology Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.sic.module.nfc.tech.chips.s4310;

import com.sic.module.nfc.tech.chips.s431x.SIC431x;
import com.sic.module.nfc.tech.chips.s431x.xCommand;
import com.sic.module.utils.Utils;

/**
 * @author Tanawat Hongthai - http://www.sic.co.th/
 * @version 1.0.1
 * @since 11/16/2014
 */
public class UART {

    public static final int SPEED_OF_UART = 1843200;

    public static final byte UART_CONFIG_STOP_BIT = (byte) 0x08;
    public static final byte UART_CONFIG_PARITY_BIT = (byte) 0x07;

    public static final byte UART_STOP_BIT_1 = (byte) 0x00;
    public static final byte UART_STOP_BIT_2 = (byte) 0x08;

    public static final byte UART_PARITY_BIT_NONE = (byte) 0x00;
    public static final byte UART_PARITY_BIT_SPACE = (byte) 0x04;
    public static final byte UART_PARITY_BIT_MARK = (byte) 0x05;
    public static final byte UART_PARITY_BIT_EVEN = (byte) 0x06;
    public static final byte UART_PARITY_BIT_ODD = (byte) 0x07;

    // Error Code UART Command
    public static final int FLAG_ERR_CODE_UCMD = 0x7E;
    public static final int ERR_IO_CONTROL_DONE = 0x15;
    public static final int ERR_IO_CONTROL_FAILED = 0x1A;
    public static final int ERR_EINK_SET_TEXT_DONE = 0x25;
    public static final int ERR_EINK_SET_CLOCK_DONE = 0x26;
    public static final int ERR_EINK_SET_DOWNCOUNT_TIMER_DONE = 0x27;
    public static final int ERR_EINK_SET_TEXT_FAILED = 0x2A;
    public static final int ERR_EINK_SET_CLOCK_FAILED = 0x2B;
    public static final int ERR_EINK_SET_DOWNCOUNT_TIMER_FAILED = 0x2C;
    public static final int ERR_WRITE_DONE = 0x35;
    public static final int ERR_WRITE_FAILED = 0x3A;
    public static final int ERR_TRANSFER_VIA_UHF_DONE = 0x45;
    public static final int ERR_TRANSFER_VIA_UHF_FAILED = 0x4A;
    public static final int ERR_READ_RETURNS_VALUES = 0x55;
    public static final int ERR_READ_FAILED = 0x5A;
    public static final int ERR_UNKNOWN_ERROR = 0xFA;

    private static UART instance;
    private GPIO mGPIO;
    private Register mRegister;
    private int bitRate;
    private byte stopBits;
    private byte parity;
    private byte mode;

    private UART() {
        mGPIO = GPIO.getInstance();
        mRegister = Register.getInstance();
    }

    public static UART getInstance() {
        if (instance == null)
            instance = new UART();
        return instance;
    }

    public void initUART() {
        initUART(115200, UART_STOP_BIT_1, UART_PARITY_BIT_NONE, (byte) (GPIO.PIN_UART_TX | GPIO.PIN_UART_RX));
    }

    public void initUART(int bitRate) {
        initUART(bitRate, UART_STOP_BIT_1, UART_PARITY_BIT_NONE, (byte) (GPIO.PIN_UART_TX | GPIO.PIN_UART_RX));
    }

    public void initUART(int bitRate, byte stopBits, byte parity, byte mode) {
        int UARTRateDiv, m, n;
        try {
            this.bitRate = bitRate;
            this.stopBits = stopBits;
            this.parity = parity;
            this.mode = mode;

            UARTRateDiv = SPEED_OF_UART / bitRate;
            m = 1;
            n = UARTRateDiv;

            while (n > 48) {
                m <<= 1;
                n >>= 1;
            }

            byte[][] buffer = {
                    {Register.UART_FRAME, (byte) (stopBits | parity)},
                    {Register.UART_DIVIDER_M, (byte) m},
                    {Register.UART_DIVIDER_N, (byte) n},
                    {Register.GPIO_MODE, (byte) (mode | mGPIO.getMode())},
                    {Register.GPIO_DIRECTION, (byte) (~(GPIO.PIN_UART_RX) & (GPIO.PIN_UART_TX | mGPIO.getDirection()))}};
            mRegister.write(buffer);

        } catch (Exception e) {
            e.getStackTrace();
        }
    }

    public int getBitRate() {
        return bitRate;
    }

    public void setBitRate(int baudRate) {
        this.bitRate = baudRate;
    }

    public byte getStopBits() {
        return stopBits;
    }

    public void setStopBits(byte stopBits) {
        this.stopBits = stopBits;
    }

    public byte getParity() {
        return parity;
    }

    public void setParity(byte parity) {
        this.parity = parity;
    }

    public byte getMode() {
        return mode;
    }

    public void setMode(byte mode) {
        this.mode = mode;
    }

    /**
     * Retrieve data from uplink FIFO and uplink to NFC device/RFID reader.
     * @return data receiver
     */
    public byte[] receive() {
        return SIC431x.getInstance().commandsTransceive(xCommand.RXUR,
                Utils.parseIntegerToByteArray(0x00));
    }

    /**
     * Transmit data from NFC device/RFID reader to downlink FIFO.
     * @param data data transmission
     * @return data receiver
     */
    public byte[] send(byte[] data) {
        return SIC431x.getInstance().commandsTransceive(xCommand.TXRU, data);
    }
}
