/*
 * Copyright (c) 2015 Silicon Craft Technology Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.sic.module.nfc.tech.chips.s4310;

import com.sic.module.nfc.tech.chips.s4310.registers.GPIODirection;
import com.sic.module.nfc.tech.chips.s4310.registers.GPIOIn;
import com.sic.module.nfc.tech.chips.s4310.registers.GPIOMode;
import com.sic.module.nfc.tech.chips.s4310.registers.GPIOOut;
import com.sic.module.nfc.tech.chips.s4310.registers.GPIOPullUp;
import com.sic.module.nfc.tech.chips.s4310.registers.OSCTuning;
import com.sic.module.nfc.tech.chips.s4310.registers.PeripheralAdjustment;
import com.sic.module.nfc.tech.chips.s4310.registers.PeripheralConfig;
import com.sic.module.nfc.tech.chips.s4310.registers.PowerStatus;
import com.sic.module.nfc.tech.chips.s4310.registers.TRxRUResponseTime;
import com.sic.module.nfc.tech.chips.s4310.registers.UARTDivisor_M;
import com.sic.module.nfc.tech.chips.s4310.registers.UARTDivisor_N;
import com.sic.module.nfc.tech.chips.s4310.registers.UARTFrame;
import com.sic.module.nfc.tech.chips.s4310.registers.UARTStatus;
import com.sic.module.nfc.tech.chips.s431x.xRegister;

/**
 * @author Tanawat Hongthai - http://www.sic.co.th/
 * @version 1.0.1
 * @since 10/16/2015
 */
public class Register extends xRegister {

    // Register Address
    public static final byte UART_STATUS = UARTStatus.ADDRESS;
    public static final byte POWER_STATUS = PowerStatus.ADDRESS;
    public static final byte TRXRU_RESPONSE_TIME = TRxRUResponseTime.ADDRESS;
    public static final byte UART_FRAME = UARTFrame.ADDRESS;
    public static final byte UART_DIVIDER_M = UARTDivisor_M.ADDRESS;
    public static final byte UART_DIVIDER_N = UARTDivisor_N.ADDRESS;
    public static final byte OSC_TUNING = OSCTuning.ADDRESS;
    public static final byte GPIO_DIRECTION = GPIODirection.ADDRESS;
    public static final byte GPIO_MODE = GPIOMode.ADDRESS;
    public static final byte GPIO_OUT = GPIOOut.ADDRESS;
    public static final byte GPIO_IN = GPIOIn.ADDRESS;
    public static final byte GPIO_PULLUP = GPIOPullUp.ADDRESS;
    public static final byte PERIPHERAL_CONFIG = PeripheralConfig.ADDRESS;
    public static final byte PERIPHERAL_ADJUSTMENT = PeripheralAdjustment.ADDRESS;
    private static final String TAG = Register.class.getSimpleName();
    private static final int REGISTER_PAGE = 0x10;
    private static Register instance;

    public static Register getInstance() {
        if (instance == null)
            instance = new Register();
        return instance;
    }

    /**
     * Get detail of each register page.
     *
     * @param addr index of requested register.
     * @return text
     */
    public static String getRegisterName(byte addr) {
        String header;
        switch (addr) {
            case UART_STATUS:
                header = "UART Status";
                break;
            case POWER_STATUS:
                header = "Power Status";
                break;
            case TRXRU_RESPONSE_TIME:
                header = "TRxRU Response time";
                break;
            case UART_FRAME:
                header = "Byte Configuration";
                break;
            case UART_DIVIDER_M:
                header = "UART Clock Divider 1";
                break;
            case UART_DIVIDER_N:
                header = "UART Clock Divider 2";
                break;
            case OSC_TUNING:
                header = "OSC Tuning";
                break;
            case GPIO_DIRECTION:
                header = "GPIO Direction";
                break;
            case GPIO_MODE:
                header = "GPIO Mode";
                break;
            case GPIO_OUT:
                header = "GPIO Out";
                break;
            case GPIO_IN:
                header = "GPIO In";
                break;
            case GPIO_PULLUP:
                header = "GPIO Pull-up";
                break;
            case PERIPHERAL_CONFIG:
                header = "Peripheral Config";
                break;
            case PERIPHERAL_ADJUSTMENT:
                header = "Peripheral Adjustment";
                break;
            default:
                header = "Unknown Register";
                break;
        }
        return header;
    }

    public static String getRegisterDescription(byte bCode) {
        switch (bCode) {
            case 0x00:
                return "Bit 7 - 6 : RFU \n"
                        + "\n"
                        + "Bit 5 : CTS - R/O - Status.\n"
                        + "		0 : CTS function is disable or not ready.\n"
                        + "		1 : External UART device is ready to receive data.\n"
                        + "\n"
                        + "Bit 4 : RTS - R/O - Status.\n"
                        + "		0 : RTS function is disable or not ready.\n"
                        + "		1 : UL FIFO is ready to receive data.\n"
                        + "\n"
                        + "Bit 3 : DL_FF_EMT - R/O - Status.\n"
                        + "		0 : - \n"
                        + "		1 : Downlink FIFO Empty Indicator flag. \n"
                        + "\n"
                        + "Bit 2 : DL_FF_OVF - R/O - Status.\n"
                        + "		0 : - \n"
                        + "		1 : Downlink FIFO Overflow Indicator flag. (Can be clear by command Clear_Flag)\n"
                        + "\n"
                        + "Bit 1 : UL_FF_EMT - R/O - Status.\n"
                        + "		0 : - \n"
                        + "		1 : Uplink FIFO Empty Indicator flag. \n"
                        + "\n"
                        + "Bit 0 : UL_FF_OVF - R/O - Status.\n"
                        + "		0 : - \n"
                        + "		1 : Uplink FIFO Overflow Indicator flag. (Can be clear by command Clear_Flag)\n"
                        + "\n";

            case 0x01:
                return "Bit 7 - 5 : RFU \n"
                        + "\n"
                        + "Bit 4 : SCAP_RDY - R/O - Status.\n"
                        + "		0 : - \n"
                        + "		1 : Voltage on pin SCAP is higher than 4.5V.\n"
                        + "\n"
                        + "Bit 3 : UART_RDY - R/O - Status.\n"
                        + "		0 : - \n"
                        + "		1 : On-chip oscillator is stable and ready to operate UART communication.\n"
                        + "\n"
                        + "Bit 2 : XVDD_RDY - R/O - Status.\n"
                        + "		0 : - \n"
                        + "		1 : Voltage on pin XVDD is higher than XVDD drop level (set from LDO_D_LV).\n"
                        + "\n"
                        + "Bit 1 : RSPW_RDY - R/O - Status.\n"
                        + "		0 : - \n"
                        + "		1 :	Reserve power from RF is higher than defined supplying level. (set from RFLM_LV).\n"
                        + "\n"
                        + "Bit 0 : LDO_ON - R/O - Status.\n"
                        + "		0 : - \n"
                        + "		1 :	On-chip LDO regulator is successfully turned on.\n"
                        + "\n";
            case 0x02:
                return "Bit 7 - 0 : RFU \n" + "\n";
            case 0x03:
                return "Bit 7 - 4 : RFU \n"
                        + "\n"
                        + "Bit 3 - 0 : TRxRU_Time - R/W - Config\n"
                        + "		Transceive Response Time = ( 2^TRxRU_Time ) x ( 1 ms ). \n"
                        + "		Maximum value of TRxRU_Time is 12. Hence, Maximum response time is 4 s.\n"
                        + "\n" + "Factory Preprogram Register Value\n"
                        + "		0x04 \n" + "\n";

            case 0x04:
                return "Bit 7 - 5 : RFU \n" + "\n"
                        + "Bit 4 : Stop_Len - R/W - Config\n"
                        + "		0 : 1 stop bit of UART.\n"
                        + "		1 : 2 stop bits of UART.\n" + "\n"
                        + "Bit 3 - 0 : Parity - R/W - Config\n"
                        + "		0XX : None parity of UART.\n"
                        + "		100 : Space (0) parity of UART.\n"
                        + "		101 : Mark (1) parity of UART.\n"
                        + "		110 : Even parity of UART.\n"
                        + "		111 : Odd parity of UART.\n" + "\n"
                        + "Factory Preprogram Register Value\n" + "		0x08 \n"
                        + "\n";

            case 0x05:
                return "Bit 7 - 6 : RFU \n"
                        + "\n"
                        + "Bit 5 - 0 : UART_DIV_m - R/W - Config\n"
                        + "		UART clock to set speed of UART communication. It must be used together with UART_Divisor_n (Reg 0x06).\n"
                        + "		0 : the divisor is 64.\n" + "\n"
                        + "Factory Preprogram Register Value\n" + "		0x01 \n"
                        + "\n";
            case 0x06:
                return "Bit 7 - 6 : RFU \n"
                        + "\n"
                        + "Bit 5 - 0 : UART_DIV_n - R/W - Config\n"
                        + "		UART clock to set speed of UART communication. It must be used together with UART_Divisor_m (Reg 0x05).\n"
                        + "		If value is less than 2 : SIC4310 can not perform communication.\n"
                        + "\n" + "Factory Preprogram Register Value\n"
                        + "		0x10 \n" + "\n";

            case 0x07:
                return "Bit 7 - 4 : RFU \n"
                        + "\n"
                        + "Bit 5 - 0 : OSC_Tuning - R/W\n"
                        + "		0000 : frequency is set to the highest adjustable value.\n"
                        + "		1111 : frequency is set to the lowest adjustable value.\n"
                        + "\n" + "Factory Preprogram Register Value\n"
                        + "		0x08 \n" + "\n";

            case 0x08:
                return "Bit 7 - 0 : DIR [7:0] - R/W - Config\n"
                        + "		0 : GPIO input direction.\n"
                        + "		1 : GPIO output direction.\n" + "\n"
                        + "Factory Preprogram Register Value\n" + "		0x03 \n"
                        + "\n";

            case 0x09:
                return "Bit 7 - 0 : Mode [7:0] - R/W - Config\n"
                        + "		0 : General Purpose I/O.\n"
                        + "		1 : Special Function.\n" + "\n"
                        + "Factory Preprogram Register Value\n" + "		0x07 \n"
                        + "\n";

            case 0x0A:
                return "Bit 7 - 0 : Out [7:0] - R/W - Config\n"
                        + "		0 : GPIO output is logic low.\n"
                        + "		1 : GPIO output is logic high.\n" + "\n" + "\n"
                        + "Factory Preprogram Register Value\n" + "		0x00 \n"
                        + "\n";

            case 0x0B:
                return "Bit 7 - 0 : IN [7:0] - R/O - Status\n"
                        + "		0 : GPIO input is logic low.\n"
                        + "		1 : GPIO input is logic high.\n" + "\n";

            case 0x0C:
                return "Bit 7 - 0 : PU [7:0] - R/W - Config\n"
                        + "		0 : Disable Pull Up resistor.\n"
                        + "		1 : Enable Pull Up resistor when set to Input.\n"
                        + "\n" + "Factory Preprogram Register Value\n"
                        + "		0x00 \n" + "\n";

            case 0x0D:
                return "Bit 7 - 4 : RFU \n"
                        + "\n"
                        + "Bit 5 - 4 : PW_LV[1:0] - R/W – Config\n"
                        + "		00 : Threshold of power: 500 µA\n"
                        + "		01 : Threshold of power: 1.25 mA\n"
                        + "		10 : Threshold of power: 2.50 mA\n"
                        + "		11 : Threshold of power: 5.00 mA\n"
                        + "Bit 3 : RFU \n"
                        + "\n"
                        + "Bit 2 : PWCHK_EN - R/W – Config\n"
                        + "		0 : Disable (on-chip LDO is on immediately without qualifying process).\n"
                        + "		1 : Enable qualifying process.\n" + "\n"
                        + "Bit 1 : LDO_EN - R/W – Config\n"
                        + "		0 : Disable LDO Regulator.\n"
                        + "		1 : Enable LDO Regulator.\n" + "\n"
                        + "Bit 0 : OSC_EN - R/W – Config\n"
                        + "		0 : Disable on-chip oscillator.\n"
                        + "		1 : Enable on-chip oscillator.\n" + "\n"
                        + "Factory Preprogram Register Value\n" + "		0x11 \n"
                        + "\n";

            case 0x0E:
                return "Bit 7 - 3 : RFU \n"
                        + "\n"
                        + "Bit 2 : LDO_D_LV - R/W - Config\n"
                        + "		0 : Threshold voltage drop level 2.4 volt.\n"
                        + "		1 : Threshold voltage drop level 2.7 volt.\n"
                        + "Bit 1 : RFLM_LV - R/W - Config\n"
                        + "		0 : RF limiter level 5.2 volt (maximum of SCAP voltage = 4.8 volt).\n"
                        + "		1 : RF limiter level 6.5 volt (maximum of SCAP voltage = 6.0 volt).\n"
                        + "Bit 0 : 2B_FLAG - R/W - Config\n"
                        + "		0 : All response package is set to 1 byte.\n"
                        + "		1 : Some response package is set to 2 bytes.\n" + "\n"
                        + "Factory Preprogram Register Value\n" + "		0x00 \n"
                        + "\n";

            default:
                return "Address Incorrect.\n";
        }
    }

    @Override
    protected int getRegisterPage() {
        return REGISTER_PAGE;
    }

    public Byte getResponseTime() {
        return read(TRXRU_RESPONSE_TIME);
    }

    /**
     * Set response time of TRxRU command.
     *
     * @param time a desired response time.
     */
    public void setResponseTime(int time) {
        write(TRXRU_RESPONSE_TIME, (byte) time);
    }
}
