/*
 * Copyright (c) 2015 Silicon Craft Technology Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.sic.module.nfc.tech.chips.s4310;

import com.sic.module.nfc.tech.chips.s431x.xGPIO;
import com.sic.module.utils.SICLog;

/**
 * @author Tanawat Hongthai - http://www.sic.co.th/
 * @version 1.0.1
 * @since 3/27/2015
 */
public class GPIO extends xGPIO {

    // Mode 1, DIR 0
    public static final byte PIN_UART_RX = Pin.parsePin(2);
    public static final byte PIN_UART_CTS = Pin.parsePin(0);
    // Mode 1, DIR 1
    public static final byte PIN_PWR_RDY = Pin.parsePin(5);
    public static final byte PIN_RF_BUSY = Pin.parsePin(4);
    public static final byte PIN_UART_RTS = Pin.parsePin(3);
    public static final byte PIN_UART_TX = Pin.parsePin(1);
    public static final byte PIN_RF_DETECT = Pin.parsePin(0);
    public static final byte DIRECTION_INPUT = 0;
    public static final byte DIRECTION_OUTPUT = 1;
    public static final byte GPIO_MODE = 0;
    public static final byte FUNCTION_MODE = 1;
    private static final String TAG = GPIO.class.getName();
    private static GPIO instance;
    private final Register mRegister;
    private byte direction;
    private byte mode;
    private byte output;
    private byte pullup;

    private GPIO() {
        super();
        mRegister = Register.getInstance();
    }

    public static GPIO getInstance() {
        if (instance == null)
            instance = new GPIO();
        return instance;
    }

    /**
     * Initialize GPIO pin
     */
    @Override
    public void sync() {
        if (!mRegister.isSync()) {
            mRegister.sync();
        }
        Byte direction = mRegister.readBuffer(Register.GPIO_DIRECTION);
        Byte mode = mRegister.readBuffer(Register.GPIO_MODE);
        Byte output = mRegister.readBuffer(Register.GPIO_OUT);
        Byte pullup = mRegister.readBuffer(Register.GPIO_PULLUP);
        if (!(direction == null || mode == null || output == null || pullup == null)) {
            this.direction = direction;
            this.mode = mode;
            this.output = output;
            this.pullup = pullup;
        }
    }

    /**
     * Configured GPIO of SIC431x.
     *
     * @param addr register address (direction, mode, out)
     *             GPIO_DIRECTION (0x08) GPIO_MODE (0x09)
     *             GPIO_OUT (0x0A)
     * @param bit  some bit of register address. 0 is logic low, another is logic
     *             high.
     * @param val  the value of this register address.
     */
    private void config(int addr, byte bit, int val) {
        switch (addr) {
            case Register.GPIO_DIRECTION:
                setBitDirection(bit, val);
                break;

            case Register.GPIO_MODE:
                setBitMode(bit, val);
                break;

            case Register.GPIO_OUT:
                setBitOutput(bit, val);
                break;
            default:
                SICLog.i(TAG, "the address is incorrectly.");
                break;
        }
    }

    /**
     * Get GPIO Direction register
     *
     * @return GPIO Direction register
     */
    public byte getDirection() {
        return direction;
    }

    /**
     * Set GPIO direction register
     *
     * @param GPIO_Dir direction
     */
    public void setDirection(byte GPIO_Dir) {
        this.direction = GPIO_Dir;
        mRegister.write(Register.GPIO_DIRECTION, direction);
    }

    /**
     * Set bit of GPIO direction register
     *
     * @param pin   GPIO
     * @param value direction
     */
    public void setBitDirection(byte pin, int value) {
        if (value == GPIO.LOGIC_LOW) {
            this.direction &= ~pin;
        } else {
            this.direction |= pin;
        }
        mRegister.write(Register.GPIO_DIRECTION, direction);
    }

    /**
     * Get GPIO Mode register
     *
     * @return GPIO mode register
     */
    public byte getMode() {
        return this.mode;
    }

    /**
     * Set GPIO mode register
     *
     * @param GPIO_Mode GPIO mode
     */
    public void setMode(byte GPIO_Mode) {
        this.mode = GPIO_Mode;
        mRegister.write(Register.GPIO_MODE, mode);
    }

    /**
     * Set bit of GPIO mode register
     *
     * @param pin   GPIO
     * @param value mode
     */
    public void setBitMode(byte pin, int value) {
        if (value == GPIO.LOGIC_LOW) {
            this.mode &= ~pin;
        } else {
            this.mode |= pin;
        }
        mRegister.write(Register.GPIO_MODE, mode);
    }

    /**
     * Get GPIO Output register
     *
     * @return GPIO Output register
     */
    @Override
    public byte getOutput() {
        return output;
    }

    /**
     * Set GPIO output register
     *
     * @param GPIO_Out GPIO output
     */
    @Override
    public void setOutput(byte GPIO_Out) {
        this.output = GPIO_Out;
        mRegister.write(Register.GPIO_OUT, this.output);
    }

    /**
     * Set bit of GPIO output register
     *
     * @param pin   GPIO
     * @param value output
     */
    @Override
    public void setBitOutput(byte pin, int value) {
        if (value == GPIO.LOGIC_LOW) {
            this.output &= ~pin;
        } else {
            this.output |= pin;
        }
        mRegister.write(Register.GPIO_OUT, output);
    }

    /**
     * Get current GPIO input port of GPIO variable. 0 is input and 1 is output.
     * if transceiver failed then return null
     *
     * @return GPIO_Input
     */
    @Override
    public Byte getInput() {
        Byte recv;
        recv = mRegister.read(Register.GPIO_IN);

        if (recv == null) {
            SICLog.e(TAG, "Can not get GPIO input.");
            return null;
        }
        return recv;
    }

    /**
     * Get a selected GPIO input port of GPIO variable. 0 is input and 1 is
     * output. if transceiver failed then return null
     *
     * @return pin selected GPIO pin
     */
    @Override
    public Byte getBitInput(byte pin) {

        Byte recv = getInput();
        if (recv == null) {
            return null;
        }

        if ((recv & pin) == pin) {
            return LOGIC_HIGH;
        } else {
            return LOGIC_LOW;
        }
    }

    /**
     * Set a selected GPIO to input mode.
     *
     * @param pin selected GPIO pin. (bit 1 for a desired port)
     */
    public void setGPIOModeInputDirection(byte pin) {
        config(Register.GPIO_DIRECTION, pin, DIRECTION_INPUT);
        config(Register.GPIO_MODE, pin, GPIO_MODE);
    }

    /**
     * Set a selected GPIO pin to output mode.
     *
     * @param pin selected GPIO pin. (bit 1 for a desired port)
     */
    public void setGPIOModeOutputDirection(byte pin) {
        config(Register.GPIO_DIRECTION, pin, DIRECTION_OUTPUT);
        config(Register.GPIO_MODE, pin, GPIO_MODE);
    }

    /**
     * Set a selected Function pin in input mode.
     *
     * @param pin selected Function pin. (bit 1 for a desired port)
     */
    public void setFunctionModeInputDirection(byte pin) {
        config(Register.GPIO_DIRECTION, pin, DIRECTION_INPUT);
        config(Register.GPIO_MODE, pin, FUNCTION_MODE);
    }

    /**
     * Set a selected Function pin in output mode.
     *
     * @param pin selected Function pin. (bit 1 for a desired port)
     */
    public void setFunctionModeOutputDirection(byte pin) {
        config(Register.GPIO_DIRECTION, pin, DIRECTION_OUTPUT);
        config(Register.GPIO_MODE, pin, FUNCTION_MODE);
    }

    /**
     * Set current GPIO pullup resistor when IO is set to GPIO and input.
     *
     * @param GPIO_Pullup the GPIO pullup bit 0 is disabled and bit 1 is enabled.
     */
    public void setPullup(byte GPIO_Pullup) {
        this.pullup = GPIO_Pullup;
        mRegister.write(Register.GPIO_PULLUP, GPIO_Pullup);
    }

    /**
     * Set current GPIO pullup resistor of particular pin when IO is set to GPIO
     * and input.
     *
     * @param pos the position bit of GPIO pin
     * @param val the value of this register address.
     */
    public void setBitPullup(int pos, int val) {
        byte bit = (byte) pos;
        if (val == 0) {
            this.pullup &= ~bit;
        } else {
            this.pullup |= bit;
        }
        mRegister.write(Register.GPIO_PULLUP, pullup);
    }
}
