/*
 * Copyright (c) 2015 Silicon Craft Technology Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.sic.module.nfc.tech.chips;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.nfc.tech.Ndef;
import android.nfc.tech.NfcA;

import com.sic.module.nfc.Flag;
import com.sic.module.nfc.tech.interfaces.ICommand;
import com.sic.module.nfc.tech.mandatories.NfcTypeA;
import com.sic.module.nfc.tech.optionals.NfcTypeNdef;
import com.sic.module.utils.SICLog;

/**
 * @author Tanawat Hongthai - http://www.sic.co.th/
 * @version 1.0.1
 * @since 3/27/2015
 */
public class SIC43xx {

    private static final String TAG = SIC43xx.class.getName();

    private static SIC43xx instance;
    protected NfcTypeA mNfcA;
    protected NfcTypeNdef mNdef;

    protected SIC43xx(Context context) {
        SICLog.i(TAG, "SIC43xx is supported NFC Type A and NDEF format");
        mNfcA = NfcTypeA.getInstance(context);
        mNdef = NfcTypeNdef.getInstance(context);
    }

    public static SIC43xx getInstance(Context context) {
        if (instance == null) {
            instance = new SIC43xx(context);
        }
        return instance;
    }

    /**
     * Call this method in your Activity's onResume() method body.
     *
     * @param activity locale activity
     */
    public void onResume(Activity activity) {
        mNfcA.onResume(activity);
        //mNdef.onResume(activity);
    }

    /**
     * Call this method in your Activity's onPause() method body.
     *
     * @param activity locale activity
     */
    public void onPause(Activity activity) {
        mNfcA.onPause(activity);
        //mNdef.onPause(activity);
    }

    public boolean onTagFinding(Intent intent) {
        boolean found = false;
        if (mNfcA.onTagFinding(intent)) {
            mNdef.onTagFinding(mNfcA.getTag());
            found = true;
        }
        return found;
    }

    public void requestTag() {
        new Thread(new Runnable() {

            @Override
            public void run() {
                try {
                    mNfcA.setTimeout(100);
                    mNfcA.connect();

                    mNfcA.transceive(new byte[]{(byte) 0xB6, 0x03, (byte) 0xFF});
                    mNfcA.transceive(new byte[]{(byte) 0xB3, 0x00});
                    mNfcA.transceive(new byte[]{(byte) 0xB0});

                    mNfcA.close();
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }

        }).start();
    }

    /**
     * NFC transceive function connects and closes NFC software service automatically.
     *
     * @param send_data raw data sent from device.
     * @return data received from NFC.
     */
    public byte[] autoTransceive(byte[] send_data) {
        return mNfcA.autoTransceive(send_data);
    }

    /**
     * NFC transceive function connects and closes NFC software service automatically.
     *
     * @param send_data raw data sent from device.
     * @return data received from NFC.
     */
    public byte[][] autoTransceive(byte[]... send_data) {
        return mNfcA.autoTransceive(send_data);
    }

    /**
     * Check communication commands is successful.
     *
     * @return if communication commands is successful, then return true
     */
    public boolean isSendCompleted() {
        return mNfcA.getResponseFlag() != Flag.DISAPPEAR && mNfcA.isSendCompleted();
    }

    public byte[] commandsTransceive(ICommand command, byte[] data) {
        return mNfcA.commandTransceive(command, data);
    }

    public byte[][] commandsTransceive(ICommand command, byte[]... data) {
        return mNfcA.commandTransceive(command, data);
    }

    public boolean isTagAlive() {
        mNfcA.isNfcTag();
        return isSendCompleted();
    }

    public int getTimeout() {
        return mNfcA.getTimeout();
    }

    public void setTimeout(int timeout) {
        mNfcA.setTimeout(timeout);
    }

    public NfcTypeNdef getNdef() {
        return mNdef;
    }

    public NfcA getRawNfcA() {
        return mNfcA.getRawNfcA();
    }

    public Ndef getRawNdef() {
        return mNdef.getRawNdef();
    }

    public byte[] getUID() {
        return mNfcA.getUID();
    }
}
