/*
 * Copyright 2018 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 21.03.2018 - [JR] - creation
 */
package com.sibvisions.rad.ui.swing.impl.component;

import javax.rad.ui.IColor;
import javax.rad.ui.IImage;
import javax.rad.ui.component.ILabeledIcon;
import javax.swing.AbstractButton;
import javax.swing.ImageIcon;

import com.sibvisions.rad.ui.swing.impl.SwingComponent;
import com.sibvisions.rad.ui.swing.impl.SwingFactory;

/**
 * The <code>SwingAbstractLabeledIconButton</code> is a labeled icon button
 * implementation for swing.
 * 
 * @param <C> instance of AbstractButton.
 * 
 * @author Ren Jahn
 * @see	javax.swing.AbstractButton
 * @see javax.rad.ui.component.ILabeledIcon
 */
public abstract class SwingAbstractLabeledIconButton<C extends AbstractButton> extends SwingComponent<C> 
                                                                               implements ILabeledIcon
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Class members
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the icon image. */ 
	protected IImage image = null;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Initialization
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creates a new instance of <code>SwingAbstractLabeledIconButton</code>.
	 * 
	 * @param pAbstractButton the instance of AbstractButton.
	 */
	public SwingAbstractLabeledIconButton(C pAbstractButton)
	{
		super(pAbstractButton);

		// set correct default values.
		super.setHorizontalAlignment(SwingFactory.getHorizontalAlignment(resource.getHorizontalAlignment()));
		super.setVerticalAlignment(SwingFactory.getVerticalAlignment(resource.getVerticalAlignment()));
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Interface implementation
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	// IButton
	
	/**
	 * {@inheritDoc}
	 */
	public String getText()
	{
		return resource.getText();
	}

	/**
	 * {@inheritDoc}
	 */
	public void setText(String pText)
	{
		resource.setText(pText);
	}

	/**
	 * {@inheritDoc}
	 */
    public IImage getImage()
    {
    	return image;
    }
    
	/**
	 * {@inheritDoc}
	 */
    public void setImage(IImage pImage)
    {
    	if (pImage == null)
    	{
    		resource.setIcon(null);
    	}
    	else
    	{
    		resource.setIcon((ImageIcon)pImage.getResource());
    	}

    	image = pImage;
    }
	
	/**
	 * {@inheritDoc}
	 */
	public boolean isPreserveAspectRatio()
	{
		return false;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setPreserveAspectRatio(boolean pPreserveAspectRatio)
	{
		// Does nothing.
	}
    
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setHorizontalAlignment(int pHorizontalAlignment)
	{
		super.setHorizontalAlignment(pHorizontalAlignment);
		resource.setHorizontalAlignment(SwingFactory.getHorizontalSwingAlignment(pHorizontalAlignment));
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setVerticalAlignment(int pVerticalAlignment)
	{
		super.setVerticalAlignment(pVerticalAlignment);
		resource.setVerticalAlignment(SwingFactory.getVerticalSwingAlignment(pVerticalAlignment));
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setBackground(IColor pBackground)
	{
		if (pBackground != null || SwingFactory.isLaFOpaque())
		{
			super.setBackground(pBackground);
		}
	}

}	// SwingAbstractLabeledIconButton
