/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 10.01.2011 - [HM] - creation
 */
package com.sibvisions.rad.ui.swing.ext;

import java.awt.BorderLayout;
import java.io.InvalidObjectException;
import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.util.Date;

import javax.rad.model.ColumnDefinition;
import javax.rad.model.IDataBook;
import javax.rad.model.IDataRow;
import javax.rad.model.ModelException;
import javax.rad.model.ui.ITableControl;
import javax.rad.util.TranslationMap;
import javax.swing.JPanel;

import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.Axis;
import org.jfree.chart.axis.CategoryAxis;
import org.jfree.chart.axis.DateAxis;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.axis.ValueAxis;
import org.jfree.chart.labels.StandardCategoryToolTipGenerator;
import org.jfree.chart.labels.StandardPieSectionLabelGenerator;
import org.jfree.chart.labels.StandardXYToolTipGenerator;
import org.jfree.chart.plot.CategoryPlot;
import org.jfree.chart.plot.PiePlot;
import org.jfree.chart.plot.Plot;
import org.jfree.chart.plot.XYPlot;
import org.jfree.chart.renderer.AbstractRenderer;
import org.jfree.chart.renderer.category.AbstractCategoryItemRenderer;
import org.jfree.chart.renderer.category.AreaRenderer;
import org.jfree.chart.renderer.category.BarRenderer;
import org.jfree.chart.renderer.category.CategoryItemRenderer;
import org.jfree.chart.renderer.category.LineAndShapeRenderer;
import org.jfree.chart.renderer.xy.ClusteredXYBarRenderer;
import org.jfree.chart.renderer.xy.XYAreaRenderer;
import org.jfree.chart.renderer.xy.XYItemRenderer;
import org.jfree.chart.renderer.xy.XYLineAndShapeRenderer;
import org.jfree.chart.urls.StandardXYURLGenerator;
import org.jfree.data.category.CategoryDataset;
import org.jfree.data.category.CategoryToPieDataset;
import org.jfree.data.category.DefaultCategoryDataset;
import org.jfree.data.general.AbstractDataset;
import org.jfree.data.xy.AbstractIntervalXYDataset;
import org.jfree.data.xy.TableXYDataset;
import org.jfree.data.xy.XYDataset;
import org.jfree.util.TableOrder;

/**
 * The <code>JVxChart</code> is a simple chart panel which also implements the {@link ITableControl}
 * interface.
 *  
 * @author Martin Handsteiner
 */
public class JVxChart extends JPanel
                      implements ITableControl,
                                 Runnable
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Constants
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** Style constant for showing a line chart. */
	public static final int STYLE_LINES = 0;
	
	/** Style constant for showing an area chart. */
	public static final int STYLE_AREA = 1;
	
	/** Style constant for showing a bar chart. */
	public static final int STYLE_BARS = 2;
	
	/** Style constant for showing a pie chart. */
	public static final int STYLE_PIE = 3;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Class members
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** The chart panel. */
	private ChartPanel chartPanel = null;

	/** The dataset. */
	private AbstractDataset dataset = null;

	/** The x axis. */
	private Axis vaX = null;
	
	/** The y axis. */
	private Axis vaY = null;

	/** The DataBook to be shown. */
	private IDataBook dataBook = null;
	
	/** The chart style. */
	private int chartStyle = STYLE_LINES;
	
	/** The title. */
	private String sTitle = null;
	
	/** The x axis title. */
	private String sXTitle = null;
	
	/** The y axis title. */
	private String sYTitle = null;
	
	/** The x column name. */
	private String sXColumnName = null;
	
	/** The y column names. */
	private String[] saYColumnNames = null;
	
	/** The translation mapping. */
	private TranslationMap tmpTranslation = null;
	
	/** Tells, if notifyRepaint is called the first time. */
	private boolean bFirstNotifyRepaintCall = true;
	
    /** Whether the translation is enabled. */
    private boolean bTranslationEnabled = true;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Initialization
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** 
	 * Constructs a <code>JVxChart</code>.
	 */
	public JVxChart()
	{
		setLayout(new BorderLayout());
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Interface implementation
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	//ITABLECONTROL
	
	/**
	 * {@inheritDoc}
	 */
	public void notifyRepaint()
	{
		if (bFirstNotifyRepaintCall)
		{
			bFirstNotifyRepaintCall = false;

			JVxUtil.invokeLater(this);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void startEditing()
	{
		// Do nothing
	}

	/**
	 * {@inheritDoc}
	 */
	public void cancelEditing()
	{
		if (dataset != null)
		{
			try
			{
				dataset.validateObject();
			}
			catch (Exception e)
			{
				// Ignore Exception
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void saveEditing() throws ModelException
	{
		// Do nothing
	}

	//ITranslatable
	
    /**
     * {@inheritDoc}
     */
    public void setTranslation(TranslationMap pTranslation)
    {
    	if (tmpTranslation != pTranslation)
    	{
        	tmpTranslation = pTranslation;

        	if (vaX != null)
        	{
        		vaX.setLabel(translate(sXTitle));
        	}
        	
        	if (vaY != null)
        	{
        		vaY.setLabel(translate(sYTitle));
        	}

        	if (chartPanel != null)
        	{
        		chartPanel.getChart().setTitle(translate(sTitle));
        	}
        	
        	notifyRepaint();
    	}
    }
    
    /**
     * {@inheritDoc}
     */
    public TranslationMap getTranslation()
    {
    	return tmpTranslation;
    }
    
    /**
     * {@inheritDoc}
     */
    public void setTranslationEnabled(boolean pEnabled)
    {
        bTranslationEnabled = pEnabled;
    }
    
    /**
     * {@inheritDoc}
     */
    public boolean isTranslationEnabled()
    {
        return bTranslationEnabled;
    }
    
    //RUNNABLE
    
    /**
     * {@inheritDoc}
     */
	public void run()
	{
		cancelEditing();
		
		bFirstNotifyRepaintCall = true;
	}
    
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // User-defined methods
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
     * Gets the DataBook shown by this control.
     *
     * @return the DataBook.
     * @see #setDataBook
     */
	public IDataBook getDataBook()
	{
		return dataBook;
	}
	
	/**
     * Sets the DataBook shown by this control.
     *
     * @param pDataBook the DataBook.
     * @see #getDataBook
     */
	public void setDataBook(IDataBook pDataBook)
	{
		uninstallChart();
		
		dataBook = pDataBook;
		
		installChart();
	}
	
	/**
	 * Gets the style of the chart.
	 * 
     * @return the chart style ({@link #STYLE_LINES}, {@link #STYLE_AREA}, {@link #STYLE_BARS}).
     */
	public int getChartStyle()
	{
		return chartStyle;
	}

	/**
	 * Sets the style of the chart.
	 * 
	 * @param pChartStyle the chart style ({@link #STYLE_LINES}, {@link #STYLE_AREA}, {@link #STYLE_BARS}).
	 */
	public void setChartStyle(int pChartStyle)
	{
		if (pChartStyle != chartStyle)
		{
			uninstallChart();
			
			chartStyle = pChartStyle;
	
			installChart();
		}
	}

	/**
	 * Gets the title.
	 * 
	 * @return the title.
	 */
	public String getTitle()
	{
		return sTitle;
	}

	/**
	 * Sets the title.
	 * 
	 * @param pTitle the title.
	 */
	public void setTitle(String pTitle)
	{
		sTitle = pTitle;
	}

	/**
	 * Gets the x axis title.
	 * 
	 * @return the x axis title.
	 */
	public String getXAxisTitle()
	{
		return sXTitle;
	}

	/**
	 * Sets the x axis title.
	 * 
	 * @param pXAxisTitle the x axis title.
	 */
	public void setXAxisTitle(String pXAxisTitle)
	{
		sXTitle = pXAxisTitle;
	}

	/**
	 * Gets the y axis title.
	 * 
	 * @return the y axis title.
	 */
	public String getYAxisTitle()
	{
		return sYTitle;
	}

	/**
	 * Sets the y axis title.
	 * 
	 * @param pYAxisTitle the y axis title.
	 */
	public void setYAxisTitle(String pYAxisTitle)
	{
		sYTitle = pYAxisTitle;
	}

	/**
	 * Gets the x column name.
	 * 
	 * @return the x column name.
	 */
	public String getXColumnName()
	{
		return sXColumnName;
	}

	/**
	 * Sets the x column name.
	 * 
	 * @param pXColumnName the x column name.
	 */
	public void setXColumnName(String pXColumnName)
	{
		uninstallChart();
		
		sXColumnName = pXColumnName;
		
		installChart();
	}

	/**
	 * Gets the y column names.
	 * 
	 * @return the y column names.
	 */
	public String[] getYColumnNames()
	{
		return saYColumnNames;
	}

	/**
	 * Sets the y column names.
	 * 
	 * @param pYColumnNames y column names.
	 */
	public void setYColumnNames(String[] pYColumnNames)
	{
		uninstallChart();
		
		saYColumnNames = pYColumnNames;
		
		installChart();
	}	
	
	/**
	 * Uninstalls the CellEditor and its CellEditorComponent.
	 */
	private void uninstallChart()
	{
		if (chartPanel != null)
		{
			dataBook.removeControl(this);
			
			remove(chartPanel);
			
			chartPanel = null;
			dataset = null;
		}
	}

	/**
     * Gets the translation.
     *
     * @param pText the text.
     * @return the translation.
     */
	public String translate(String pText)
	{
		if (!bTranslationEnabled || tmpTranslation == null || pText == null)
		{
			return pText;
		}
		else
		{
			return tmpTranslation.translate(pText);
		}
	}

	/**
	 * Installs the CellEditor and its CellEditorComponent.
	 */
	private void installChart()
	{
		if (dataBook != null && sXColumnName != null && saYColumnNames != null)
		{
			try
			{
				Plot plot = null;
				
				if (chartStyle == STYLE_PIE)
				{
					dataset = new DataBookCategoryDataset(dataBook, sXColumnName, saYColumnNames, this);
					
					TableOrder tableOrder = TableOrder.BY_COLUMN;
					if (saYColumnNames.length > 1)
					{
						tableOrder = TableOrder.BY_ROW;
					}
					
					plot = new PiePlot(new CategoryToPieDataset((CategoryDataset)dataset, tableOrder, 0));
					
					((PiePlot)plot).setLabelGenerator(new StandardPieSectionLabelGenerator(
							"{0}: {2}",
							new DecimalFormat("0"),
							new DecimalFormat("0%")));
				}
				else
				{
					ColumnDefinition xColumnDefinition = dataBook.getRowDefinition().getColumnDefinition(sXColumnName);
					
					boolean isXDate = Date.class.isAssignableFrom(xColumnDefinition.getDataType().getTypeClass());
					boolean isXString = String.class.isAssignableFrom(xColumnDefinition.getDataType().getTypeClass());
					
					if (isXString)
					{
						dataset = new DataBookCategoryDataset(dataBook, sXColumnName, saYColumnNames, this);
					}
					else
					{
						dataset = new DataBookXYDataset(dataBook, sXColumnName, saYColumnNames, this);
					}
					
					AbstractRenderer renderer;
					if (chartStyle == STYLE_AREA)
					{
						if (isXString)
						{
							renderer = new AreaRenderer();
						}
						else
						{
							renderer = new XYAreaRenderer(XYAreaRenderer.AREA);
						}
					}
					else if (chartStyle == STYLE_BARS)
					{
						if (isXString)
						{
							renderer = new BarRenderer();
						}
						else
						{
							renderer = new ClusteredXYBarRenderer(0, true);
						}
					}
					else
					{
						
						if (isXString)
						{
							renderer = new LineAndShapeRenderer(true, true);
						}
						else
						{
							renderer = new XYLineAndShapeRenderer(true, true);
						}
					}
					
					if (isXDate)
					{
						vaX = new DateAxis(translate(sXTitle));
					}
					else if (isXString)
					{
						vaX = new CategoryAxis(translate(sXTitle));
						if (chartStyle != STYLE_BARS)
						{
							((CategoryAxis)vaX).setCategoryMargin(0);
						}
					}
					else
					{
						vaX = new NumberAxis(translate(sXTitle));
				        ((NumberAxis)vaX).setAutoRangeIncludesZero(false);
					}
					
			        vaY = new NumberAxis(translate(sYTitle));
			        
			        if (isXString)
			        {
			        	plot = new CategoryPlot((CategoryDataset)dataset, (CategoryAxis)vaX, (ValueAxis)vaY, (CategoryItemRenderer)renderer);
			        }
			        else
			        {
			        	plot = new XYPlot((XYDataset)dataset, (ValueAxis)vaX, (ValueAxis)vaY, (XYItemRenderer)renderer);
			        }
			        
			        if (chartStyle == STYLE_AREA)
			        {
			        	plot.setForegroundAlpha(0.65F);
			        }
	
			        if (renderer instanceof XYItemRenderer)
			        {
				        ((XYItemRenderer)renderer).setBaseToolTipGenerator(new StandardXYToolTipGenerator());
				        ((XYItemRenderer)renderer).setURLGenerator(new StandardXYURLGenerator());
			        }
			        else if (renderer instanceof AbstractCategoryItemRenderer)
			        {
			        	((AbstractCategoryItemRenderer)renderer).setBaseToolTipGenerator(new StandardCategoryToolTipGenerator());
			        }
				}
				
				JFreeChart chart = new JFreeChart(translate(sTitle), JFreeChart.DEFAULT_TITLE_FONT, plot, true);
				
				// Hide the legend if it is a pie chart.
				chart.getLegend().setVisible(chartStyle != STYLE_PIE);
				
				chartPanel = new ChartPanel(chart);
				
				add(chartPanel, BorderLayout.CENTER);
				
				dataBook.addControl(this);
				
				notifyRepaint();
			}
			catch (ModelException ex)
			{
				// Do Nothing
			}
		}
	}

	/**
	 * Gets the ChartPanel displayed, or null if non is displayed.
	 * @return the ChartPanel.
	 */
	public ChartPanel getChartPanel()
	{
		return chartPanel;
	}
	
	//****************************************************************
	// Subclass definition
	//****************************************************************
	
	/**
	 * The {@link DataBookCategoryDataset} extends the
	 * {@link DefaultCategoryDataset} and provides data from an
	 * {@link IDataBook}.
	 * 
	 * @author Robert Zenz
	 */
	public static class DataBookCategoryDataset extends DefaultCategoryDataset
	{
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Class members
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		
	    /** The associated chart. */
	    private JVxChart chart;
		
	    /** The DataBook that serves as data source. */
	    private IDataBook dataBook = null;
	    
	    /** Column name for the x axis. */
	    private String xColumnName = null;
	    
	    /** Column names for the y axis. */
	    private String[] yColumnNames = null;
		
	    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	    // Initialization
	    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	    
		/**
		 * Creates a new instance of {@link DataBookCategoryDataset}.
		 *
		 * @param pDatabook the {@link IDataBook databook}.
		 * @param pXColumnName the {@link String X column name}.
		 * @param pYColumnNames the {@link String Y column names}.
		 * @param pChart the associated chart.
		 */
    	public DataBookCategoryDataset(IDataBook pDatabook, String pXColumnName, String[] pYColumnNames, JVxChart pChart)
		{
			dataBook = pDatabook;
			xColumnName = pXColumnName;
			yColumnNames = pYColumnNames;
			chart = pChart;
		}
    	
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	// Overwritten methods
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	
		/**
	     * {@inheritDoc}
	     */
	    @Override
		public void validateObject() throws InvalidObjectException
		{
			clear();
			
			try
			{
				if (chart.chartStyle == STYLE_PIE && yColumnNames.length > 1)
				{
					String rowKey = chart.translate(dataBook.getValueAsString(xColumnName));
					
					for (String yColumnName : yColumnNames)
					{
						String yColumnTitle = getColumnLabel(yColumnName);
						BigDecimal value = (BigDecimal)dataBook.getValue(yColumnName);
						
						addValue(value, rowKey, yColumnTitle);
					}
				}
				else
				{
					dataBook.fetchAll();
					
					for (int index = 0; index < dataBook.getRowCount(); index++)
					{
						IDataRow dataRow = dataBook.getDataRow(index);
						String rowKey =  chart.translate(dataRow.getValueAsString(xColumnName));
						for (String yColumnName : yColumnNames)
						{
							String yColumnTitle = getColumnLabel(yColumnName);
							
							BigDecimal value = (BigDecimal)dataRow.getValue(yColumnName);
							
							if (chart.chartStyle == STYLE_PIE)
							{
								addValue(value, rowKey, yColumnTitle);
							}
							else
							{
								addValue(value, yColumnTitle, rowKey);
							}
						}
					}
				}
			}
			catch (ModelException e)
			{
				throw new InvalidObjectException(e.getMessage());
			}
			
			super.validateObject();
		}
		
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// User-defined methods
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		
		/**
		 * Gets the label for the given column.
		 * 
		 * @param pColumnName the name of the column.
		 * @return the label of the column.
		 * @throws ModelException if accessing the column information failed.
		 */
		private String getColumnLabel(String pColumnName) throws ModelException
		{
			ColumnDefinition columnDefinition = dataBook.getRowDefinition().getColumnDefinition(pColumnName);
			
			String columnLabel = columnDefinition.getLabel();
			
			if (columnLabel == null)
			{
				columnLabel = columnDefinition.getDefaultLabel();
			}
			
			columnLabel = chart.translate(columnLabel);
			
			return columnLabel;
		}
	
	}	// DataBookCategoryDataset
	
	/**
	 * This class provides an {@link AbstractIntervalXYDataset} implementation for a DataBook.  
	 */
	public static class DataBookXYDataset extends AbstractIntervalXYDataset 
	                                      implements XYDataset, 
	                                                 TableXYDataset 
	{
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Class members
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		/** The range. */
		private static BigDecimal percentage = new BigDecimal("0.9");
		
	    /** The associated chart. */
	    private JVxChart chart;

	    /** The DataBook that serves as data source. */
	    private IDataBook dataBook = null;

	    /** Column name for the x axis. */
	    private String xColumnName = null;
	    
	    /** Column names for the y axis. */
	    private String[] yColumnNames = null;

	    /** minimal difference of x Values. */
	    private BigDecimal minXDiff = null;

	    /** If there is a date axis. */
	    private boolean bDateAxis;

		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Initialization
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	    /**
	     * Creates a new JDBCXYDataset (initially empty) with no database
	     * connection.
	     * 
	     * @param pDataBook the DataBook
	     * @param pXColumnName the x column name
	     * @param pYColumnNames the y column names
	     * @param pChart the chart
	     * 
	     * @throws ModelException if an Exception occurs.
	     */
	    public DataBookXYDataset(IDataBook pDataBook, String pXColumnName, String[] pYColumnNames, JVxChart pChart) throws ModelException
	    {
	    	dataBook = pDataBook;
	    	
	    	xColumnName = pXColumnName;
	    	yColumnNames = pYColumnNames; 
	    	
	    	chart = pChart;
    		bDateAxis = Date.class.isAssignableFrom(dataBook.getRowDefinition().getColumnDefinition(xColumnName).getDataType().getTypeClass());
	    }

		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	    // Interface implementation
	    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		/**
	     * {@inheritDoc}
	     */
	    @Override
		public void validateObject() throws InvalidObjectException
		{
	    	minXDiff = null;
	    	
			super.validateObject();
		}
	    
	    /**
	     * {@inheritDoc}
	     */
	    public Number getX(int pSeriesIndex, int pItemIndex)
	    {
	    	try
	    	{
		    	IDataRow dataRow = dataBook.getDataRow(pItemIndex);

		    	if (bDateAxis)
		    	{
		    		return Long.valueOf(((Date)dataRow.getValue(xColumnName)).getTime());
		    	}
		    	else
		    	{
		    		return (Number)dataRow.getValue(xColumnName);
		    	}
	    	}
	    	catch (Exception ex)
	    	{
	    		return null;
	    	}
	    }

	    /**
	     * {@inheritDoc}
	     */
	    public Number getY(int pSeriesIndex, int pItemIndex) 
	    {
	    	try
	    	{
		    	IDataRow dataRow = dataBook.getDataRow(pItemIndex);

		        return (Number)dataRow.getValue(yColumnNames[pSeriesIndex]);
	    	}
	    	catch (Exception ex)
	    	{
	    		return null;
	    	}
	    }

	    /**
	     * {@inheritDoc}
	     */
		public Number getStartX(int pSeriesIndex, int pItemIndex)
		{
			return getX(pSeriesIndex, pItemIndex);
		}

	    /**
	     * {@inheritDoc}
	     */
		public Number getEndX(int pSeriesIndex, int pItemIndex)
		{
			if (minXDiff == null)
			{
				int count = getItemCount();
				
				Number value = getX(pSeriesIndex, 0);
				
				for (int i = 1; i < count; i++)
				{
					Number nextValue = getX(pSeriesIndex, i);
					
					if (nextValue != null)
					{
						if (value != null)
						{
							double diff = Math.abs(nextValue.doubleValue() - value.doubleValue());
							
							if (minXDiff == null || diff < minXDiff.doubleValue())
							{
								minXDiff = BigDecimal.valueOf(diff);
							}
						}
						value = nextValue;
					}
				}
				
				if (minXDiff == null)
				{
					if (value == null || value.doubleValue() == 0)
					{
						minXDiff = BigDecimal.valueOf(1);
					}
					else
					{
						minXDiff = BigDecimal.valueOf(value.doubleValue());
					}
					
				}
			}
			
			Number startX = getX(pSeriesIndex, pItemIndex);
			
			if (startX == null)
			{
				return null;
			}
			else
			{
				return BigDecimal.valueOf(startX.doubleValue() + minXDiff.doubleValue() * percentage.doubleValue());
			}
		}

	    /**
	     * {@inheritDoc}
	     */
		public Number getStartY(int pSeriesIndex, int pItemIndex)
		{
			return getY(pSeriesIndex, pItemIndex);
		}

	    /**
	     * {@inheritDoc}
	     */
		public Number getEndY(int pSeriesIndex, int pItemIndex)
		{
			double start = getY(pSeriesIndex, pItemIndex).doubleValue();
			
			double diff;
			if (pItemIndex < getItemCount() - 1)
			{
				diff = Math.abs(getY(pSeriesIndex, pItemIndex + 1).doubleValue() - start);
			}
			else if (pItemIndex == 0)
			{
				if (start == 0)
				{
					diff = start;
				}
				else
				{
					diff = 1;
				}
			}
			else
			{
				diff = Math.abs(start - getY(pSeriesIndex, pItemIndex - 1).doubleValue());
			}
			
			return BigDecimal.valueOf(start + diff * percentage.doubleValue());
		}

	    /**
	     * {@inheritDoc}
	     */
	    public int getItemCount(int seriesIndex) 
	    {
	    	return getItemCount();
	    }

	    /**
	     * {@inheritDoc}
	     */
	    public int getItemCount() 
	    {
	    	try
	    	{
	    		dataBook.fetchAll();
	    		
	    		return dataBook.getRowCount();
	    	}
	    	catch (Exception ex)
	    	{
	    		return 0;
	    	}
	    }

	    /**
	     * {@inheritDoc}
	     */
	    public int getSeriesCount() 
	    {
	        return yColumnNames.length;
	    }

	    /**
	     * {@inheritDoc}
	     */
	    public Comparable getSeriesKey(int pSeriesIndex) 
	    {
	    	try
	    	{
		        return chart.translate(dataBook.getRowDefinition().getColumnDefinition(yColumnNames[pSeriesIndex]).getLabel());
	    	}
	    	catch (Exception ex)
	    	{
		    	return chart.translate(yColumnNames[pSeriesIndex]);
	    	}
	    }
	    
	}	// DataBookXYDataset

}	// JVxChart
