/*
 * Copyright 2015 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 11.11.2015 - [RZ] - creation
 */
package com.sibvisions.rad.ui.celleditor;

import java.lang.ref.WeakReference;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.rad.genui.celleditor.UICellEditor;
import javax.rad.model.ColumnView;
import javax.rad.model.IDataBook;
import javax.rad.model.IDataPage;
import javax.rad.model.IDataRow;
import javax.rad.model.IRowDefinition;
import javax.rad.model.ModelException;
import javax.rad.model.condition.CompareCondition;
import javax.rad.model.condition.Equals;
import javax.rad.model.condition.ICondition;
import javax.rad.model.condition.Like;
import javax.rad.model.condition.LikeIgnoreCase;
import javax.rad.model.condition.OperatorCondition;
import javax.rad.model.datatype.IDataType;
import javax.rad.model.reference.ColumnMapping;
import javax.rad.model.reference.ReferenceDefinition;
import javax.rad.model.ui.ICellEditor;
import javax.rad.model.ui.IControl;
import javax.rad.model.ui.ITableControl;
import javax.rad.ui.IDimension;
import javax.rad.ui.celleditor.ILinkedCellEditor;
import javax.rad.ui.celleditor.IStyledCellEditor;

import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.type.CommonUtil;

/**
 * The {@link AbstractLinkedCellEditor} is an {@link ILinkedCellEditor}
 * implementation, which provides a base implementation.
 * 
 * @author Robert Zenz
 */
public abstract class AbstractLinkedCellEditor extends AbstractComboCellEditor implements ILinkedCellEditor
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/** True, if header should be visible depending of column number. */
	protected boolean autoTableHeaderVisibility = true;
	
	/** The additional condition. */
	protected ICondition additionalCondition;
	
	/** The {@link ColumnView}. */
	protected ColumnView columnView;
	
	/** The name of the display referenced column. */
	protected String displayReferencedColumnName = null;
	
	/** The display concat mask. */
	protected String displayConcatMask = null;
	
	/** The link reference. */
	protected ReferenceDefinition linkReference;
	
	/** The size used for the popup. */
	protected IDimension popupSize;
	
	/** The {@link ColumnMapping}. */
	protected ColumnMapping searchColumnMapping;
	
	/** If the text should be searched anywhere inside a column. */
	protected boolean searchTextAnywhere = true;
	
	/** If the text should be searched in all visible table columns. */
	protected boolean searchInAllTableColumns = false;
	
	/** If the values should be sorted by column name. */
	protected boolean sortByColumnName;
	
	/** If the table header should be visible. */
	protected boolean tableHeaderVisible = false;
	
	/** If the table should be read-only. */
	protected boolean tableReadOnly = true;
	
	/** If only values from the table are allowed. */
	protected boolean validationEnabled = true;
	
	/**
	 * The display values will be cached in this {@link Map} so that the
	 * conversion to a String is only done once in a render cycle.
	 */
	private Map<String, Integer> displayValueCache = new HashMap<String, Integer>();
	
	/** The last column, with which was searched. */
	private String lastColumnForSearch = null;
	
	/** The last display column. */
	private String lastDisplayColumn = null;
	
	/** The last data page, in which was searched. */
	private WeakReference<IDataPage> lastPageForSearch = null;
	
	/**
	 * The last search row, which is used when building the display value cache.
	 */
	private int lastSearchRow = 0;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Creates a new instance of {@link AbstractLinkedCellEditor}.
	 */
	protected AbstractLinkedCellEditor()
	{
		super();
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * {@inheritDoc}
	 */
	public ICondition getAdditionalCondition()
	{
		return additionalCondition;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public ColumnView getColumnView()
	{
		return columnView;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public String getDisplayReferencedColumnName()
	{
		return displayReferencedColumnName;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public String getDisplayConcatMask()
	{
		return displayConcatMask;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public ReferenceDefinition getLinkReference()
	{
		return linkReference;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public IDimension getPopupSize()
	{
		return popupSize;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public ColumnMapping getSearchColumnMapping()
	{
		return searchColumnMapping;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public boolean isSearchTextAnywhere()
	{
		return searchTextAnywhere;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public boolean isSearchInAllTableColumns()
	{
		return searchInAllTableColumns;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public boolean isSortByColumnName()
	{
		return sortByColumnName;
	}
	
	/**
	 * The header of a table can't be hidden.
	 * 
	 * @return always {@code false}.
	 */
	public boolean isTableHeaderVisible()
	{
		return tableHeaderVisible;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public boolean isTableReadonly()
	{
		return tableReadOnly;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public boolean isValidationEnabled()
	{
		return validationEnabled;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setAdditionalCondition(ICondition pCondition)
	{
		additionalCondition = pCondition;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setColumnView(ColumnView pColumnView)
	{
		columnView = pColumnView;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setDisplayReferencedColumnName(String pDisplayReferencedColumnName)
	{
		displayReferencedColumnName = pDisplayReferencedColumnName;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setDisplayConcatMask(String pDisplayConcatMask)
	{
		displayConcatMask = pDisplayConcatMask;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setLinkReference(ReferenceDefinition pReferenceDefinition)
	{
		linkReference = pReferenceDefinition;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setPopupSize(IDimension pPopupSize)
	{
		popupSize = pPopupSize;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setSearchColumnMapping(ColumnMapping pSearchColumnNames)
	{
		searchColumnMapping = pSearchColumnNames;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setSearchTextAnywhere(boolean pSearchTextAnywhere)
	{
		searchTextAnywhere = pSearchTextAnywhere;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setSearchInAllTableColumns(boolean pSearchInAllTableColumns)
	{
		searchInAllTableColumns = pSearchInAllTableColumns;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setSortByColumnName(boolean pSortByColumnName)
	{
		sortByColumnName = pSortByColumnName;
	}
	
	/**
	 * Does nothing, the header of a table can't be hidden.
	 * 
	 * @param pTableHeaderVisible ignored.
	 */
	public void setTableHeaderVisible(boolean pTableHeaderVisible)
	{
		autoTableHeaderVisibility = false;
		
		tableHeaderVisible = pTableHeaderVisible;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setTableReadonly(boolean pTableReadonly)
	{
		tableReadOnly = pTableReadonly;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setValidationEnabled(boolean pValidationEnabled)
	{
		validationEnabled = pValidationEnabled;
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isDirectCellEditor()
	{
		return false;
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Gets the default horizontal alignment based on data type.
	 * If default alignment is unknown, ALIGN_LEFT is returned.
	 * 
	 * @param pDataRow the data row
	 * @param pColumnName the column name
	 * @return the default horizontal alignment
	 */
	public int getDefaultHorizontalAlignment(IDataRow pDataRow, String pColumnName)
	{
		int hAlignment = getHorizontalAlignment();
		if (hAlignment == ALIGN_DEFAULT)
		{
			try
			{
				IDataType dataType = pDataRow.getRowDefinition().getColumnDefinition(pColumnName).getDataType();
				ICellEditor editor = dataType.getCellEditor();
				if (editor != this && editor instanceof IStyledCellEditor)
				{
					hAlignment = ((IStyledCellEditor)editor).getHorizontalAlignment();
				}
				
				if (hAlignment == ALIGN_DEFAULT && displayReferencedColumnName == null && displayConcatMask == null)
				{
					editor = UICellEditor.getDefaultCellEditor(dataType.getTypeClass());
					
					if (editor != this && editor instanceof IStyledCellEditor)
					{
						hAlignment = ((IStyledCellEditor)editor).getHorizontalAlignment();
					}
				}
			}
			catch (Exception pException)
			{
				// Ignore
			}
			if (hAlignment == ALIGN_DEFAULT)
			{
				hAlignment = ALIGN_LEFT;
			}
		}
		
		return hAlignment;
	}

	/**
	 * Gets the correct display value for the given row and column.
	 * 
	 * @param pDataRow the data row
	 * @param pColumnName the column
	 * @return the display value.
	 * @throws ModelException if it fails.
	 */
	protected String getDisplayValue(IDataRow pDataRow, String pColumnName) throws ModelException
	{
		IDataType dataType = pDataRow.getRowDefinition().getColumnDefinition(pColumnName).getDataType();
		
		Object value = pDataRow.getValue(pColumnName);
		String stringValue = dataType.convertToString(value);
		
		if ((displayReferencedColumnName != null || displayConcatMask != null) && linkReference != null)
		{
			Integer index = displayValueCache.get(stringValue);

			IDataBook refDataBook = linkReference.getReferencedDataBook();
			IDataPage page = refDataBook.getDataPage();
			
			String refColumn = linkReference.getReferencedColumnName(pColumnName);
			IDataType refDataType = refDataBook.getRowDefinition().getColumnDefinition(refColumn).getDataType();
			
			IDataRow refDataRow = null;
			String refStringValue = null;
			
			ICondition originalFilter = refDataBook.getFilter();
			IControl[] controls = null;
			int selectedRow = -1;
			if (originalFilter != null) // ensure, that the filter is null, otherwise value will not be found
			{
				controls = refDataBook.getControls();
				for (IControl control : controls)
				{
					refDataBook.removeControl(control);
				}
				
				selectedRow = refDataBook.getSelectedRow();
				refDataBook.setFilter(null);
			}

			if (index != null)
			{
				refDataRow = refDataBook.getDataRow(index.intValue());
				if (refDataRow != null)
				{
					Object refValue = refDataRow.getValue(refColumn);
					try
					{
						refStringValue = dataType.convertToString(dataType.convertToTypeClass(refValue));
					}
					catch (Exception ex)
					{
						refStringValue = refDataType.convertToString(refValue);
					}
				}
			}
			
			if (lastPageForSearch == null 
					|| lastPageForSearch.get() != page
					|| !CommonUtil.equals(refColumn, lastColumnForSearch)
					|| !CommonUtil.equals(displayReferencedColumnName, lastDisplayColumn)
					|| (index != null && (refDataRow == null || !CommonUtil.equals(stringValue, refStringValue))))
			{
				displayValueCache.clear();
				
				lastPageForSearch = new WeakReference<IDataPage>(page);
				lastColumnForSearch = refColumn;
				lastDisplayColumn = displayReferencedColumnName;
				
				lastSearchRow = 0;
				index = null;
			}
			
			if (index == null)
			{
				IDataRow curDataRow = refDataBook.getDataRow(lastSearchRow);
				
				while (curDataRow != null && refDataRow == null) // step through the data book beginning with the last search row
				{
					Object refValue = curDataRow.getValue(refColumn);
					try
					{
						refStringValue = dataType.convertToString(dataType.convertToTypeClass(refValue));
					}
					catch (Exception ex)
					{
						refStringValue = refDataType.convertToString(refValue);
					}

					displayValueCache.put(refStringValue, Integer.valueOf(lastSearchRow));
					
					if (CommonUtil.equals(stringValue, refStringValue))
					{
						refDataRow = curDataRow;
					}
					else
					{
						lastSearchRow++;
						curDataRow = refDataBook.getDataRow(lastSearchRow);
					}
				}
			}
			if (originalFilter != null) // ensure, that the filter is back, for correct displaying
			{
				refDataBook.setFilter(originalFilter);
				refDataBook.setSelectedRow(selectedRow); // without this, no sync will occur, and endless repaint will start.
				
				for (IControl control : controls)
				{
					refDataBook.addControl(control);
				}
			}
			if (refDataRow != null)
			{
				String displayValue = getDisplayValueFromRow(refDataRow);
				
				if (displayValue != null)
				{
					return displayValue;
				}
			}
		}

		return stringValue;
	}
	
	/**
	 * Gets the display value depending on displayConcatMask or displayReferencedColumnName.
	 * @param pDataRow the data row.
	 * @return the display value depending on displayConcatMask or displayReferencedColumnName.
	 * @throws ModelException the model Exception.
	 */
	protected String getDisplayValueFromRow(IDataRow pDataRow) throws ModelException
	{
		if (displayConcatMask != null)
		{
			ColumnView cv = columnView;
			if (cv == null)
			{
				cv = linkReference.getReferencedDataBook().getRowDefinition().getColumnView(ITableControl.class);
			}
			int count = cv.getColumnCount();
			
			StringBuilder result = new StringBuilder();
			int index = displayConcatMask.indexOf('*');
			if (index < 0)
			{
				if (count == 0)
				{
					return null;
				}
				else
				{
					result.append(CommonUtil.nvl(pDataRow.getValueAsString(cv.getColumnName(0)), ""));
					for (int i = 1; i < count; i++)
					{
						result.append(displayConcatMask);
						result.append(CommonUtil.nvl(pDataRow.getValueAsString(cv.getColumnName(i)), ""));
					}
				}
			}
			else
			{
				int i = 0;
				int start = 0;
				
				while (index >= 0)
				{
					result.append(displayConcatMask.substring(start, index));
					
					if (i < count)
					{
						result.append(CommonUtil.nvl(pDataRow.getValueAsString(cv.getColumnName(i)), ""));
						i++;
					}
					else
					{
						result.append("");
					}
					start = index + 1;
					index = displayConcatMask.indexOf('*', start);
				}	
				result.append(displayConcatMask.substring(start));
			}
			
			return result.toString();
		}
		else
		{
			return pDataRow.getValueAsString(displayReferencedColumnName);
		}
	}
	
	/**
	 * Searches for columns, that should not be cleared, if the value of this linked cell editor is cleared.
	 * @param pDataRow the data row
	 * @param pColumnName the column name
	 * @return do not clear columns. 
	 * @throws ModelException if it fails.
	 */
	protected String[] getClearColunms(IDataRow pDataRow, String pColumnName) throws ModelException
	{
		IRowDefinition rowDef = pDataRow.getRowDefinition();
		String[] allColumns = rowDef.getColumnNames();
		String[] linkColumns = getLinkReference().getColumnNames();
		String[] searchColumns = getAllSearchColumns(getSearchColumnMapping(), pDataRow, getAdditionalCondition()); 
		
		ArrayUtil<String> columns = new ArrayUtil<String>();
		columns.addAll(linkColumns);
		
		for (String colName : allColumns)
		{
			ICellEditor ce = rowDef.getColumnDefinition(colName).getDataType().getCellEditor();
			
			if (ce instanceof ILinkedCellEditor)
			{
				ILinkedCellEditor linkCe = (ILinkedCellEditor)ce;
				ReferenceDefinition refDef = linkCe.getLinkReference();
				
				if (refDef != null)
				{
					String[] refDefCols = refDef.getColumnNames();
					if (ArrayUtil.intersect(refDefCols, searchColumns).length > 0
						&& !ArrayUtil.contains(refDefCols, pColumnName))
					{
						columns.removeAll(refDefCols);
					}
				}
			}
		}
		return columns.toArray(new String[columns.size()]);
	}
	
	/**
	 * Searches for additional columns to be cleared on save.
	 * 
	 * @param pDataRow the data row.
	 * @return additional clear columns.
	 * @throws ModelException if it fails.
	 */
	protected String[] getAdditionalClearColumns(IDataRow pDataRow) throws ModelException
	{
		ArrayUtil<String> columns = new ArrayUtil<String>();
		
		IRowDefinition rowDef = pDataRow.getRowDefinition();
		String[] allColumns = rowDef.getColumnNames();
		String[] linkColumns = getLinkReference().getColumnNames();
		String[] searchColumns = getAllSearchColumns(getSearchColumnMapping(), pDataRow, getAdditionalCondition()); 
		
		for (String colName : allColumns)
		{
			ICellEditor ce = rowDef.getColumnDefinition(colName).getDataType().getCellEditor();
			
			if (ce instanceof ILinkedCellEditor)
			{
				ILinkedCellEditor linkCe = (ILinkedCellEditor)ce;
				ReferenceDefinition refDef = linkCe.getLinkReference();
				String[] searchCols = ArrayUtil.removeAll(
						getAllSearchColumns(linkCe.getSearchColumnMapping(), pDataRow, linkCe.getAdditionalCondition()), searchColumns);
				
				if (refDef != null && searchCols != null
						&& ArrayUtil.intersect(linkColumns, searchCols).length > 0)
				{
						String[] addCols = ArrayUtil.removeAll(refDef.getColumnNames(), linkColumns);
    						
					for (String col : addCols)
					{
						if (!columns.contains(col))
						{
							columns.add(col);
						}
					}
				}
			}
		}

		return columns.toArray(new String[columns.size()]);
	}
	
	/**
	 * Gets all search columns using the given data row as filter.
	 * 
	 * @param pSearchColumnMapping the columnMapping.
	 * @param pDataRow the data row.
	 * @param pCondition the additional condition.
	 * @return the list of all found condition columns.
	 */
	protected String[] getAllSearchColumns(ColumnMapping pSearchColumnMapping, IDataRow pDataRow, ICondition pCondition)
	{
		ArrayUtil<String> conditionColumns = new ArrayUtil<String>();
		
		if (pSearchColumnMapping != null)
		{
			conditionColumns.addAll(pSearchColumnMapping.getColumnNames());
		}
		
		fillInConditionColumns(pCondition, pDataRow, conditionColumns);
		
		int size = conditionColumns.size();
		if (size == 0)
		{
			return null;
		}
		else
		{
			return conditionColumns.toArray(new String[size]);
		}
	}

	/**
	 * Searches the condition columns.
	 * 
	 * @param pCondition the condition.
	 * @param pDataRow the datarow.
	 * @param pConditionColumns the list of all found condition columns.
	 */
	private void fillInConditionColumns(ICondition pCondition, IDataRow pDataRow, List<String> pConditionColumns)
	{
		if (pCondition instanceof CompareCondition)
		{
			CompareCondition cond = (CompareCondition)pCondition;
			String colName = cond.getDataRowColumnName();
			if (pDataRow == cond.getDataRow() && !pConditionColumns.contains(colName))
			{
				pConditionColumns.add(colName);
			}
		}
		else if (pCondition instanceof OperatorCondition)
		{
			ICondition[] conditions = ((OperatorCondition)pCondition).getConditions();
			
			for (int i = 0; i < conditions.length; i++)
			{
				fillInConditionColumns(conditions[i], pDataRow, pConditionColumns);
			}
		}
	}
		
	/**
	 * Creates a search string.
	 * 
	 * @param pItem item
	 * @return a search string.
	 */
	protected String getWildCardString(Object pItem)
	{
		if (isSearchTextAnywhere())
		{
			return "*" + pItem + "*";
		}
		else
		{
			return pItem + "*";
		}
	}
	
	/**
	 * Creates a Condition including the search columns.
	 * 
	 * @param pDataRow   the base data row.
	 * @param pCondition the base condition.
	 * @return a Condition including the search columns.
	 */
	protected ICondition getSearchCondition(IDataRow pDataRow, ICondition pCondition)
	{
		if (pCondition == null)
		{
			pCondition = additionalCondition;
		}
		else if (additionalCondition != null)
		{
			pCondition = pCondition.and(additionalCondition);
		}
		
		if (searchColumnMapping != null)
		{
			String[] searchColumns = searchColumnMapping.getColumnNames();
			String[] referencedSearchColumns = searchColumnMapping.getReferencedColumnNames();
			
			for (int i = 0; i < searchColumns.length; i++)
			{
				ICondition condition = new Equals(pDataRow, searchColumns[i], referencedSearchColumns[i]);
				
				if (pCondition == null)
				{
					pCondition = condition;
				}
				else
				{
					pCondition = pCondition.and(condition);
				}
			}
		}

		return pCondition;
	}

	/**
	 * Gets the search condition for the input item.
	 * 
	 * @param pSearchWildCard true, if wildcard search should be done.
	 * @param pRelevantSearchColumnName the relevant search column.
	 * @param pItem the item to search.
	 * @return the search condition for the input item.
	 */
	protected ICondition getItemSearchCondition(boolean pSearchWildCard, String pRelevantSearchColumnName, Object pItem)
	{
		Object searchString = pSearchWildCard ? getWildCardString(pItem) : pItem;
		String[] columns = null;
		if (searchInAllTableColumns || displayConcatMask != null)
		{
			if (columnView == null)
			{
				columns = linkReference.getReferencedDataBook().getRowDefinition().getColumnView(ITableControl.class).getColumnNames();
			}
			else
			{
				columns = columnView.getColumnNames();
			}
		}
		if (columns == null || columns.length == 0)
		{
			columns = new String[] {pRelevantSearchColumnName};
		}

		ICondition result = pSearchWildCard ? new LikeIgnoreCase(columns[0], searchString) : new Like(columns[0], pItem);
		
		for (int i = 1; i < columns.length; i++)
		{
			result = result.or(pSearchWildCard ? new LikeIgnoreCase(columns[i], searchString) : new Like(columns[i], pItem));
		}
		
		return result;
	}
	
}	// AbstractLinkedCellEditor
