/*
 * Copyright 2011 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 * 
 * 11.12.2011 - [JR] - creation
 * 18.06.2015 - [JR] - #1413: IFileHandle support
 * 21.06.2018 - [JR] - recording implemented
 */
package com.sibvisions.rad.server.http.rest.service;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;

import javax.rad.remote.IConnectionConstants;

import org.restlet.Request;
import org.restlet.data.Status;
import org.restlet.representation.Representation;
import org.restlet.resource.Post;

import com.sibvisions.rad.server.DirectServerSession;
import com.sibvisions.rad.server.http.rest.JSONUtil;
import com.sibvisions.rad.server.http.rest.RESTAdapter;
import com.sibvisions.rad.server.protocol.ICategoryConstants;
import com.sibvisions.rad.server.protocol.ICommandConstants;
import com.sibvisions.rad.server.protocol.ProtocolFactory;
import com.sibvisions.rad.server.protocol.Record;
import com.sibvisions.util.log.LoggerFactory;
import com.sibvisions.util.type.CommonUtil;

/**
 * The <code>AdminService</code> is a special administrative service.
 * 
 * @author Ren Jahn
 */
public class AdminService extends AbstractService
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/** the DoS cache. */
	private static HashMap<String, Object[]> hmpDoSUser = new HashMap<String, Object[]>();
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Executes special admin calls.
	 * 
	 * @param pParameter the parameters
	 * @return the result
	 */
	@Post
	public Representation execute(Representation pParameter)
	{
		Record record = ProtocolFactory.openRecord(ICategoryConstants.REST, ICommandConstants.REST_GET);
		
		try
		{
			Request request = getRequest();
			
			ConcurrentMap<String, Object> cmpAttrib = request.getAttributes();
			
        	String sApplicationName = (String)getAttribute(RESTAdapter.PARAM_APP_NAME);
        	
			String sAction = (String)cmpAttrib.get(RESTAdapter.PARAM_ADMIN_ACTION);
			String sParam = (String)cmpAttrib.get(RESTAdapter.PARAM_ADMIN_ACTION_PARAM);
			
			LoggerFactory.getInstance(AdminService.class).debug(sAction);
			
			if ("changePassword".equals(sAction))
			{
				HashMap<String, Object> hmpInput = JSONUtil.getObject(pParameter, HashMap.class);
				
				if (hmpInput == null)
				{
					hmpInput = new HashMap<String, Object>();
				}
				
				HashMap<String, Object> hmpParams = RESTAdapter.createConnectionProperties(((RESTAdapter)getApplication()), request);

				String sUserName = (String)hmpInput.get("username");
				
				if (sUserName == null)
				{
					sUserName = sParam;
				}
				
				checkDoS(sUserName);
				
				String sOldPassword = (String)hmpInput.get("oldpassword");
				String sNewPassword = (String)hmpInput.get("newpassword");
				
				if (sNewPassword != null)
				{
					hmpParams.put(IConnectionConstants.OLDPASSWORD, sOldPassword);
					
					if (!hmpParams.containsKey(IConnectionConstants.NEWPASSWORD))
					{
						hmpParams.put(IConnectionConstants.NEWPASSWORD, sNewPassword);
					}
				}
				
				DirectServerSession session = DirectServerSession.createMasterSession(sApplicationName, sUserName, sOldPassword, hmpParams);
				
				try
				{
					session.destroy();
				}
				catch (Throwable th)
				{
					//ignore
				}
				finally
				{
					clearDoS(sUserName);
				}
				
				setStatus(Status.SUCCESS_NO_CONTENT);
			}
			else if ("testAuthentication".equals(sAction))
			{
				HashMap<String, Object> hmpInput = JSONUtil.getObject(pParameter, HashMap.class);
				
				if (hmpInput == null)
				{
					hmpInput = new HashMap<String, Object>();
				}
				
				HashMap<String, Object> hmpParams = RESTAdapter.createConnectionProperties(((RESTAdapter)getApplication()), request);

				String sUserName = (String)hmpInput.get("username");
				
				checkDoS(sUserName);
				
				if (sUserName == null)
				{
					sUserName = sParam;
				}
				
				String sPassword = (String)hmpInput.get("password");
				
				DirectServerSession session = DirectServerSession.createMasterSession(sApplicationName, sUserName, sPassword, hmpParams);
				
				try
				{
					session.destroy();
				}
				catch (Throwable th)
				{
					//ignore
				}
				finally
				{
					clearDoS(sUserName);
				}
				
				setStatus(Status.SUCCESS_NO_CONTENT);
			}
			else
			{
				setStatus(Status.CLIENT_ERROR_NOT_FOUND);
				return null;
			}
			
		    return toInternalRepresentation(null);
		}
		catch (Throwable th)
		{
			LoggerFactory.getInstance(AdminService.class).error(th);			
			
			if (record != null)
			{
				record.setException(th);
			}
			
		    return handleException(th);
		}
		finally
		{
			CommonUtil.close(record);
		}
	}
	
	/**
	 * Checks whether a user is able to use the service. A user is unable to use it, if 3 connect attempts failed.
	 * The user will be able to try it again after a timeout of 3 minutes.
	 * 
	 * @param pUserName the user name
	 */
	protected void checkDoS(String pUserName)
	{
		//cleanup
		Set<Entry<String, Object[]>> set = hmpDoSUser.entrySet();
		
		long lNow = System.currentTimeMillis();
		
		for (Iterator<Entry<String, Object[]>> it = set.iterator(); it.hasNext();)
		{
			Entry<String, Object[]> entry = it.next();
			
			Object[] oValue = entry.getValue();
			
			//remove after 3 minutes "inactivity"
			if (((Long)oValue[0]).longValue() + 180000 < lNow)
			{
				set.remove(entry);
			}
		}
		
		//check
		Object[] oUser = (Object[])hmpDoSUser.get(pUserName);
		
		
		Long lTime = Long.valueOf(System.currentTimeMillis());
		
		int iCount;
		
		if (oUser != null)
		{
			iCount = ((Integer)oUser[1]).intValue() + 1;
		}
		else
		{
			iCount = 0;
		}			
	
		if (iCount >= 3)
		{
			//update time
			hmpDoSUser.put(pUserName, new Object[] {lTime, Integer.valueOf(2)});
			
			setStatus(Status.CLIENT_ERROR_TOO_MANY_REQUESTS);
			throw new SecurityException();
		}
		else
		{
			hmpDoSUser.put(pUserName, new Object[] {lTime, Integer.valueOf(iCount)});
		}
	}
	
	/**
	 * Removes the user from the DoS cache.
	 * 
	 * @param pUserName the user name
	 */
	private void clearDoS(String pUserName)
	{
		hmpDoSUser.remove(pUserName);
	}
	
}	// AdminService
