/*
 * Copyright 2018 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 * 
 * 03.12.2018 - [JR] - creation
 */
package com.sibvisions.rad.server.http.rest.security;

import java.util.logging.Level;

import org.restlet.Context;
import org.restlet.Request;
import org.restlet.Response;
import org.restlet.data.Status;
import org.restlet.security.Authenticator;
import org.restlet.security.Verifier;

import com.sibvisions.rad.server.http.rest.LifeCycleConnector;

/**
 * The <code>ForwardAuthenticator</code> is a simple {@link Authenticator} for authentication with a simple {@link Verifier}.
 * 
 * @author Ren Jahn
 */
public class ForwardAuthenticator extends Authenticator
{
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Class members
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/** the verifier to use. */
	private Verifier verifier;
	
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Initialization
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Creates a new instance of <code>ForwardAuthenticator</code>.
	 * 
	 * @param pContext the context
	 */
	public ForwardAuthenticator(Context pContext)
	{
		super(pContext);
	}
	
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Abstract methods implementation
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	protected boolean authenticate(Request pRequest, Response pResponse) 
	{
		boolean loggable = pRequest.isLoggable() && getLogger().isLoggable(Level.FINE);
		
		
		if (verifier != null)
		{
			int iVerification = verifier.verify(pRequest, pResponse);
			
			if (iVerification == Verifier.RESULT_VALID)
			{
				if (loggable)
				{
					getLogger().fine("Authentication succeeded. Valid credentials provided.");
				}
				
				return true;
			}
			else
			{
				if (!(super.getNext() instanceof Authenticator))
				{
					forbid(pResponse);
				}

				return false;
			}
		}
		
		if (loggable)
		{
			getLogger().fine("Authentication failed. Invalid credentials provided.");
		}		
		
		return false;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected int unauthenticated(Request pRequest, Response pResponse)
	{
		if (super.getNext() instanceof Authenticator)
		{
			//the forwarder shouldn't stop the authentication mechanism. If there's another authenticator as next
			return CONTINUE;
		}
		else
		{
			return super.unauthenticated(pRequest, pResponse);
		}
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void afterHandle(Request pRequest, Response pResponse)
	{
		try
		{
			super.afterHandle(pRequest, pResponse);
		}
		finally
		{
			LifeCycleConnector.destory(pRequest);
		}
	}
	
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // User-defined methods
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	

	/**
	 * Sets the credentials verifier.
	 * 
	 * @param pVerifier the credentials verifier
	 */
	public void setVerifier(Verifier pVerifier)
	{
		verifier = pVerifier;
	}
	
	/**
	 * Gets the credentials verifier.
	 * 
	 * @return the credentials verifier
	 */
	public Verifier getVerifier()
	{
		return verifier;
	}
	
    /**
     * Rejects the call due to a failed authentication or authorization. This can be overridden to 
     * change the default behavior, for example to display an error page. By default, if
     * authentication is required, the challenge method is invoked, otherwise the call status is 
     * set to CLIENT_ERROR_FORBIDDEN.
     * 
     * @param pResponse the reject response
     */
    private void forbid(Response pResponse) 
    {
        if (pResponse.getRequest().isLoggable() 
        	&& getLogger().isLoggable(Level.FINE)) 
        {
            getLogger().log(Level.FINE, "Authentication or authorization failed for this URI: " + pResponse.getRequest().getResourceRef());
        }

        pResponse.setStatus(Status.CLIENT_ERROR_FORBIDDEN);
    }
	
}	// ForwardAuthenticator
