/*
 * Copyright 2018 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 * 
 * 20.06.2018 - [JR] - creation
 */
package com.sibvisions.rad.server;

import java.util.Map;

import javax.rad.remote.IConnection;
import javax.rad.remote.IConnectionConstants;
import javax.rad.server.ICloseableSession;
import javax.rad.server.IConfiguration;
import javax.rad.server.ISession;
import javax.rad.server.InjectObject;
import javax.rad.server.SessionContext;
import javax.rad.server.security.IAccessChecker;

import com.sibvisions.rad.server.config.ServerZone;
import com.sibvisions.util.ChangedHashtable;

/**
 * The <code>AbstractSessionContext</code> is a general {@link javax.rad.server.SessionContext} implementation.
 * 
 * @author Ren Jahn
 */
public abstract class AbstractSessionContext extends SessionContext
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /** the master session. */
    private WrappedSession wsessMaster;

    /** the associated session. */
	private WrappedSession wsessCurrent;
	
	/** the access checker. */
	private WrappedAccessChecker wachecker;
	
    /** the name of the object from which the method will be called. */
	private String sObjectName = null;
	
	/** the name of the method which will be called. */
	private String sMethodName = null;

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creates a new instance of <code>SessionContextImpl</code> for a specific
	 * {@link ISession}.
	 * 
	 * @param pSession the associated session for this {@link javax.rad.server.SessionContext}
	 */
	protected AbstractSessionContext(ISession pSession)
	{
		if (pSession instanceof AbstractSession)
		{
			wsessCurrent = new WrappedSession((AbstractSession)pSession);
		}
		else if (pSession instanceof WrappedSession)
		{
			wsessCurrent = (WrappedSession)pSession;
		}
		else if (pSession instanceof DirectServerSession)
		{
			wsessCurrent = new WrappedSession((AbstractSession)((DirectServerSession)pSession).getSession());
		}
		else
		{
			throw new IllegalArgumentException("Given session doesn't offer an 'AbstractSession'!");
		}
		
		setCurrentInstance(this);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Abstract methods implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void destroy()
	{
		setCurrentInstance(null);

		wsessMaster = null;
		wsessCurrent = null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ISession getSession()
	{
		return wsessCurrent;
	}
	
    /**
     * {@inheritDoc}
     */
    @Override
    public ISession getMasterSession()
    {
        if (wsessMaster != null)
        {
            return wsessMaster;
        }
        else
        {
            if (wsessCurrent == null)
            {
                return null;
            }
            
            AbstractSession sessorig = wsessCurrent.session;
                    
            if (sessorig instanceof MasterSession)
            {
                wsessMaster = wsessCurrent;
                
                return wsessMaster;
            }
            else if (sessorig instanceof SubSession)
            {
                wsessMaster = new WrappedSession(((SubSession)sessorig).getMasterSession());
                
                return wsessMaster;
            }
        }
        
        return null;
    }
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public IConfiguration getSessionConfig()
	{
		if (wsessCurrent == null)
		{
			return null;
		}
		
		return wsessCurrent.getConfig();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IConfiguration getServerConfig()
	{
		if (wsessCurrent == null)
		{
			return null;
		}
		
        ServerZone zone = wsessCurrent.session.getApplicationZone().getServerZone();
        
        if (zone != null)
        {
            return zone.getConfig();
        }
        
        return null;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public IConnection getServerConnection()
	{
		if (wsessCurrent == null)
		{
			return null;
		}
		
		return new DirectServerConnection((IDirectServer)wsessCurrent.session.getSessionManager().getServer());
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getObjectName()
	{
		return sObjectName;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getMethodName()
	{
		return sMethodName;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public InjectObject putObject(InjectObject pObject)
	{
		return wsessCurrent.session.putObject(pObject);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public InjectObject removeObject(InjectObject pObject)
	{
		return wsessCurrent.session.removeObject(pObject);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public InjectObject getObject(String pName)
	{
		return wsessCurrent.session.getObject(pName);
	}
    
    /**
     * {@inheritDoc}
     */
    @Override
    public ICloseableSession createSubSession(String pLifeCycleName, Map<String, Object> pProperties) throws Throwable
    {
        ChangedHashtable<String, Object> chtProperties = DirectServerSession.createProperties(pProperties);
        chtProperties.put(IConnectionConstants.LIFECYCLENAME, pLifeCycleName);
        
        Object oSessionId = Server.getInstance().createSubSession(getMasterSession().getId(), chtProperties);
        
        return new WrappedSession(Server.getInstance().getSessionManager().get(oSessionId), true);
    }
    
    /**
     * {@inheritDoc}
     */
    @Override
    public IAccessChecker getAccessChecker()
    {
    	if (wachecker == null)
    	{
    		wachecker = new WrappedAccessChecker(wsessCurrent.session.getAccessController());
    	}
    		
    	return wachecker;
    }
    
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // User-defined methods
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Sets the name of the current object from which the method will be called.
	 * 
	 * @param pObjectName the object name or <code>null</code> if the object is unknown
	 * @see #getObjectName()
	 */
	protected void setObjectName(String pObjectName)
	{
		sObjectName = pObjectName;
	}
	
	/**
	 * Sets the name of the method which will be called.
	 * 
	 * @param pMethodName the method name or <code>null</code> if the method is unknown
	 * @see #getMethodName()
	 */
	protected void setMethodName(String pMethodName)
	{
		sMethodName = pMethodName;
	}
	
}	// SessionContextImpl
