/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 12.05.2009 - [RH] - creation.
 * 23.11.2009 - [RH] - ColumnMetaData && PrimaryKey Column is replaced with MetaData class
 * 02.03.2010 - [RH] - reorganized MetaData -> ServerMetaData, ColumnMetaData -> ServerColumnMetaData
 * 10.03.2010 - [JR] - set NLS_COMP='BINARY'
 * 27.03.2010 - [JR] - #92: default value support    
 * 28.03.2010 - [JR] - #47: getAllowedValues implemented
 * 06.04.2010 - [JR] - #115: getUKs: prepared statement closed  
 * 06.05.2010 - [JR] - open: close statement(s)     
 * 09.10.2010 - [JR] - #114: used CheckConstraintSupport to detect allowed values       
 * 19.11.2010 - [RH] - getUKs, getPKs return Type changed to a <code>Key</code> based result.         
 * 29.11.2010 - [RH] - getUKs Oracle select statement fixed, that it only returns UKs - no PKs. 
 * 01.12.2010 - [RH] - getFKs Oracle select statement fixed, that also returns FKs over UKs.        
 * 14.12.2010 - [RH] - getUKS is solved in DBAccess.
 * 23.12.2010 - [RH] - #227: getFKs returned PK <-> FK columns wrong related, wrong sort fixed!
 * 28.12.2010 - [RH] - #230: quoting of all DB objects like columns, tables, views. 
 * 03.01.2011 - [RH] - schema detecting made better in getColumnMetaData()
 * 06.01.2011 - [JR] - #234: used ColumnMetaDataInfo
 * 24.02.2011 - [RH] - #295: just return the PK, UKs and FKs for the table && schema.
 * 11.03.2011 - [RH] - #308: DB specific automatic quoting implemented          
 * 19.07.2011 - [RH] - #432: OracleDBAccess return list of UKs wrong.   
 * 21.07.2011 - [RH] - #436: OracleDBAccess and PostgresDBAccess should translate JVx quotes in specific insert                      
 * 18.11.2011 - [RH] - #510: All XXDBAccess should provide a SQLException format method 
 * 18.03.2013 - [RH] - #632: DBStorage: Update on Synonym (Oracle) doesn't work - Synonym Support implemented
 * 15.10.2013 - [RH] - #837: DBOracleAccess MetaData determining is very slow in 11g
 * 15.05.2014 - [JR] - #1038: CommonUtil.close used
 * 01.04.2015 - [JR] - TNS Names support via TNS_ADMIN environment property                            
 */
package com.sibvisions.rad.persist.jdbc;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.sql.Blob;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Types;
import java.util.ArrayList;
import java.util.List;

import javax.rad.model.datatype.TimestampDataType;
import javax.rad.persist.DataSourceException;
import javax.rad.type.bean.Bean;
import javax.rad.type.bean.BeanType;
import javax.rad.type.bean.IBean;

import com.sibvisions.rad.persist.jdbc.param.AbstractParam;
import com.sibvisions.util.type.StringUtil;

import oracle.jdbc.driver.OracleConnection;
import oracle.sql.DATE;
import oracle.sql.Datum;
import oracle.sql.TIMESTAMP;
import oracle.sql.TIMESTAMPLTZ;
import oracle.sql.TIMESTAMPTZ;

/**
 * The <code>OracleDBAccess</code> is the implementation for Oracle databases.<br>
 *  
 * @see com.sibvisions.rad.persist.jdbc.DBAccess
 * 
 * @author Roland Hrmann
 */
public class OracleDBAccess extends AbstractOracleDBAccess
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Constants
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** The select statement to get the Synonyms. */
	private static String sSynonymSelect = "select s.table_owner, s.table_name, s.db_link " +
										   "FROM user_synonyms s " +
										   "WHERE s.synonym_name = ?";			 
    
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Constructs a new OracleDBAccess Object.
	 */
	public OracleDBAccess()
	{
		setDriver("oracle.jdbc.OracleDriver");
		
		configureTnsAdmin();
	}
		
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Abstract methods implementation
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    @SuppressWarnings("deprecation")
    @Override
    protected Object convertToArray(AbstractParam pParam) throws SQLException
    {
        Object val = pParam.getValue();
        
        if (pParam.getSqlType() == Types.ARRAY && val != null)
        {
            oracle.sql.ArrayDescriptor arrDescr = oracle.sql.ArrayDescriptor.createDescriptor(pParam.getTypeName(), getConnection()); 
            Object[] data;
            if (val instanceof List)
            {
                data = ((List)val).toArray();
            }
            else if (val instanceof Object[])
            {
                data = ((Object[])val).clone();
            }
            else
            {
                throw new SQLException("Unsupported array value,only List and Object[] are supported!");
            }
            
            if (arrDescr.getBaseType() == Types.STRUCT)
            {
                oracle.sql.StructDescriptor structDescr = oracle.sql.StructDescriptor.createDescriptor(arrDescr.getBaseName(), getConnection());
                ResultSetMetaData rsmd = structDescr.getMetaData();
    
                String[] propertyNames = new String[rsmd.getColumnCount()];
                String[] pojoNames = new String[propertyNames.length];
                for (int j = 0; j < propertyNames.length; j++)
                {
                    propertyNames[j] = rsmd.getColumnName(j + 1);
                    pojoNames[j] = StringUtil.convertToMemberName(propertyNames[j]);
                }
                
                for (int i = 0; i < data.length; i++)
                {
                    Object item = data[i];
                    
                    if (!(item instanceof Object[]))
                    {
                        Object[] itemResult = new Object[propertyNames.length];
                        String[] cols;
                        IBean bean;
                        if (item instanceof IBean)
                        {
                            bean = (IBean)item;
                            cols = propertyNames;
                        }
                        else
                        {
                            bean = new Bean(item);
                            cols = pojoNames;
                        }
                        for (int j = 0; j < cols.length; j++)
                        {
                            itemResult[j] = bean.get(cols[j]);
                        }
                        data[i] = itemResult;
                    }
                }
            }
            
            return ((OracleConnection)getConnection()).createARRAY(pParam.getTypeName(), data);
        }
        else if (pParam.getSqlType() == Types.STRUCT && val != null)
        {
            oracle.sql.StructDescriptor structDescr = oracle.sql.StructDescriptor.createDescriptor(pParam.getTypeName(), getConnection());
            ResultSetMetaData rsmd = structDescr.getMetaData();

            String[] propertyNames = new String[rsmd.getColumnCount()];
            String[] pojoNames = new String[propertyNames.length];
            for (int j = 0; j < propertyNames.length; j++)
            {
                propertyNames[j] = rsmd.getColumnName(j + 1);
                pojoNames[j] = StringUtil.convertToMemberName(propertyNames[j]);
            }
            
            Object[] data;

            if (val instanceof Object[])
            {
            	data = (Object[])val;
            }
            else
            {
            	data = new Object[propertyNames.length];
                String[] cols;
                IBean bean;
                if (val instanceof IBean)
                {
                    bean = (IBean)val;
                    cols = propertyNames;
                }
                else
                {
                    bean = new Bean(val);
                    cols = pojoNames;
                }
                for (int j = 0; j < cols.length; j++)
                {
                	data[j] = bean.get(cols[j]);
                }
            }
            return ((OracleConnection)getConnection()).createStruct(pParam.getTypeName(), data);
        }
        else
        {
            return val;
        }
    }

    /**
     * Converts arrays to {@link List} of {@link IBean}.
     * @param pParam the param to check
     * @return the param or a list in case of array.
     * @throws SQLException the exception
     */
    @SuppressWarnings("deprecation")
    @Override
    protected Object convertArrayToList(Object pParam) throws SQLException
    {
        if (pParam instanceof oracle.sql.ARRAY)
        {
            oracle.sql.ARRAY array = (oracle.sql.ARRAY)pParam;
            
            BeanType beanType = null;
            List<Object> result = new ArrayList<Object>();
            
            Object[] arr = (Object[])array.getArray();
            
            for (int i = 0; i < arr.length; i++)
            {
                Object item = arr[i];
                
                if (item instanceof oracle.sql.STRUCT)
                {
                    oracle.sql.STRUCT struct = (oracle.sql.STRUCT)item;
                    
                    if (beanType == null)
                    {
                        ResultSetMetaData rsmd = struct.getDescriptor().getMetaData();

                        String[] propertyNames = new String[rsmd.getColumnCount()];
                        for (int j = 0; j < propertyNames.length; j++)
                        {
                            propertyNames[j] = rsmd.getColumnName(j + 1);
                        }
                        beanType = new BeanType(propertyNames);
                    }
                    
                    Object[] attributes = struct.getAttributes();
                    Bean bean = new Bean(beanType);
                    for (int j = 0; j < attributes.length; j++)
                    {
                        bean.put(j, attributes[j]);
                    }
                    
                    result.add(bean);
                }
                else
                {
                    result.add(item);
                }
            }
            
            return result;
        }
        else if (pParam instanceof oracle.sql.STRUCT)
        {
            oracle.sql.STRUCT struct = (oracle.sql.STRUCT)pParam;
            
            ResultSetMetaData rsmd = struct.getDescriptor().getMetaData();

            String[] propertyNames = new String[rsmd.getColumnCount()];
            for (int j = 0; j < propertyNames.length; j++)
            {
                propertyNames[j] = rsmd.getColumnName(j + 1);
            }
            BeanType beanType = new BeanType(propertyNames);
            
            Object[] attributes = struct.getAttributes();
            Bean bean = new Bean(beanType);
            for (int j = 0; j < attributes.length; j++)
            {
                bean.put(j, attributes[j]);
            }
            
            return bean;
        }
        else
        {
            return pParam;
        }
    }
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
			
	/**
	 * {@inheritDoc}
	 */
	@SuppressWarnings("deprecation")
	@Override
	protected Object getObjectFromResultSet(ResultSet pResultSet, int pIndex) throws SQLException
	{
		Object result = super.getObjectFromResultSet(pResultSet, pIndex);
		
		if (result instanceof oracle.sql.BFILE)
		{
			result = new BlobFromBFILE((oracle.sql.BFILE)result);
		}
		else if (result instanceof TIMESTAMPLTZ
				|| result instanceof TIMESTAMPTZ
				|| result instanceof TIMESTAMP
				|| result instanceof DATE)
		{
			result = pResultSet.getTimestamp(pIndex);
		}
		
		return result;
	}

    /**
     * {@inheritDoc}
     */
    @Override
    protected String getTableForSynonymIntern(String pSynomyn) throws DataSourceException
    {
        return getTableForSynonymIntern(sSynonymSelect, pSynomyn);
    }
	
	/**
	 * {@inheritDoc} 
	 */
	@Override
	protected Object convertDatabaseSpecificObjectToValue(ServerColumnMetaData pColumnMetaData, Object pValue) throws SQLException
	{
		if (pValue instanceof TIMESTAMPLTZ)
		{
			return ((TIMESTAMPLTZ)pValue).timestampValue(getConnectionIntern());
		}
		else if (pValue instanceof TIMESTAMPTZ)
		{
			return ((TIMESTAMPTZ)pValue).timestampValue(getConnectionIntern());
		}
		else if (pValue instanceof Datum && pColumnMetaData.getTypeIdentifier() == TimestampDataType.TYPE_IDENTIFIER)
		{
			return ((Datum)pValue).timestampValue();
		}
		
		return pValue;
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Sets the correct environment for oracle jdbc driver, to support tns names files.
	 */
	private void configureTnsAdmin()
	{
		try
		{
			if (System.getProperty("oracle.net.tns_admin") == null)
			{
				String tnsAdmin = System.getenv("TNS_ADMIN");
				
				if (tnsAdmin == null)
				{
					String oracleHome = System.getenv("ORACLE_HOME");
					
					if (oracleHome != null)
					{
						tnsAdmin = oracleHome + "/network/admin";
					}
					else
					{
						String fullPath = System.getenv("PATH");
						
						if (fullPath != null)
						{
							String[] paths = fullPath.split(File.pathSeparator);
							
							for (int i = 0; tnsAdmin == null && i < paths.length; i++)
							{
								String path = paths[i].toLowerCase().replace('\\', '/');
								if (path.contains("oracle") && path.endsWith("/bin"))
								{
									path = paths[i].substring(0, path.length() - 4) + "/network/admin";
									
									if (new File(path).exists())
									{
										tnsAdmin = path;
									}
								}
							}
						}
					}
				}
				
				if (tnsAdmin != null)
				{
					System.setProperty("oracle.net.tns_admin", tnsAdmin);
				}
			}
		}
		catch (Exception ex)
		{
			debug("Configure tns admin failed!" + ex);
		}
	}
	
	//****************************************************************
    // Subclass definition
    //****************************************************************
	
	/**
	 * The <code>Blob</code> is the implementation for Oracle <code>BFILE</code>.<br>
	 *  
	 * @author Martin Handsteiner
	 */
	@SuppressWarnings("deprecation")
	public static class BlobFromBFILE implements java.sql.Blob
	{
		/** The bfile. */
		private oracle.sql.BFILE bFile;
		
		/**
		 * Gets a Blob compatible Object for BFile.
		 * @param pBFILE the bFile.
		 */
		public BlobFromBFILE(oracle.sql.BFILE pBFILE)
		{
			bFile = pBFILE;
		}

		/**
		 * {@inheritDoc}
		 */
		public long length() throws SQLException 
		{
			return bFile.length();
		}

		/**
		 * {@inheritDoc}
		 */
		public byte[] getBytes(long pPos, int pLength) throws SQLException 
		{
			try
			{
				bFile.openFile();

				return bFile.getBytes(pPos, pLength);
			}
			finally
			{
				bFile.closeFile();
			}
		}

		/**
		 * {@inheritDoc}
		 */
		public InputStream getBinaryStream() throws SQLException 
		{
			return new ByteArrayInputStream(getBytes(1, (int)length()));
		}

		/**
		 * {@inheritDoc}
		 */
		public InputStream getBinaryStream(long pPos, long pLength) throws SQLException 
		{
			return bFile.getBinaryStream(pPos);
		}
		
		/**
		 * {@inheritDoc}
		 */
		public long position(byte[] pPattern, long pStart) throws SQLException 
		{
			return bFile.position(pPattern, pStart);
		}

		/**
		 * {@inheritDoc}
		 */
		public long position(Blob pPattern, long pStart) throws SQLException 
		{
			// TODO Auto-generated method stub
			return bFile.position(((BlobFromBFILE)pPattern).bFile, pStart);
		}

		/**
		 * {@inheritDoc}
		 */
		public int setBytes(long pPos, byte[] pBytes) throws SQLException 
		{
			throw new UnsupportedOperationException("Changing BFILE is not supported.");
		}

		/**
		 * {@inheritDoc}
		 */
		public int setBytes(long pPos, byte[] pBytes, int offset, int len) throws SQLException 
		{
			throw new UnsupportedOperationException("Changing BFILE is not supported.");
		}

		/**
		 * {@inheritDoc}
		 */
		public OutputStream setBinaryStream(long pos) throws SQLException 
		{
			throw new UnsupportedOperationException("Changing BFILE is not supported.");
		}

		/**
		 * {@inheritDoc}
		 */
		public void truncate(long len) throws SQLException 
		{
			throw new UnsupportedOperationException("Changing BFILE is not supported.");
		}

		/**
		 * {@inheritDoc}
		 */
		public void free() throws SQLException 
		{
			throw new UnsupportedOperationException("Changing BFILE is not supported.");
		}

	}  // BlobFromBFILE
	
} 	// OracleDBAccess
