/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 17.11.2008 - [HM] - creation
 * 22.04.2011 - [JR] - getMaximizedConstraints implemented
 * 24.10.2012 - [JR] - #604: added constructor
 */
package javax.rad.genui.layout;

import javax.rad.genui.UIFactoryManager;
import javax.rad.genui.UILayout;
import javax.rad.ui.IComponent;
import javax.rad.ui.layout.IFormLayout;

import com.sibvisions.util.ArrayUtil;

/**
 * Platform and technology independent form oriented layout.
 * It is designed for use with AWT, Swing, SWT, JSP, JSF,... .
 * 
 * @author Martin Handsteiner
 */
public class UIFormLayout extends UILayout<IFormLayout, IFormLayout.IConstraints> 
						  implements IFormLayout
{

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	  
    /**
     * Creates a new instance of <code>UIFormLayout</code>.
     *
     * @see IFormLayout
     */
	public UIFormLayout()
	{
		super(UIFactoryManager.getFactory().createFormLayout());
	}

    /**
     * Creates a new instance of <code>UIFormLayout</code> with the given
     * layout.
     *
     * @param pLayout the layout
     * @see IFormLayout
     */
	public UIFormLayout(IFormLayout pLayout)
	{
		super(pLayout);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * {@inheritDoc}
     */
	public int getHorizontalAlignment()
	{
		return uiResource.getHorizontalAlignment();
	}
	
    /**
     * {@inheritDoc}
     */
	public void setHorizontalAlignment(int pHorizontalAlignment)
	{
		uiResource.setHorizontalAlignment(pHorizontalAlignment);
	}
	
    /**
     * {@inheritDoc}
     */
	public int getVerticalAlignment()
	{
		return uiResource.getVerticalAlignment();
	}

    /**
     * {@inheritDoc}
     */
    public void setVerticalAlignment(int pVerticalAlignment)
    {
    	uiResource.setVerticalAlignment(pVerticalAlignment);
    }
    
	/**
	 * {@inheritDoc}
	 */
    public int getNewlineCount()
    {
    	return uiResource.getNewlineCount();
    }
    
	/**
	 * {@inheritDoc}
	 */
    public void setNewlineCount(int pNewlineCount)
    {
    	uiResource.setNewlineCount(pNewlineCount);
    }

	/**
	 * {@inheritDoc}
	 */
	public IAnchor getLeftAnchor()
    {
		return uiResource.getLeftAnchor();
    }
	
	/**
	 * {@inheritDoc}
	 */
	public IAnchor getRightAnchor()
    {
		return uiResource.getRightAnchor();
    }
	
	/**
	 * {@inheritDoc}
	 */
	public IAnchor getTopAnchor()
    {
		return uiResource.getTopAnchor();
    }
	
	/**
	 * {@inheritDoc}
	 */
	public IAnchor getBottomAnchor()
    {
		return uiResource.getBottomAnchor();
    }
	
	/**
	 * {@inheritDoc}
	 */
	public IAnchor getLeftMarginAnchor()
    {
		return uiResource.getLeftMarginAnchor();
    }
	
	/**
	 * {@inheritDoc}
	 */
	public IAnchor getRightMarginAnchor()
    {
		return uiResource.getRightMarginAnchor();
    }
	
	/**
	 * {@inheritDoc}
	 */
	public IAnchor getTopMarginAnchor()
    {
		return uiResource.getTopMarginAnchor();
    }
	
	/**
	 * {@inheritDoc}
	 */
	public IAnchor getBottomMarginAnchor()
    {
		return uiResource.getBottomMarginAnchor();
    }
	
	/**
	 * {@inheritDoc}
	 */
	public IAnchor[] getHorizontalAnchors()
	{
    	return uiResource.getHorizontalAnchors();
	}
	
	/**
	 * {@inheritDoc}
	 */
	public IAnchor[] getVerticalAnchors()
	{
		return uiResource.getVerticalAnchors();
	}

	/**
	 * {@inheritDoc}
	 */
	public IAnchor createAnchor(IAnchor pRelatedAnchor)
    {
    	return uiResource.createAnchor(pRelatedAnchor);
    }

	/**
	 * {@inheritDoc}
	 */
	public IAnchor createAnchor(IAnchor pRelatedAnchor, int pPosition)
    {
		return uiResource.createAnchor(pRelatedAnchor, pPosition);
    }
	
	/**
	 * {@inheritDoc}
	 */
    public IConstraints getConstraints(int pColumn, int pRow)
    {
    	return uiResource.getConstraints(pColumn, pRow);
    }
    
	/**
	 * {@inheritDoc}
	 */
    public IConstraints getConstraints(int pBeginColumn, int pBeginRow, int pEndColumn, int pEndRow)
    {
    	return uiResource.getConstraints(pBeginColumn, pBeginRow, pEndColumn, pEndRow);
    }

    /**
	 * {@inheritDoc}
	 */
    public IConstraints getConstraints(IAnchor pTopAnchor, IAnchor pLeftAnchor, IAnchor pBottomAnchor, IAnchor pRightAnchor)
    {
    	return uiResource.getConstraints(pTopAnchor, pLeftAnchor, pBottomAnchor, pRightAnchor);
    }

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
	/**
	 * Gets the left anchor of the given column.
	 * 
	 * @param pColumnIndex the column index.
	 * @return the left IAnchor 
	 */
	public IAnchor getColumnLeftAnchor(int pColumnIndex)
	{
		return getConstraints(pColumnIndex, 0).getLeftAnchor();
	}
	
	/**
	 * Gets the right anchor of the given column.
	 * 
	 * @param pColumnIndex the column index.
	 * @return the right IAnchor 
	 */
	public IAnchor getColumnRightAnchor(int pColumnIndex)
	{
		return getConstraints(pColumnIndex, 0).getRightAnchor();
	}
	
	/**
	 * Gets the top anchor of the given row.
	 * 
	 * @param pRowIndex the row index.
	 * @return the top IAnchor 
	 */
	public IAnchor getRowTopAnchor(int pRowIndex)
	{
		return getConstraints(0, pRowIndex).getTopAnchor();
	}
	
	/**
	 * Gets the bottom anchor of the given row.
	 * 
	 * @param pRowIndex the row index.
	 * @return the bottom IAnchor 
	 */
	public IAnchor getRowBottomAnchor(int pRowIndex)
	{
		return getConstraints(0, pRowIndex).getBottomAnchor();
	}

	/**
	 * Gets the anchor configuration for default anchors.
	 * 
	 * @return the anchor configuration for default anchors.
	 */
	public String getAnchorConfiguration()
	{
		StringBuilder anchorConfiguration = new StringBuilder();
		
		IAnchor[] anchors = getHorizontalAnchors();
		
		int i = 0;
		IAnchor anchor = getColumnRightAnchor(i);
		IAnchor nextAnchor = getColumnRightAnchor(i + 1);
		while (ArrayUtil.indexOfReference(anchors, anchor) >= 0
				|| ArrayUtil.indexOfReference(anchors, nextAnchor) >= 0
				|| ArrayUtil.indexOfReference(anchors, nextAnchor.getRelatedAnchor()) >= 0)
		{
			if (i > 0 && getColumnLeftAnchor(i).getPosition() != getHorizontalGap())
			{
				anchorConfiguration.append('l');
				anchorConfiguration.append(i);
				anchorConfiguration.append('=');
				anchorConfiguration.append(getColumnLeftAnchor(i).getPosition());
				anchorConfiguration.append(',');
			}
			if (!anchor.isAutoSize())
			{
				anchorConfiguration.append('r');
				anchorConfiguration.append(i);
				anchorConfiguration.append('=');
				anchorConfiguration.append(anchor.getPosition());
				anchorConfiguration.append(',');
			}
			
			i++;
			anchor = nextAnchor;
			nextAnchor = getColumnRightAnchor(i + 1);
		}
		if (i > 0 && getColumnLeftAnchor(i).getPosition() != getHorizontalGap())
		{
			anchorConfiguration.append('l');
			anchorConfiguration.append(i);
			anchorConfiguration.append('=');
			anchorConfiguration.append(getColumnLeftAnchor(i).getPosition());
			anchorConfiguration.append(',');
		}
		i = -1;
		anchor = getColumnLeftAnchor(i);
		nextAnchor = getColumnLeftAnchor(i - 1);
		while (ArrayUtil.indexOfReference(anchors, anchor) >= 0
				|| ArrayUtil.indexOfReference(anchors, nextAnchor) >= 0
				|| ArrayUtil.indexOfReference(anchors, nextAnchor.getRelatedAnchor()) >= 0)
		{
			if (i < -1 && getColumnRightAnchor(i).getPosition() != -getHorizontalGap())
			{
				anchorConfiguration.append('r');
				anchorConfiguration.append(i);
				anchorConfiguration.append('=');
				anchorConfiguration.append(getColumnRightAnchor(i).getPosition());
				anchorConfiguration.append(',');
			}
			if (!anchor.isAutoSize())
			{
				anchorConfiguration.append('l');
				anchorConfiguration.append(i);
				anchorConfiguration.append('=');
				anchorConfiguration.append(anchor.getPosition());
				anchorConfiguration.append(',');
			}
			
			i--;
			anchor = nextAnchor;
			nextAnchor = getColumnLeftAnchor(i - 1);
		}
		if (i < -1 && getColumnRightAnchor(i).getPosition() != -getHorizontalGap())
		{
			anchorConfiguration.append('r');
			anchorConfiguration.append(i);
			anchorConfiguration.append('=');
			anchorConfiguration.append(getColumnRightAnchor(i).getPosition());
			anchorConfiguration.append(',');
		}
		
		anchors = getVerticalAnchors();
		
		i = 0;
		anchor = getRowBottomAnchor(i);
		nextAnchor = getRowBottomAnchor(i + 1);
		while (ArrayUtil.indexOfReference(anchors, anchor) >= 0
				|| ArrayUtil.indexOfReference(anchors, nextAnchor) >= 0
				|| ArrayUtil.indexOfReference(anchors, nextAnchor.getRelatedAnchor()) >= 0)
		{
			if (i > 0 && getRowTopAnchor(i).getPosition() != getVerticalGap())
			{
				anchorConfiguration.append('t');
				anchorConfiguration.append(i);
				anchorConfiguration.append('=');
				anchorConfiguration.append(getRowTopAnchor(i).getPosition());
				anchorConfiguration.append(',');
			}
			if (!anchor.isAutoSize())
			{
				anchorConfiguration.append('b');
				anchorConfiguration.append(i);
				anchorConfiguration.append('=');
				anchorConfiguration.append(anchor.getPosition());
				anchorConfiguration.append(',');
			}
			
			i++;
			anchor = nextAnchor;
			nextAnchor = getRowBottomAnchor(i + 1);
		}
		if (i > 0 && getRowTopAnchor(i).getPosition() != getVerticalGap())
		{
			anchorConfiguration.append('t');
			anchorConfiguration.append(i);
			anchorConfiguration.append('=');
			anchorConfiguration.append(getRowTopAnchor(i).getPosition());
			anchorConfiguration.append(',');
		}
		i = -1;
		anchor = getRowTopAnchor(i);
		nextAnchor = getRowTopAnchor(i - 1);
		while (ArrayUtil.indexOfReference(anchors, anchor) >= 0
				|| ArrayUtil.indexOfReference(anchors, nextAnchor) >= 0
				|| ArrayUtil.indexOfReference(anchors, nextAnchor.getRelatedAnchor()) >= 0)
		{
			if (i < -1 && getRowBottomAnchor(i).getPosition() != -getVerticalGap())
			{
				anchorConfiguration.append('b');
				anchorConfiguration.append(i);
				anchorConfiguration.append('=');
				anchorConfiguration.append(getRowBottomAnchor(i).getPosition());
				anchorConfiguration.append(',');
			}
			if (!anchor.isAutoSize())
			{
				anchorConfiguration.append('t');
				anchorConfiguration.append(i);
				anchorConfiguration.append('=');
				anchorConfiguration.append(anchor.getPosition());
				anchorConfiguration.append(',');
			}
			
			i--;
			anchor = nextAnchor;
			nextAnchor = getRowTopAnchor(i - 1);
		}
		if (i < -1 && getRowBottomAnchor(i).getPosition() != -getVerticalGap())
		{
			anchorConfiguration.append('b');
			anchorConfiguration.append(i);
			anchorConfiguration.append('=');
			anchorConfiguration.append(getRowBottomAnchor(i).getPosition());
			anchorConfiguration.append(',');
		}
		
		if (anchorConfiguration.length() > 0)
		{
			anchorConfiguration.setLength(anchorConfiguration.length() - 1);
			return anchorConfiguration.toString();
		}
		else
		{
			return null;
		}
	}
	
	/**
	 * Sets the anchor configuration for default anchors.
	 * 
	 * @param pAnchorConfiguration the anchor configuration for default anchors.
	 */
	public void setAnchorConfiguration(String pAnchorConfiguration)
	{
		if (pAnchorConfiguration != null)
		{
			String[] anchorConfigurations = pAnchorConfiguration.split(",");
			
			for (int i = 0; i < anchorConfigurations.length; i++)
			{
				String anchorConfig = anchorConfigurations[i].trim();
				
				int eqIndex = anchorConfig.indexOf("=");
				
				if (eqIndex > 0)
				{
					try
					{
						char anchorType = anchorConfig.charAt(0);
						int  colOrRow   = Integer.parseInt(anchorConfig.substring(1, eqIndex).trim());
						int  position   = Integer.parseInt(anchorConfig.substring(eqIndex + 1).trim());
						
						IAnchor anchor;
						switch (anchorType)
						{
							case 'l': anchor = getColumnLeftAnchor(colOrRow); break;
							case 'r': anchor = getColumnRightAnchor(colOrRow); break;
							case 't': anchor = getRowTopAnchor(colOrRow); break;
							case 'b': anchor = getRowBottomAnchor(colOrRow); break;
							default: anchor = null; break;
						}
						if (anchor != null)
						{
							anchor.setAutoSize(false);
							anchor.setPosition(position);
						}
					}
					catch (Exception ex)
					{
						// Ignore wrong configured anchors...
					}
				}
			}
		}
	}
	
	/**
	 * Creates vertical centered constraints for the given column and row.
	 * 
	 * @param pColumn the column.
	 * @param pRow the row.
	 * @return the constraints for the given column and row.
	 */
    public IConstraints getVCenterConstraints(int pColumn, int pRow)
    {
    	return getVCenterConstraints(pColumn, pRow, pColumn, pRow);
    }
    
	/**
	 * Creates vertical centered constraints for the given column and row.
	 * 
	 * @param pBeginColumn the column.
	 * @param pBeginRow the row.
	 * @param pEndColumn the column count.
	 * @param pEndRow the row count.
	 * @return the constraints for the given column and row.
	 */
    public IConstraints getVCenterConstraints(int pBeginColumn, int pBeginRow, int pEndColumn, int pEndRow)
    {
    	IConstraints result = uiResource.getConstraints(pBeginColumn, pBeginRow, pEndColumn, pEndRow);
    	result.setTopAnchor(createAnchor(result.getTopAnchor()));
    	result.setBottomAnchor(createAnchor(result.getBottomAnchor()));
    	
    	return result;
    }
    
	/**
	 * Creates vertical centered constraints for the given column and row.
	 * 
	 * @param pColumn the column.
	 * @param pRow the row.
	 * @return the constraints for the given column and row.
	 */
    public IConstraints getHCenterConstraints(int pColumn, int pRow)
    {
    	return getHCenterConstraints(pColumn, pRow, pColumn, pRow);
    }
    
	/**
	 * Creates vertical centered constraints for the given column and row.
	 * 
	 * @param pBeginColumn the column.
	 * @param pBeginRow the row.
	 * @param pEndColumn the column count.
	 * @param pEndRow the row count.
	 * @return the constraints for the given column and row.
	 */
    public IConstraints getHCenterConstraints(int pBeginColumn, int pBeginRow, int pEndColumn, int pEndRow)
    {
    	IConstraints result = uiResource.getConstraints(pBeginColumn, pBeginRow, pEndColumn, pEndRow);
    	result.setLeftAnchor(createAnchor(result.getLeftAnchor()));
    	result.setRightAnchor(createAnchor(result.getRightAnchor()));
    	
    	return result;
    }
    
	/**
	 * Creates horizontal and vertical centered constraints for the given column and row.
	 * 
	 * @param pColumn the column.
	 * @param pRow the row.
	 * @return the constraints for the given column and row.
	 */
    public IConstraints getCenterConstraints(int pColumn, int pRow)
    {
    	return getHCenterConstraints(pColumn, pRow, pColumn, pRow);
    }
    
	/**
	 * Creates horizontal and vertical centered constraints for the given column and row.
	 * 
	 * @param pBeginColumn the column.
	 * @param pBeginRow the row.
	 * @param pEndColumn the column count.
	 * @param pEndRow the row count.
	 * @return the constraints for the given column and row.
	 */
    public IConstraints getCenterConstraints(int pBeginColumn, int pBeginRow, int pEndColumn, int pEndRow)
    {
    	IConstraints result = uiResource.getConstraints(pBeginColumn, pBeginRow, pEndColumn, pEndRow);
    	result.setLeftAnchor(createAnchor(result.getLeftAnchor()));
    	result.setRightAnchor(createAnchor(result.getRightAnchor()));
    	result.setTopAnchor(createAnchor(result.getTopAnchor()));
    	result.setBottomAnchor(createAnchor(result.getBottomAnchor()));
    	
    	return result;
    }
    
	/**
	 * Creates the constraints left of the center constraint.
	 * 
	 * @param pCenterComponent the center component.
	 * @return the constraints left of the center constraint.
	 */
    public IConstraints getLeftSpaceConstraints(IComponent pCenterComponent)
    {
    	IConstraints centerConstraints = getConstraints(pCenterComponent);
    	
		return getConstraints(centerConstraints.getTopAnchor(), 
							  createAnchor(centerConstraints.getLeftAnchor().getRelatedAnchor(), 0), 
							  centerConstraints.getBottomAnchor(), 
							  centerConstraints.getLeftAnchor());
    }
    
	/**
	 * Creates the constraints right of the center constraint.
	 * 
	 * @param pCenterComponent the center component.
	 * @return the constraints right of the center constraint.
	 */
    public IConstraints getRightSpaceConstraints(IComponent pCenterComponent)
    {
    	IConstraints centerConstraints = getConstraints(pCenterComponent);
    	
		return getConstraints(centerConstraints.getTopAnchor(), 
							  centerConstraints.getRightAnchor(), 
							  centerConstraints.getBottomAnchor(), 
							  createAnchor(centerConstraints.getRightAnchor().getRelatedAnchor(), 0));
    }
    
	/**
	 * Creates the constraints top of the center constraint.
	 * 
	 * @param pCenterComponent the center component.
	 * @return the constraints top of the center constraint.
	 */
    public IConstraints getTopSpaceConstraints(IComponent pCenterComponent)
    {
    	IConstraints centerConstraints = getConstraints(pCenterComponent);
    	
		return getConstraints(createAnchor(centerConstraints.getTopAnchor().getRelatedAnchor(), 0),
							  centerConstraints.getLeftAnchor(), 
							  centerConstraints.getTopAnchor(), 
							  centerConstraints.getRightAnchor());
    }
    
	/**
	 * Creates the constraints bottom of the center constraint.
	 * 
	 * @param pCenterComponent the center component.
	 * @return the constraints bottom of the center constraint.
	 */
    public IConstraints getBottomSpaceConstraints(IComponent pCenterComponent)
    {
    	IConstraints centerConstraints = getConstraints(pCenterComponent);
    	
		return getConstraints(centerConstraints.getBottomAnchor(), 
							  centerConstraints.getLeftAnchor(), 
							  createAnchor(centerConstraints.getBottomAnchor().getRelatedAnchor(), 0),
							  centerConstraints.getRightAnchor());
    }
    
	/**
	 * Creates top aligned constraints for the given column and row.
	 * 
	 * @param pColumn the column.
	 * @param pRow the row.
	 * @return the constraints for the given column and row.
	 */
    public IConstraints getTopAlignedConstraints(int pColumn, int pRow)
    {
    	return getTopAlignedConstraints(pColumn, pRow, pColumn, pRow);
    }
    
	/**
	 * Creates top aligned constraints for the given column and row.
	 * 
	 * @param pBeginColumn the column.
	 * @param pBeginRow the row.
	 * @param pEndColumn the column.
	 * @param pEndRow the row.
	 * @return the constraints for the given column and row.
	 */
    public IConstraints getTopAlignedConstraints(int pBeginColumn, int pBeginRow, int pEndColumn, int pEndRow)
    {
    	IConstraints result = uiResource.getConstraints(pBeginColumn, pBeginRow, pEndColumn, pEndRow);
    	result.setBottomAnchor(null);
    	
    	return result;
    }
    
	/**
	 * Creates bottom aligned constraints for the given column and row.
	 * 
	 * @param pColumn the column.
	 * @param pRow the row.
	 * @return the constraints for the given column and row.
	 */
    public IConstraints getBottomAlignedConstraints(int pColumn, int pRow)
    {
    	return getBottomAlignedConstraints(pColumn, pRow, pColumn, pRow);
    }
    
	/**
	 * Creates bottom aligned constraints for the given column and row.
	 * 
	 * @param pBeginColumn the column.
	 * @param pBeginRow the row.
	 * @param pEndColumn the column.
	 * @param pEndRow the row.
	 * @return the constraints for the given column and row.
	 */
    public IConstraints getBottomAlignedConstraints(int pBeginColumn, int pBeginRow, int pEndColumn, int pEndRow)
    {
    	IConstraints result = uiResource.getConstraints(pBeginColumn, pBeginRow, pEndColumn, pEndRow);
    	result.setTopAnchor(null);
    	
    	return result;
    }
    
	/**
	 * Creates left aligned constraints for the given column and row.
	 * 
	 * @param pColumn the column.
	 * @param pRow the row.
	 * @return the constraints for the given column and row.
	 */
    public IConstraints getLeftAlignedConstraints(int pColumn, int pRow)
    {
    	return getLeftAlignedConstraints(pColumn, pRow, pColumn, pRow);
    }
    
	/**
	 * Creates left aligned constraints for the given column and row.
	 * 
	 * @param pBeginColumn the column.
	 * @param pBeginRow the row.
	 * @param pEndColumn the column.
	 * @param pEndRow the row.
	 * @return the constraints for the given column and row.
	 */
    public IConstraints getLeftAlignedConstraints(int pBeginColumn, int pBeginRow, int pEndColumn, int pEndRow)
    {
    	IConstraints result = uiResource.getConstraints(pBeginColumn, pBeginRow, pEndColumn, pEndRow);
    	result.setRightAnchor(null);
    	
    	return result;
    }
    
	/**
	 * Creates right aligned constraints for the given column and row.
	 * 
	 * @param pColumn the column.
	 * @param pRow the row.
	 * @return the constraints for the given column and row.
	 */
    public IConstraints getRightAlignedConstraints(int pColumn, int pRow)
    {
    	return getRightAlignedConstraints(pColumn, pRow, pColumn, pRow);
    }
    
	/**
	 * Creates right aligned constraints for the given column and row.
	 * 
	 * @param pBeginColumn the column.
	 * @param pBeginRow the row.
	 * @param pEndColumn the column.
	 * @param pEndRow the row.
	 * @return the constraints for the given column and row.
	 */
    public IConstraints getRightAlignedConstraints(int pBeginColumn, int pBeginRow, int pEndColumn, int pEndRow)
    {
    	IConstraints result = uiResource.getConstraints(pBeginColumn, pBeginRow, pEndColumn, pEndRow);
    	result.setLeftAnchor(null);
    	
    	return result;
    }
    
    /**
     * Creates maximized constraints for top, left, bottom and right anchor.
     *  
     * @return the maximized constraints.
     */
    public IConstraints getMaximizedConstraints()
    {
    	return uiResource.getConstraints(getTopAnchor(), getLeftAnchor(), getBottomAnchor(), getRightAnchor());
    }
    
	//****************************************************************
	// Subinterface definition
	//****************************************************************
/*
    *//**
	 * The <code>UIAnchor</code> gives the possible horizontal and vertical positions.
	 * 
	 * @author Martin Handsteiner
	 *//*
	public static class UIAnchor extends UIResource<IAnchor> 
	                      		 implements IAnchor
	{
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Initialization
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		*//**
		 * Constructs an anchor relative to pRelatedAnchor with pPosition pixels.
		 * 
		 * @param pRelatedAnchor the related anchor for this anchor.
		 *//*
		public UIAnchor(IAnchor pRelatedAnchor)
		{
			super(pRelatedAnchor.getLayout().createAnchor(pRelatedAnchor));
		}

		*//**
		 * Constructs an anchor relative to pRelatedAnchor with pPosition pixels.
		 * 
		 * @param pRelatedAnchor the related anchor for this anchor.
		 * @param pPosition the position relative to the related anchor.
		 *//*
		public UIAnchor(IAnchor pRelatedAnchor, int pPosition)
		{
			super(pRelatedAnchor.getLayout().createAnchor(pRelatedAnchor, pPosition));
		}
		
		*//**
		 * Constructs an centered anchor between the related and second related anchor.
		 * @param pRelatedAnchor the related anchor for this anchor.
		 * @param pSecondRelatedAnchor the second related anchor for this anchor.
		 * @param pRelativePosition the relative position between first and second anchor.
		 *//*
		public UIAnchor(IAnchor pRelatedAnchor, IAnchor pSecondRelatedAnchor, float pRelativePosition)
		{
			super(pRelatedAnchor.getLayout().createAnchor(pRelatedAnchor, pSecondRelatedAnchor, pRelativePosition));
		}

		*//**
		 * Constructs an centered anchor between the related and second related anchor.
		 * @param pRelatedAnchor the related anchor for this anchor.
		 * @param pSecondRelatedAnchor the second related anchor for this anchor.
		 *//*
		public UIAnchor(IAnchor pRelatedAnchor, IAnchor pSecondRelatedAnchor)
		{
			super(pRelatedAnchor.getLayout().createAnchor(pRelatedAnchor, pSecondRelatedAnchor));
		}

		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Interface implementation
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		*//**
		 * {@inheritDoc}
		 *//*
		public IFormLayout getLayout()
	    {
	    	return uiResource.getLayout();
	    }

		*//**
		 * {@inheritDoc}
		 *//*
		public int getOrientation()
	    {
	    	return uiResource.getOrientation();
	    }

		*//**
		 * {@inheritDoc}
		 *//*
		public IAnchor getRelatedAnchor()
	    {
	    	return uiResource.getRelatedAnchor();
	    }

		*//**
		 * {@inheritDoc}
		 *//*
		public boolean isAutoSize()
	    {
	    	return uiResource.isAutoSize();
	    }

		*//**
		 * {@inheritDoc}
		 *//*
		public void setAutoSize(boolean pAutoSize)
	    {
	    	uiResource.setAutoSize(pAutoSize);
	    }

		*//**
		 * {@inheritDoc}
		 *//*
		public int getPosition()
	    {
	    	return uiResource.getPosition();
	    }

		*//**
		 * {@inheritDoc}
		 *//*
		public void setPosition(int pPosition)
	    {
	    	uiResource.setPosition(pPosition);
	    }
		
		*//**
		 * {@inheritDoc}
		 *//*
		public IAnchor getSecondRelatedAnchor()
		{
	    	return uiResource.getSecondRelatedAnchor();
		}

		*//**
		 * {@inheritDoc}
		 *//*
		public float getRelativePosition()
		{
	    	return uiResource.getRelativePosition();
		}

		*//**
		 * {@inheritDoc}
		 *//*
		public void setRelativePosition(float pRelativePosition)
		{
	    	uiResource.setRelativePosition(pRelativePosition);
		}
		
	}	// UIAnchor
	
	*//**
	 * The <code>UIConstraint</code> stores the top, left, bottom and right Anchor for layouting a component.
	 * 
	 * @author Martin Handsteiner
	 *//*
	public static class UIConstraint extends UIResource<IConstraints> 
	                                 implements IConstraints
	{
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Initialization
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		
		*//**
		 * Creates the default constraints for the given anchors.
		 * 
		 * @param pTopAnchor the top anchor.
		 * @param pLeftAnchor the left anchor.
		 * @param pBottomAnchor the bottom anchor.
		 * @param pRightAnchor the right anchor.
		 *//*
	    public UIConstraint(IAnchor pTopAnchor, IAnchor pLeftAnchor, IAnchor pBottomAnchor, IAnchor pRightAnchor)
	    {
	    	super(createConstraints(pTopAnchor, pLeftAnchor, pBottomAnchor, pRightAnchor));
	    }
	    
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// Interface implementation
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		*//**
		 * {@inheritDoc}
		 *//*
		public IAnchor getLeftAnchor()
	    {
	    	return uiResource.getLeftAnchor();
	    }

		*//**
		 * {@inheritDoc}
		 *//*
		public IAnchor getRightAnchor()
	    {
	    	return uiResource.getRightAnchor();
	    }

		*//**
		 * {@inheritDoc}
		 *//*
		public IAnchor getTopAnchor()
	    {
	    	return uiResource.getTopAnchor();
	    }

		*//**
		 * {@inheritDoc}
		 *//*
		public IAnchor getBottomAnchor()
	    {
	    	return uiResource.getBottomAnchor();
	    }

		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		// User-defined methods
		//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	    
	    *//**
	     * creates a new Constraint with the IFormLayout that is related to the Anchors.
	     * @param pTopAnchor the Top Anchor.
	     * @param pLeftAnchor the Left Anchor.
	     * @param pBottomAnchor the Bottom Anchor.
	     * @param pRightAnchor the Right Anchor.
	     * @return the IConstraint.
	     *//*
	    private static IConstraints createConstraints(IAnchor pTopAnchor, IAnchor pLeftAnchor, IAnchor pBottomAnchor, IAnchor pRightAnchor)
	    {
	    	IFormLayout layout = null;
	    	if (pTopAnchor != null)
	    	{
	    		layout = pTopAnchor.getLayout();
	    	}
	    	else if (pLeftAnchor != null)
	    	{
	    		layout = pLeftAnchor.getLayout();
	    	}
	    	else if (pBottomAnchor != null)
	    	{
	    		layout = pBottomAnchor.getLayout();
	    	}
	    	else
	    	{
	    		layout = pRightAnchor.getLayout();
	    	}
	    	return layout.getConstraints(pTopAnchor, pLeftAnchor, pBottomAnchor, pRightAnchor);
	    }
		
	}	// UIConstraint
*/    
}	// UIFormLayout
