/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 16.11.2008 - [HM] - creation
 * 08.12.2008 - [JR] - default button functionality implemented
 * 10.12.2008 - [JR] - moved accelerator methods from UIMenuItem
 * 24.10.2012 - [JR] - #604: added constructor
 * 22.09.2014 - [RZ] - #1103: the name is now created from the listeners or action command.
 */
package javax.rad.genui.component;

import java.util.regex.Pattern;

import javax.rad.genui.UIFactoryManager;
import javax.rad.ui.IImage;
import javax.rad.ui.component.IButton;
import javax.rad.ui.event.IActionListener;

import com.sibvisions.util.type.StringUtil;

/**
 * Platform and technology independent button.
 * It is designed for use with AWT, Swing, SWT, JSP, JSF, ... .
 * 
 * @author Martin Handsteiner
 * 
 */
public class UIButton extends AbstractUIButton<IButton> 
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * The pattern used for testing if an action command could be a fully
	 * qualified classname.
	 * <p/>
	 * <code><pre>
	 * ^                At the start of the string
	 * [a-zA-Z]+        One or more letters
	 * (\\.[a-zA-Z]+)+  Followed by at least one group of a leading dot
	 *                  and one or more letters.
	 * $                Until the string ends.
	 * </pre></code>
	 */
	private static final Pattern FULLY_QUALIFIED_CLASSNAME_PATTERN = Pattern.compile("^[a-zA-Z]+(\\.[a-zA-Z]+)+$");
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	  
    /**
     * Creates a new instance of <code>UIButton</code>.
     *
     * @see IButton
     */
	public UIButton()
	{
		super(UIFactoryManager.getFactory().createButton());
	}

    /**
     * Creates a new instance of <code>UIButton</code> with the given
     * button.
     *
     * @param pButton the button
     * @see IButton
     */
	protected UIButton(IButton pButton)
	{
		super(pButton);
	}	
	
    /**
     * Creates a new instance of <code>UIButton</code>.
     *
     * @param pText the label of the button.
     * @see IButton
     */
	public UIButton(String pText)
	{
		this();
		setText(pText);
	}

	/**
	 * Creates a new instance of {@link UIButton}.
	 *
	 * @param pText the {@link String text}.
	 * @param pImage the {@link IImage image}.
	 * @see #setImage(IImage)
	 * @see #setText(String)
	 */
	public UIButton(String pText, IImage pImage)
	{
		this();
		
		setText(pText);
		setImage(pImage);
	}
	
	/**
	 * Creates a new instance of {@link UIButton}.
	 *
	 * @param pText the {@link String text}.
	 * @param pActionListener the {@link IActionListener action listener}.
	 * @see #eventAction()
	 * @see #setText(String)
	 */
	public UIButton(String pText, IActionListener pActionListener)
	{
		this();
		
		setText(pText);
		eventAction().addListener(pActionListener);
	}
	
	/**
	 * Creates a new instance of {@link UIButton}.
	 *
	 * @param pText the {@link String text}.
	 * @param pListener the {@link Object listener}.
	 * @param pMethodName the {@link String method name}.
	 * @see #eventAction()
	 * @see #setText(String)
	 */
	public UIButton(String pText, Object pListener, String pMethodName)
	{
		this();
		
		setText(pText);
		eventAction().addListener(pListener, pMethodName);
	}
	
	/**
	 * Creates a new instance of {@link UIButton}.
	 *
	 * @param pText the {@link String text}.
	 * @param pImage the {@link IImage image}.
	 * @param pActionListener the {@link IActionListener action listener}.
	 * @see #eventAction()
	 * @see #setImage(IImage)
	 * @see #setText(String)
	 */
	public UIButton(String pText, IImage pImage, IActionListener pActionListener)
	{
		this();
		
		setText(pText);
		setImage(pImage);
		eventAction().addListener(pActionListener);
	}
	
	/**
	 * Creates a new instance of {@link UIButton}.
	 *
	 * @param pText the {@link String text}.
	 * @param pImage the {@link IImage image}.
	 * @param pListener the {@link Object listener}.
	 * @param pMethodName the {@link String method name}.
	 * @see #eventAction()
	 * @see #setImage(IImage)
	 * @see #setText(String)
	 */
	public UIButton(String pText, IImage pImage, Object pListener, String pMethodName)
	{
		this();
		
		setText(pText);
		setImage(pImage);
		eventAction().addListener(pListener, pMethodName);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String createComponentName()
	{
		String name = getNameFromActionListeners();
		
		String actionCommand = getActionCommand();
		
		if (!StringUtil.isEmpty(actionCommand))
		{
			if (FULLY_QUALIFIED_CLASSNAME_PATTERN.matcher(actionCommand).matches())
			{
				actionCommand = StringUtil.getShortenedWords(actionCommand, 3);
				actionCommand = actionCommand.replace('.', '-');
			}
			
			if (!StringUtil.isEmpty(name))
			{
				name = name + "_" + actionCommand;
			}
			else
			{
				name = actionCommand;
			}
		}
		
		if (!StringUtil.isEmpty(name))
		{
			String rootPrefix = "";
			
			if (getRootName() != null)
			{
				rootPrefix = getRootName() + "_";
			}
			
			return incrementNameIfExists(rootPrefix + "B_" + name.toUpperCase(), getExistingNames(), false);
		}
		else
		{
			return super.createComponentName();
		}
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Gets the name of the method from the registered action listeners. Returns
	 * {@code null} if there is no suitable listener registered.
	 * 
	 * @return the name of method. {@code null} if there is no suitable listener registered.
	 */
	private String getNameFromActionListeners()
	{
		for (IActionListener listener : eventAction().getListeners())
		{
			String listenerString = listener.toString();
			if (listenerString != null && !listenerString.contains("@"))
			{
				int arrowIndex = listenerString.indexOf(" -> ");
				
				if (arrowIndex >= 0)
				{
					listenerString = listenerString.substring(arrowIndex + 4);
				}
				
				// We are now working with something looking like this:
				// public void com.sibvisions.Object.doAction(UIActionEvent pEvent) throws Throwable
				
				// Split it at the parentheses to receive this part:
				// public void com.sibvisions.Object.doAction
				String[] parantheseSplitted = listenerString.split("\\(");
				
				// Split it at the dot and spaces, the last part is the name of the method.
				String[] dotSplitted = parantheseSplitted[0].split("[\\. ]");
				
				String name = dotSplitted[dotSplitted.length - 1];
				
				// Sanity check.
				if (!name.contains(" ") && !name.contains("."))
				{
					return name;
				}
			}
		}
		
		return null;
	}
	
}	// UIButton
