/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 26.10.2009 - [HM] - creation
 * 13.02.2010 - [JR] - format with Date and long parameter implemented
 * 24.02.2013 - [JR] - convert timezone implemented
 */
package com.sibvisions.util.type;

import java.text.DateFormat;
import java.text.DateFormatSymbols;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Locale;
import java.util.TimeZone;

/**
 * The <code>DateUtil</code> is a utility class for date conversion and for formatting dates
 * as string.
 *  
 * @author Martin Handsteiner
 */
public class DateUtil
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
	/** The text separator. */
	private static final char TEXTSEPARATOR = '\'';

	/** The date format. */
	private SimpleDateFormat dateFormat;
	/** The date format. */
	private SimpleDateFormat parseDateFormat;
	
	/** The locale base for creation. */
	private Locale creationLocale = null;
	/** The pattern base for creation. */
	private String creationPattern = null;
	
	/** Performance tuning, pattern. */
	private List<String>[] parsedPattern = null;
	/** Performance tuning, pattern. */
	private boolean[] ignorePattern = null;
	/** Performance tuning, reference. */
    private List<String>[] parsedReference = null;
	/** Performance tuning, months. */
    private String[] months = null;
	/** Performance tuning, shortMonths. */
    private String[] shortMonths = null;
	/** Performance tuning, months. */
    private String[][] lowerMonths = null;
	/** Performance tuning, shortMonths. */
    private String[][] lowerShortMonths = null;

	/** Performance tuning, weeks. */
    private String[] weekdays = null;
	/** Performance tuning, shortWeeks. */
    private String[] shortWeekdays = null;
	/** Performance tuning, weekdays. */
    private String[] lowerWeekdays = null;
	/** Performance tuning, weekdays. */
    private String[] lowerShortWeekdays = null;

	/** Performance tuning, amPm. */
    private String[] amPm = null;
	/** Performance tuning, amPm. */
    private String[] lowerAmPm = null;

	/** Performance tuning, time zone strings. */
    private String[][] zoneStrings = null;
	/** Performance tuning, time zone strings. */
    private String[][] lowerZoneStrings = null;

    /** True, if the format should be checked strict. */
    private boolean strictFormatCheck = false;
    /** index of hours. */
    private int hoursIndex;
    /** index of amPm. */
    private int amPmIndex;
	
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** 
	 * Constructs a new instance of <code>DateUtil</code> with default date format.
	 */
	public DateUtil()
	{
		setDateFormat(null);
	}

	/** 
	 * Constructs a new instance of <code>DateUtil</code> that supports empty Strings and null values.
	 * 
	 * @param pDateFormat the formatter that should support empty Strings and null values
	 */
	public DateUtil(DateFormat pDateFormat)
	{
		setDateFormat(pDateFormat);
	}

	/** 
	 * Constructs a new instance of <code>DateUtil</code> that supports empty Strings and null values.
	 * 
	 * @param pDatePattern the pattern that should support empty Strings and null values
	 */
	public DateUtil(String pDatePattern)
	{
		setDatePattern(pDatePattern);
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Parses the date from text.
	 * 
	 * @param pText the text.
	 * @return the parsed date.
     * @throws ParseException if there is an error in the conversion
	 */
	public Date parse(String pText) throws ParseException
	{
		if (pText == null || pText.length() == 0)
		{
			return null;
		}
		else
		{
			if (creationLocale != null)
			{
				setDateFormatIntern(LocaleUtil.getDefault(), creationPattern);
			}
			
			return parseStringIntern(pText);
		}
	}

	/**
	 * Formats the date to text.
	 * 
	 * @param pDate the date.
	 * @return the formatted text.
	 */
	public String format(Date pDate)
	{
		if (pDate == null)
		{
			return null;
		}
		else
		{
			if (creationLocale != null)
			{
				setDateFormatIntern(LocaleUtil.getDefault(), creationPattern);
			}
			
			return dateFormat.format(pDate);
		}
	}

	/**
	 * Gets the date format.
	 * 
	 * @return the date format.
	 */
	public DateFormat getDateFormat()
	{
		return dateFormat;
	}

	/**
	 * Sets the new date format, if something was changed.
	 * @param pLocale the locale
	 * @param pDatePattern the pattern
	 */
	private void setDateFormatIntern(Locale pLocale, String pDatePattern)
	{
		if (pDatePattern == null)
		{
			if (pLocale != creationLocale || creationPattern != null)
			{
				dateFormat = (SimpleDateFormat)DateFormat.getDateTimeInstance(DateFormat.LONG, DateFormat.SHORT, LocaleUtil.getDefault());
				
				creationLocale = pLocale;
				creationPattern = pDatePattern;

				createReferenceDate();
			}
		}
		else
		{
			if (pLocale != creationLocale || !pDatePattern.equals(creationPattern))
			{
				dateFormat = new SimpleDateFormat(pDatePattern, LocaleUtil.getDefault());
				
				creationLocale = pLocale;
				creationPattern = pDatePattern;

				createReferenceDate();
			}
		}
	}
	
	/**
	 * Creates a Reference Date.
	 */
	private void createReferenceDate()
	{
		GregorianCalendar cal = new GregorianCalendar();
	    cal.set(Calendar.HOUR_OF_DAY, 0);
	    cal.set(Calendar.MINUTE, 0);
	    cal.set(Calendar.SECOND, 0);
	    cal.set(Calendar.MILLISECOND, 0);

		DateFormatSymbols symbols = dateFormat.getDateFormatSymbols();
		String pattern = dateFormat.toPattern();
		
		parsedPattern = getParsed(pattern, true);

		ignorePattern = new boolean[parsedPattern[0].size()];
		boolean containsDay = false;
		boolean containsDate = false;
		hoursIndex = -1;
		amPmIndex = -1;
		for (int i = 0; i < ignorePattern.length; i++)
		{
			char patChar = parsedPattern[0].get(i).charAt(0);
			
			switch (patChar)
			{
				case 'd':
				case 'D':
				case 'u':
				case 'F': containsDay = true;
				case 'y':
				case 'Y':
				case 'M':
				case 'E':
				case 'W':
				case 'w': containsDate = true; break;
				case 'h':
				case 'H': hoursIndex = i; break;
				case 'a': amPmIndex = i; break;
				default:
			}
		}
		if (!containsDate)
		{
			cal.set(Calendar.MONTH, 0);
			cal.set(Calendar.YEAR, 1970);
			cal.set(Calendar.DAY_OF_MONTH, 1);
		}
	    
		StringBuilder parsePattern = new StringBuilder(48);
		parsePattern.append(parsedPattern[1].get(0));
		for (int i = 0; i < ignorePattern.length; i++)
		{
			String pat = parsedPattern[0].get(i);
			boolean ignoreWeekDay = containsDay && pat.startsWith("E");
			
			ignorePattern[i] = ignoreWeekDay;
			if (!ignoreWeekDay)
			{
				if (pat.startsWith("E"))
				{
					cal.set(Calendar.DAY_OF_WEEK, Calendar.MONDAY);
				}
				parsePattern.append(pat);
				parsePattern.append(parsedPattern[1].get(i + 1));
			}
		}
		parseDateFormat = new SimpleDateFormat(parsePattern.toString(), symbols);
		parseDateFormat.setLenient(false);

		for (int i = 0, size = parsedPattern[1].size(); i < size; i++)
		{
			parsedPattern[1].set(i, eliminateSingleQuote(parsedPattern[1].get(i)));
		}
		
		months = symbols.getMonths();
		shortMonths = symbols.getShortMonths();
		lowerMonths = new String[2][months.length];
		lowerShortMonths = new String[2][months.length];
		for (int i = 0; i < months.length; i++)
		{
			lowerMonths[0][i] = months[i].toLowerCase();
			lowerShortMonths[0][i] = shortMonths[i].toLowerCase();
			lowerMonths[1][i] = toLowerAndWithoutUmlauts(months[i]);
			lowerShortMonths[1][i] = toLowerAndWithoutUmlauts(shortMonths[i]);
		}
		weekdays = symbols.getWeekdays();
		shortWeekdays = symbols.getShortWeekdays();
		lowerWeekdays = new String[weekdays.length];
		lowerShortWeekdays = new String[weekdays.length];
		for (int i = 0; i < weekdays.length; i++)
		{
			lowerWeekdays[i] = toLowerAndWithoutUmlauts(weekdays[i]);
			lowerShortWeekdays[i] = toLowerAndWithoutUmlauts(shortWeekdays[i]);
		}
		amPm = symbols.getAmPmStrings();
		lowerAmPm = new String[amPm.length];
		for (int i = 0; i < amPm.length; i++)
		{
			lowerAmPm[i] = toLowerAndWithoutUmlauts(amPm[i]);
		}
		
		zoneStrings = symbols.getZoneStrings();
		lowerZoneStrings = new String[zoneStrings.length][];
		for (int i = 0; i < zoneStrings.length; i++)
		{
			lowerZoneStrings[i] = new String[5];
			for (int j = 0; j < 5; j++)
			{
				lowerZoneStrings[i][j] = toLowerAndWithoutUmlauts(zoneStrings[i][j]);
			}
		}

	    String referenceDate = dateFormat.format(cal.getTime());
		parsedReference = getParsed(referenceDate, false);
	}

	/**
	 * To lower case and replaces umlauts.
	 * @param pText the original month
	 * @return the converted month.
	 */
	private String toLowerAndWithoutUmlauts(String pText)
	{
		StringBuilder lowerResult = new StringBuilder(pText.length());
		
		char oldCh = Character.MAX_VALUE;
		for (int i = 0, length = pText.length(); i < length; i++)
		{
			char ch = Character.toLowerCase(pText.charAt(i));
			
			switch (ch) 
			{
				case '':
				case '':
				case '':
				case '': lowerResult.append('a'); break;
				case '':
				case '':
				case '':
				case '': lowerResult.append('u'); break;
				case '':
				case '':
				case '':
				case '': lowerResult.append('o'); break;
				case '':
				case '':
				case '': lowerResult.append('o'); break;
				case '':
				case '': lowerResult.append('i'); break;
				case 'e': if (oldCh == 'a' || oldCh == 'u' || oldCh == 'o')
						  {
							  break;
						  }
				default:  lowerResult.append(ch); break;
			}
			oldCh = ch;
		}
		return lowerResult.toString();
	}
	
	/**
	 * Gets the date format.
	 * 
	 * @param pDateFormat the date format.
	 */
	public void setDateFormat(DateFormat pDateFormat)
	{
		if (pDateFormat == null)
		{
			setDateFormatIntern(LocaleUtil.getDefault(), null);
		}
		else if (pDateFormat instanceof SimpleDateFormat)
		{
			dateFormat = (SimpleDateFormat)pDateFormat;

			creationLocale = null;
			creationPattern = null;
			
			createReferenceDate();
		}
		else
		{
			throw new IllegalArgumentException("Only SimpleDateFormat is supported!");
		}
	}

	/**
	 * Gets the date format pattern.
	 * 
	 * @return the date format pattern.
	 */
	public String getDatePattern()
	{
		return dateFormat.toPattern();
	}

	/**
	 * Gets the date format pattern.
	 * 
	 * @param pDatePattern the date format pattern.
	 */
	public void setDatePattern(String pDatePattern)
	{
		if (pDatePattern == null)
		{
			setDateFormat(null);
		}
		else
		{
			setDateFormatIntern(LocaleUtil.getDefault(), pDatePattern);
		}
	}

	/**
	 * Strict format check defines whether the set pattern should be exactly by format, or be more flexibel in analysing the given date.
	 * This has nothing to do with lenient in SimpleDateFormat. Lenient is set to false anyway.
	 * Only correct dates are allowed.
	 * The following will be allowed without strict check, with locale de_AT and pattern: dd. MMMM.yyyy HH:mm
	 *   01.01.2016
	 *   01.Januar.2016 00:00
	 *   01.Jnner.2016
	 *   01.01
	 *   01.01.16
	 *   01.01
	 *   01 01 2016 00 00
	 *   010116
	 *   01012016
	 *   
	 * The result will always be 01.Januar.2016 00:00
	 *  
	 * @return true, if format should be checked strict.
	 */
	public boolean isStrictFormatCheck()
	{
		return strictFormatCheck;
	}
	
	/**
	 * Strict format check defines whether the set pattern should be exactly by format, or be more flexibel in analysing the given date.
	 * This has nothing to do with lenient in SimpleDateFormat. Lenient is set to false anyway.
	 * Only correct dates are allowed.
	 * The following will be allowed without strict check, with locale de_AT and pattern: dd. MMMM.yyyy HH:mm
	 *   01.01.2016
	 *   01.Januar.2016 00:00
	 *   01.Jnner.2016
	 *   01.01
	 *   01.01.16
	 *   01.01
	 *   01 01 2016 00 00
	 *   010116
	 *   01012016
	 *   
	 * The result will always be 01.Januar.2016 00:00
	 *  
	 * @param pStrictFormatCheck true, if format should be checked strict.
	 */
	public void setStrictFormatCheck(boolean pStrictFormatCheck)
	{
		strictFormatCheck = pStrictFormatCheck;
		
	}
	
	/**
	 * Gets the index of found name. 
	 * @param pLowerPart the part to search
	 * @param pNames the names
	 * @param pShortNames the short names
	 * @return the index.
	 */
	private int findIndex(String pLowerPart, String[] pNames, String[] pShortNames)
	{
		for (int i = 0; i < pNames.length; i++)
		{
			if (pNames[i].startsWith(pLowerPart) || (pLowerPart.length() <= pNames[i].length() + 2 && pShortNames[i].length() > 0 && pLowerPart.startsWith(pShortNames[i])))
			{
				return i;
			}
		}
		return -1;
	}
	
	/**
	 * Gets the correct spelled months depending on the pattern, ignoring umlauts, and korrekt length.
	 * @param pPart the part to parse
	 * @param pPattern the exact month pattern
	 * @return the correct spelled months depending on the pattern, ignoring umlauts, and korrekt length.
	 */
	private String getMonthPart(String pPart, String pPattern)
	{
		String[] lowerPart = new String[] {pPart.toLowerCase(), toLowerAndWithoutUmlauts(pPart)};
		for (int i = 0; i < 2; i++)
		{
			int mon = findIndex(lowerPart[i], lowerMonths[i], lowerShortMonths[i]) + 1;
			if (mon > 0)
			{
				if (pPattern.length() == 2 && mon < 10)
				{
					return "0" + mon;
				}
				else if (pPattern.length() <= 2)
				{
					return String.valueOf(mon);
				}
				else if (pPattern.length() == 3)
				{
					return shortMonths[mon - 1];
				}
				else
				{
					return months[mon - 1];
				}
			}
		}
		return pPart;
	}
	
	/**
	 * Gets the correct spelled weekday depending on the pattern, ignoring umlauts, and korrekt length.
	 * @param pPart the part to parse
	 * @param pPattern the exact weekday pattern
	 * @return the correct spelled weekday depending on the pattern, ignoring umlauts, and korrekt length.
	 */
	private String getWeekdayPart(String pPart, String pPattern)
	{
		int index = findIndex(toLowerAndWithoutUmlauts(pPart), lowerWeekdays, lowerShortWeekdays);
		if (index >= 0)
		{
			if (pPattern.length() <= 3)
			{
				return shortWeekdays[index];
			}
			else
			{
				return weekdays[index];
			}
		}
		return pPart;
	}

	/**
	 * Gets the correct spelled am pm part.
	 * @param pPart the am pm string
	 * @param pPattern the pattern
	 * @return the correct spelled am pm part.
	 */
	private String getAmPmPart(String pPart, String pPattern)
	{
		int index = findIndex(toLowerAndWithoutUmlauts(pPart), lowerAmPm, lowerAmPm);
		if (index >= 0)
		{
			return amPm[index];
		}
		return pPart;
	}
	
	/**
	 * Gets the best transformation of the date part for the pattern.
	 * 
	 * @param pPart the date part
	 * @param pPattern the pattern
	 * @param pReference the the reference date part
	 * @return the best transformation of the date part for the pattern
	 */
	private String getPart(String pPart, String pPattern, String pReference)
	{
		try
		{
			if (pPattern.startsWith("MMM"))
			{
				int number = Integer.parseInt(pPart);
				if (pPattern.length() == 3)
				{
					return shortMonths[(number - 1) % 12];
				}
				else
				{
					return months[(number - 1) % 12];
				}
			}
			else if (pPattern.toLowerCase().startsWith("yy"))
			{
				if (pReference.length() > pPart.length())
				{
					int year = Integer.parseInt(pReference.substring(0, pReference.length() - pPart.length()) + pPart);
					int refYear = Integer.parseInt(pReference);
					if (pPart.length() == 2 && year >= refYear + 50)
					{
						year -= 100;
					}

					return String.valueOf(year);
				}
			}
		}
		catch (Exception ex)
		{
			// Do nothing
		}
		
		switch (pPattern.charAt(0))
		{
			case 'M':	return getMonthPart(pPart, pPattern);
			case 'E':	return getWeekdayPart(pPart, pPattern);
			default:	return pPart;
		}
	}

	/**
	 * Gets the stric transformation of the date part for the pattern.
	 * 
	 * @param pPart the date part
	 * @param pPattern the pattern
	 * @param pReference the the reference date part
	 * @return the best transformation of the date part for the pattern
	 */
	private String getStrictPart(String pPart, String pPattern, String pReference)
	{
		if (pPattern.startsWith("MMM"))
		{
			return getMonthPart(pPart, pPattern);
		}
		else if (pPattern.startsWith("E"))
		{
			return getWeekdayPart(pPart, pPattern);
		}

		return pPart;
	}
	
	/**
	 * Gets the character type depending on format or date parsing.
	 * @param pCharacter the character
	 * @param pFormat the format
	 * @return the character type
	 */
	private char getCharacterType(char pCharacter, boolean pFormat)
	{
		if (Character.isLetterOrDigit(pCharacter))
		{
			if (pFormat)
			{
				return pCharacter;
			}
			else
			{
				return Character.isDigit(pCharacter) ? '0' : 'a';
			}
		}
		else if (pFormat && pCharacter == TEXTSEPARATOR)
		{
			return pCharacter;
		}
		else
		{
			return ' ';
		}
	}
	
	/**
	 * Eliminates text separator characters.
	 * @param pSeparator the separator string
	 * @return separator string
	 */
	private String eliminateSingleQuote(String pSeparator)
	{
		int index = pSeparator.indexOf(TEXTSEPARATOR);
		
		if (index >= 0)
		{
			StringBuilder result = new StringBuilder(pSeparator.length());
			int start = 0;
			while (index >= 0)
			{
				result.append(pSeparator.substring(start, index));
				
				start = index + 1;
				index = pSeparator.indexOf(TEXTSEPARATOR, start);
				if (index == start)
				{
					start--;
				}
			}
			result.append(pSeparator.substring(start));
			
			return result.toString();
		}
		
		return pSeparator;
	}
	
	/**
	 * Adds a separator and initializes the next separatorPattern. 
	 * @param pSeperatorPart the separator part
	 * @param pSeparator the separator
	 * @param pFormat if a format is parsed.
	 * @return the next text separator.
	 */
	private String addSeparator(List<String> pSeperatorPart, String pSeparator, boolean pFormat)
	{
		pSeperatorPart.add(pSeparator);
		if (pFormat)
		{
			return null;
		}
		else if (pSeperatorPart.size() < parsedPattern[1].size())
		{
			return parsedPattern[1].get(pSeperatorPart.size());
		}
		else
		{
			return "";
		}
	}
	
	/**
	 * Parse and check time zone info.
	 * @param pText the text to parse
	 * @param pTimeZonePart the result timeZonePart
	 * @return the endIndex.
	 */
	private int getTimeZonePart(String pText, StringBuilder pTimeZonePart)
	{
		String lowerText = toLowerAndWithoutUmlauts(pText);
		
		int pos = 0;
		if (lowerText.startsWith("gmt"))
		{
			pos = 3;
			
			while (pos < lowerText.length() && Character.isWhitespace(lowerText.charAt(pos)))
			{
				pos++;
			}
		}
		if (pos < lowerText.length())
		{
			char sign = lowerText.charAt(pos);
			if (sign == '+' || sign == '-')
			{
				pos++;
			}
			else if (pos > 0)
			{
				sign = '+';
			}
	
			if (pos == 0)
			{
				for (int i = 0; i < lowerZoneStrings.length; i++)
				{
					for (int j = 0; j < 5; j++)
					{
						if (lowerText.startsWith(lowerZoneStrings[i][j]))
						{
							pTimeZonePart.append(zoneStrings[i][j]);
							
							return pTimeZonePart.length();
						}
					}
				}
				if (sign == 'z') // Zulu time support.
				{
					pTimeZonePart.append("+0000");
					return 1;
				}
			}
			else 
			{
				char ch = lowerText.charAt(pos);
				if (Character.isDigit(ch))
				{
					pTimeZonePart.append(sign);
					do
					{
						if (ch == ':')
						{
							if (pTimeZonePart.length() < 3)
							{
								pTimeZonePart.insert(1, '0');
							}
						}
						else
						{
							pTimeZonePart.append(ch);
						}
						
						pos++;
						if (pos < lowerText.length())
						{
							ch = lowerText.charAt(pos);
						}
						else
						{
							ch = Character.MAX_VALUE;
						}
					}
					while (pTimeZonePart.length() < 5 && (Character.isDigit(ch) || ch == ':'));
					
					if (pTimeZonePart.length() < 5)
					{
						if (pTimeZonePart.charAt(1) != '0')
						{
							pTimeZonePart.insert(1, '0');
						}
						while (pTimeZonePart.length() < 5)
						{
							pTimeZonePart.append('0');
						}
					}
					
					return pos;
				}
			}
		}
		if (pos > 0)
		{
			pTimeZonePart.append("+0000");
			return pos;
		}
		else
		{
			return -1;
		}
	}
	
	
	/**
	 * Parses the date parts and the seperator parts.
	 * 
	 * @param pText the date
	 * @param pFormat true, if it is a format
	 * @return the date parts and the seperator parts
	 */
	private List<String>[] getParsed(String pText, boolean pFormat)
	{
		List<String> datePart = new ArrayList<String>();
		List<String> seperatorPart = new ArrayList<String>();
		
		int len = pText.length();
		int pos = 0;
		char ch = pText.charAt(0);
		boolean isLetterOrDigit = Character.isLetterOrDigit(ch);
		char characterType = getCharacterType(ch, pFormat);
		boolean isTextSeparator = false;
		String textSeparator = pFormat ? null : parsedPattern[1].get(0);
		for (int i = 1; i < len; i++)
		{
			char newCh = pText.charAt(i);
			
			boolean newIsLetterOrDigit = Character.isLetterOrDigit(newCh);
			char newCharacterType = getCharacterType(newCh, pFormat);

			if (pFormat && characterType == TEXTSEPARATOR)
			{
				if (isTextSeparator)
				{
					if (newCharacterType != TEXTSEPARATOR)
					{
						isTextSeparator = false;
					}
				}
				else
				{
					if (newCharacterType != TEXTSEPARATOR)
					{
						isTextSeparator = true;
					}
				}
			}
			
			if (!isTextSeparator)
			{
				// Find textual separator and prevent it is parsed as date part.
				if (!pFormat && textSeparator.length() > 0 && pText.substring(i - 1).startsWith(textSeparator))
				{
					isLetterOrDigit = false; 
					characterType = ' ';

					i += textSeparator.length() - 1;
					newCh = pText.charAt(i);
					newIsLetterOrDigit = Character.isLetterOrDigit(newCh);
					newCharacterType = getCharacterType(newCh, pFormat);
				}
				String part = pText.substring(pos, i);
				if (!pFormat)
				{
					int maxPrecision = 0; // 0 means not a number, >0 means max size of the number for the corresponding parsed pattern
					char patternChar = Character.MAX_VALUE;
					boolean isTimeZone = false;
					if (datePart.size() < parsedPattern[0].size())
					{
						patternChar = parsedPattern[0].get(datePart.size()).charAt(0);
						switch (patternChar)
						{
							case 'y': maxPrecision = 4; break;
							case 'S': maxPrecision = 3; break;
							case 'z':
							case 'Z':
							case 'X': isTimeZone = true;
							case 'E':
							case 'a':
							case 'G': maxPrecision = 0; break;
							default: maxPrecision = 2;
						}
					}
					
					if (!pFormat && isTimeZone && (newIsLetterOrDigit || newCh == '+' || newCh == '-'))
					{
						if (!isLetterOrDigit)
						{
							textSeparator = addSeparator(seperatorPart, part, pFormat);
							pos = i;
						}
						
						StringBuilder timeZonePart = new StringBuilder(pText.length());
							
						int endIndex = getTimeZonePart(pText.substring(pos), timeZonePart);
						
						if (endIndex < 0)
						{
							datePart.add(parsedReference[0].get(datePart.size()));
						}
						else
						{
							datePart.add(timeZonePart.toString());
							pos += endIndex;
							i = pos - 1;
							newCh = pText.charAt(i);
							newIsLetterOrDigit = true;
							newCharacterType = 'z';
							isLetterOrDigit = newIsLetterOrDigit;
							characterType = newCharacterType;
						}
					}
					// if it is a large number, delimiters should be filled in automatically based on the max precision of the number. 
					if (isLetterOrDigit && newIsLetterOrDigit && characterType == '0' && newCharacterType == '0' && part.length() == maxPrecision)
					{
						characterType = '1';
					}
					// fill in numbers from reference, if we know that there should be one.
					else if (newCharacterType == 'a' && maxPrecision > 0 && patternChar != 'M') 
					{
						if (isLetterOrDigit)
						{
							datePart.add(part);
							isLetterOrDigit = false;
							characterType = ' ';
							pos = i;
						}
						else
						{
							datePart.add(parsedReference[0].get(datePart.size()));
						}
						i--;
						textSeparator = addSeparator(seperatorPart, "", pFormat);
						newCh = pText.charAt(i);
						newIsLetterOrDigit = isLetterOrDigit;
						newCharacterType = characterType;
					}
					// if the weekday is missing, add a proper one.
					else if (isLetterOrDigit && datePart.size() < parsedPattern[0].size() && parsedPattern[0].get(datePart.size()).startsWith("E") && lowerWeekdays != null
							&& findIndex(toLowerAndWithoutUmlauts(part), lowerWeekdays, lowerShortWeekdays) < 0)
					{
						datePart.add(parsedReference[0].get(datePart.size())); // Use from reference date.
//						datePart.add(getWeekdayPart(lowerWeekdays[Calendar.MONDAY], parsedPattern[0].get(datePart.size()))); // use monday.
						textSeparator = addSeparator(seperatorPart, "", pFormat);
					}
				}
				
				if (isLetterOrDigit != newIsLetterOrDigit
						|| (isLetterOrDigit && characterType != newCharacterType))
				{
					if (isLetterOrDigit)
					{
						datePart.add(part);

						if (seperatorPart.size() == 0)
						{
							textSeparator = addSeparator(seperatorPart, "", pFormat);
						}
						if (isLetterOrDigit == newIsLetterOrDigit && characterType != newCharacterType)
						{
							textSeparator = addSeparator(seperatorPart, "", pFormat);
						}
					}
					else
					{
						textSeparator = addSeparator(seperatorPart, part, pFormat);
					}
					pos = i;
				}
			}
			isLetterOrDigit = newIsLetterOrDigit;
			characterType = newCharacterType;
			ch = newCh;
		}
		String part = pText.substring(pos);
		if (isLetterOrDigit)
		{
			datePart.add(part);
			if (seperatorPart.size() == 0)
			{
				seperatorPart.add("");
			}
			seperatorPart.add("");
		}
		else
		{
			seperatorPart.add(part);
		}

		return new List[] {datePart, seperatorPart};
	}
	
    /**
     * Parses <code>text</code> returning an Date. Some
     * formatters may return null.
     *
     * @param pText String to convert
     * @return Date representation of text
     * @throws ParseException if there is an error in the conversion
     */
	private Date parseStringIntern(String pText) throws ParseException
	{
		try
		{
		    List<String>[] parsedDate = getParsed(pText, false);
		    
		    int datePartCount = parsedDate[0].size();
		    int patternSeperatorCount = strictFormatCheck ? parsedDate[1].size() : parsedPattern[0].size();

		    String amPmPart = null;
		    if (!strictFormatCheck)
    		{
		    	if (amPmIndex >= 0)
		    	{
		    		if (amPmIndex < parsedDate[0].size())
		    		{
		    			amPmPart = getAmPmPart(parsedDate[0].get(amPmIndex), parsedPattern[0].get(amPmIndex));
		    		}
		    		else
		    		{
		    			amPmPart = parsedReference[0].get(amPmIndex);
		    		}
		    		if (hoursIndex >= 0 && hoursIndex < parsedDate[0].size())
		    		{
		    			try
		    			{
		    				int hours = Integer.parseInt(parsedDate[0].get(hoursIndex));
		    				boolean h24 = parsedPattern[0].get(hoursIndex).charAt(hoursIndex) == 'H'; 
	    					if (hours >= 12)
	    					{
	    						amPmPart = amPm[Calendar.PM];
	    						if (!h24)
	    						{
	    							parsedDate[0].set(hoursIndex, String.valueOf(88 + hours).substring(1));
	    						}
	    					}
	    					else if (h24)
	    					{
	    						amPmPart = amPm[Calendar.AM];
	    					}
		    			}
		    			catch (Exception ex)
		    			{
		    				// Do nothing
		    			}
		    		}
		    	}
    		}
		    
		    StringBuilder buffer = new StringBuilder(48);
		    if (strictFormatCheck)
    		{
    			buffer.append(parsedDate[1].get(0));
    		}
    		else
    		{
    			buffer.append(parsedPattern[1].get(0));
    		}
		    for (int i = 0, size = parsedPattern[0].size(); i < size; i++)
		    {
		    	if (!ignorePattern[i])
		    	{
		    		if (i == amPmIndex && amPmPart != null)
		    		{
		    			buffer.append(amPmPart);
		    		}
		    		else if (i < datePartCount)
			    	{
			    		if (strictFormatCheck)
			    		{
			    			buffer.append(getStrictPart(parsedDate[0].get(i), parsedPattern[0].get(i), parsedReference[0].get(i)));
			    		}
			    		else
			    		{
			    			buffer.append(getPart(parsedDate[0].get(i), parsedPattern[0].get(i), parsedReference[0].get(i)));
			    		}
			    	}
			    	else if (!strictFormatCheck)
			    	{
			    		buffer.append(parsedReference[0].get(i));
			    	}
			    	if (i < patternSeperatorCount)
			    	{
			    		if (strictFormatCheck)
			    		{
			    			buffer.append(parsedDate[1].get(i + 1));
			    		}
			    		else
			    		{
			    			buffer.append(parsedPattern[1].get(i + 1));
			    		}
			    	}
		    	}
		    }

		    return parseDateFormat.parse(buffer.toString());
		}
		catch (ParseException exception)
		{
			throw (ParseException)exception;
		}
		catch (Exception exception)
		{
			throw new ParseException("Wrong Dateformat", 0);
		}
	}

	/**
	 * Formats a time string.
	 * 
	 * @param pDate the time (in millis since 01.01.1970 00:00) value to be formatted into a time string
	 * @param pFormat the format 
	 * @return the formatted date/time string
	 * @see SimpleDateFormat
	 */
	public static String format(long pDate, String pFormat)
	{
		return format(new Date(pDate), pFormat);
	}
	
	/**
	 * Formats a Date into a date/time string.
	 * 
	 * @param pDate the time value to be formatted into a time string
	 * @param pFormat the format 
	 * @return the formatted date/time string
	 * @see SimpleDateFormat
	 */
	public static String format(Date pDate, String pFormat)
	{
		SimpleDateFormat sdf = new SimpleDateFormat(pFormat);
		
		return sdf.format(pDate);
	}
	
	/**
	 * Creates a new date instance.
	 * 
	 * @param pDay the day of month
	 * @param pMonth the month
	 * @param pYear the year
	 * @param pHour the hour of day
	 * @param pMinutes the minutes
	 * @param pSeconds the seconds
	 * @return the date
	 */
	public static Date getDate(int pDay, int pMonth, int pYear, int pHour, int pMinutes, int pSeconds)
	{
		Calendar cal = GregorianCalendar.getInstance();
		
		cal.set(Calendar.DAY_OF_MONTH, pDay);
		cal.set(Calendar.MONTH, pMonth - 1);
		cal.set(Calendar.YEAR, pYear);
		cal.set(Calendar.HOUR_OF_DAY, pHour);
		cal.set(Calendar.MINUTE, pMinutes);
		cal.set(Calendar.SECOND, pSeconds);
		cal.set(Calendar.MILLISECOND, 0);
		
		return cal.getTime();
	}
	
	/**
	 * Converts a date from one timezone to another timezone.
	 * 
	 * @param pDate the date
	 * @param pFromTimeZone the timezone of the date
	 * @param pToTimeZone the expected timezone
	 * @return the date converted to the expected timezone
	 */
	public static Date convert(Date pDate, String pFromTimeZone, String pToTimeZone)
	{
		return convert(pDate, TimeZone.getTimeZone(pFromTimeZone), TimeZone.getTimeZone(pToTimeZone));
	}

	/**
	 * Converts a date from one timezone to another timezone.
	 * 
	 * @param pDate the date
	 * @param pFromTimeZone the timezone of the date
	 * @param pToTimeZone the expected timezone
	 * @return the date converted to the expected timezone
	 */
	public static Date convert(Date pDate, TimeZone pFromTimeZone, TimeZone pToTimeZone)
	{
		if (pDate == null)
		{
			return null;
		}
		
		Calendar fromCal = Calendar.getInstance(pFromTimeZone);
		fromCal.setTime(pDate);
		   
		Calendar toCal = Calendar.getInstance(pToTimeZone);
		toCal.setTime(pDate);

		int iFromOffset = fromCal.get(Calendar.ZONE_OFFSET) + fromCal.get(Calendar.DST_OFFSET);
		int iToOffset = toCal.get(Calendar.ZONE_OFFSET) + toCal.get(Calendar.DST_OFFSET);

		return new Date(toCal.getTimeInMillis() + iToOffset - iFromOffset);		
	}
	
}	// DateUtil
