/*
 * Copyright 2015 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 29.09.2015 - [JR] - creation
 */
package com.sibvisions.rad.ui.swing.ext;

import java.awt.Color;
import java.awt.Component;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.GraphicsEnvironment;
import java.awt.Image;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.font.FontRenderContext;
import java.awt.font.TextLayout;
import java.awt.image.BufferedImage;
import java.io.InputStream;
import java.util.HashMap;

import javax.swing.ImageIcon;

import com.sibvisions.rad.ui.ApplicationUtil;
import com.sibvisions.rad.ui.swing.ext.fonts.FontAwesome;
import com.sibvisions.util.type.StringUtil;

/**
 * The <code>JVxFontAwesomeIcon</code> is an {@link ImageIcon} implementation for FontAwesome font.
 * 
 * @author Ren Jahn
 * @see <a href="http://fortawesome.github.io/Font-Awesome/cheatsheet/">http://fortawesome.github.io/Font-Awesome/cheatsheet/</a>
 */
public class JVxFontAwesomeIcon extends ImageIcon
{
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Class members
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    /** the font base. */
    private static final Font   BASEFONT;

    
    /** the cached image (paint performance). */
    private BufferedImage       buffer;

    /** the icon definition. */
    private FontAwesome         faIconDefinition;

    /** the current font. */
    private Font                font;

    /** the set color. */
    private Color               color;

    /** the set size. */
    private int                 size;
    
    /** the "real" width. */
    private int                 width;
    
    /** the real height. */
    private int                 height;
    
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Initialization
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    static
    {
        try
        {
            InputStream stream = JVxFontAwesomeIcon.class.getResourceAsStream("/com/sibvisions/rad/ui/swing/ext/fonts/fontawesome-webfont.ttf");
            
            try
            {
                BASEFONT = Font.createFont(Font.TRUETYPE_FONT, stream);
                
                GraphicsEnvironment.getLocalGraphicsEnvironment().registerFont(BASEFONT);
            }
            finally
            {
                stream.close();
            }
        }
        catch (Exception ex)
        {
            throw new RuntimeException(ex);
        }
    }

    /**
    * Creates a new instance of <code>JVxFontAwesomeIcon</code> with default size of 16px.
    * 
    * @param pFont the icon definition
    */
    public JVxFontAwesomeIcon(FontAwesome pFont) 
    {
        size = 16;
        
        init();
    }

    /**
     * Creates a new instance of <code>JVxFontAwesomeIcon</code>.
     * 
     * @param pDefinition The icon definition (only the name or followed by a list of properties:
     *                    server;size=20;color=0xFFAACC
     */
    public JVxFontAwesomeIcon(String pDefinition)
    {
        parse(pDefinition);
        
        init();
    }

    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Interface implementation
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * {@inheritDoc}
     */
    public synchronized void paintIcon(Component pComponent, Graphics pGraphics, int pX, int pY)
    {
        if (buffer == null)
        {
            buffer = new BufferedImage(getIconWidth(), getIconHeight(), BufferedImage.TYPE_INT_ARGB);

            Graphics2D g2 = (Graphics2D)buffer.getGraphics();
            g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
            g2.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);

            g2.setFont(font);
            g2.setPaint(color);

            FontMetrics fontMetrics = g2.getFontMetrics(font);

            int drawAtX = (buffer.getWidth() - fontMetrics.stringWidth(String.valueOf(faIconDefinition.getCode()))) / 2;
            int drawAtY = (buffer.getHeight() - fontMetrics.getHeight()) / 2 + fontMetrics.getAscent();

            g2.drawString(String.valueOf(faIconDefinition.getCode()), (int)drawAtX, (int)drawAtY);
            g2.dispose();
        }

        pGraphics.drawImage(buffer, pX, pY, null);
    }
    
    /**
     * {@inheritDoc}
     */
    public int getIconWidth()
    {
        return width;
    }

    /**
     * {@inheritDoc}
     */
    public int getIconHeight()
    {
        return height;
    }
    
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Overwritten methods
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * {@inheritDoc}
     */
    @Override
    public Image getImage()
    {
        BufferedImage image = new BufferedImage(getIconWidth(), getIconHeight(), BufferedImage.TYPE_INT_ARGB);
        
        Graphics2D g2 = (Graphics2D) image.getGraphics();
        paintIcon(null, g2, 0, 0);
        g2.dispose();
        
        return image;    
    }
    
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // User-defined methods
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * Initializes internal members and resets cache.
     */
    private void init()
    {
        font = BASEFONT.deriveFont(Font.PLAIN, size);
        
        //get "real" icon size
        BufferedImage tmp = new BufferedImage(size, size, BufferedImage.TYPE_INT_ARGB);
        
        Graphics2D g2 = GraphicsEnvironment.getLocalGraphicsEnvironment().createGraphics(tmp);
        g2.setFont(font);

        FontRenderContext ctxt = g2.getFontRenderContext();
        
        TextLayout tlChar = new TextLayout("" + faIconDefinition.getCode(), font, ctxt);
        
        Rectangle rectBounds = tlChar.getPixelBounds(ctxt, 0, 0);

        width = Math.max(size, rectBounds.width);
        height = Math.max(size, rectBounds.height + 1);
        
        g2.dispose();        
        
        //recreate cache
        buffer = null;
    }
    
    /**
     * Parses given icon definition.
     * 
     * @param pDefinition the icon definition: fontname[;prop1=value1;prop2=value2]
     * @throws IllegalArgumentException if icon name wasn't set or parsing failed
     */
    private void parse(String pDefinition)
    {
        if (StringUtil.isEmpty(pDefinition))
        {
            throw new IllegalArgumentException("Invalid icon definition: " + pDefinition);
        }

        HashMap<String, String> hmpProps = ApplicationUtil.splitImageProperties(pDefinition);
        faIconDefinition = FontAwesome.resolve(hmpProps.get("name"));
        
        String sSize = hmpProps.get("size");
        
        if (sSize == null)
        {
            sSize = hmpProps.get("font-size");
            
            if (sSize == null)
            {
                sSize = "16";
            }
        }
        
        try
        {
            size = Integer.parseInt(sSize);
        }
        catch (NumberFormatException nfe)
        {
            //remove optional unit
            sSize = StringUtil.getText(sSize, '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', ',', '.');
        }
        
        size = Integer.parseInt(sSize);
        
        String sColor = hmpProps.get("color");
        
        int[] iaColor = StringUtil.parseColor(sColor);
        
        if (iaColor != null)
        {
            if (iaColor.length == 3)
            {
                color = new Color(iaColor[0], iaColor[1], iaColor[2]);
            }
            else
            {
                color = new Color(iaColor[0], iaColor[1], iaColor[2], iaColor[3]);
            }
        }
        
        if (color == null)
        {
            color = Color.BLACK;
        }
    }
    
    /**
     * Sets the size.
     * 
     * @param pSize the size
     */
    public void setSize(int pSize)
    {
        size = pSize;
        
        init();
    }
    
    /**
     * Gets the size.
     * 
     * @return the size
     */
    public int getSize()
    {
        return size;
    }

    /**
     * Sets the color.
     * 
     * @param pColor the color
     */
    public void setColor(Color pColor)
    {
        color = pColor;
        
        buffer = null;
    }
    
    /**
     * Gets the color.
     * 
     * @return the color
     */
    public Color getColor()
    {
        return color;
    }
    
}   // JVxFontAwesomeIcon
