/*
 * Copyright 2015 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 11.11.2015 - [RZ] - creation
 */
package com.sibvisions.rad.ui.celleditor;

import java.lang.ref.WeakReference;
import java.util.HashMap;
import java.util.Map;

import javax.rad.model.ColumnView;
import javax.rad.model.IDataBook;
import javax.rad.model.IDataPage;
import javax.rad.model.IDataRow;
import javax.rad.model.ModelException;
import javax.rad.model.condition.ICondition;
import javax.rad.model.datatype.IDataType;
import javax.rad.model.reference.ColumnMapping;
import javax.rad.model.reference.ReferenceDefinition;
import javax.rad.ui.IDimension;
import javax.rad.ui.celleditor.ILinkedCellEditor;

import com.sibvisions.util.type.CommonUtil;

/**
 * The {@link AbstractLinkedCellEditor} is an {@link ILinkedCellEditor}
 * implementation, which provides a base implementation.
 * 
 * @author Robert Zenz
 */
public abstract class AbstractLinkedCellEditor extends AbstractComboCellEditor implements ILinkedCellEditor
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/** True, if header should be visible depending of column number. */
	protected boolean autoTableHeaderVisibility = true;
	
	/** The additional condition. */
	protected ICondition additionalCondition;
	
	/** The {@link ColumnView}. */
	protected ColumnView columnView;
	
	/** The name of the display referenced column. */
	protected String displayReferencedColumnName = null;
	
	/** The link reference. */
	protected ReferenceDefinition linkReference;
	
	/** The size used for the popup. */
	protected IDimension popupSize;
	
	/** The {@link ColumnMapping}. */
	protected ColumnMapping searchColumnMapping;
	
	/** If the text should be searched anywhere. */
	protected boolean searchTextAnywhere = true;
	
	/** If the values should be sorted by column name. */
	protected boolean sortByColumnName;
	
	/** If the table header should be visible. */
	protected boolean tableHeaderVisible = false;
	
	/** If the table should be read-only. */
	protected boolean tableReadOnly = true;
	
	/** If only values from the table are allowed. */
	protected boolean validationEnabled = true;
	
	/**
	 * The display values will be cached in this {@link Map} so that the
	 * conversion to a String is only done once in a render cycle.
	 */
	private Map<String, Integer> displayValueCache = new HashMap<String, Integer>();
	
	/** The last column, with which was searched. */
	private String lastColumnForSearch = null;
	
	/** The last display column. */
	private String lastDisplayColumn = null;
	
	/** The last data page, in which was searched. */
	private WeakReference<IDataPage> lastPageForSearch = null;
	
	/**
	 * The last search row, which is used when building the display value cache.
	 */
	private int lastSearchRow = 0;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Creates a new instance of {@link AbstractLinkedCellEditor}.
	 */
	protected AbstractLinkedCellEditor()
	{
		super();
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * {@inheritDoc}
	 */
	public ICondition getAdditionalCondition()
	{
		return additionalCondition;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public ColumnView getColumnView()
	{
		return columnView;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public String getDisplayReferencedColumnName()
	{
		return displayReferencedColumnName;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public ReferenceDefinition getLinkReference()
	{
		return linkReference;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public IDimension getPopupSize()
	{
		return popupSize;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public ColumnMapping getSearchColumnMapping()
	{
		return searchColumnMapping;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public boolean isSearchTextAnywhere()
	{
		return searchTextAnywhere;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public boolean isSortByColumnName()
	{
		return sortByColumnName;
	}
	
	/**
	 * The header of a table can't be hidden.
	 * 
	 * @return always {@code false}.
	 */
	public boolean isTableHeaderVisible()
	{
		return tableHeaderVisible;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public boolean isTableReadonly()
	{
		return tableReadOnly;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public boolean isValidationEnabled()
	{
		return validationEnabled;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setAdditionalCondition(ICondition pCondition)
	{
		additionalCondition = pCondition;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setColumnView(ColumnView pColumnView)
	{
		columnView = pColumnView;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setDisplayReferencedColumnName(String pDisplayReferencedColumnName)
	{
		displayReferencedColumnName = pDisplayReferencedColumnName;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setLinkReference(ReferenceDefinition pReferenceDefinition)
	{
		linkReference = pReferenceDefinition;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setPopupSize(IDimension pPopupSize)
	{
		popupSize = pPopupSize;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setSearchColumnMapping(ColumnMapping pSearchColumnNames)
	{
		searchColumnMapping = pSearchColumnNames;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setSearchTextAnywhere(boolean pSearchTextAnywhere)
	{
		searchTextAnywhere = pSearchTextAnywhere;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setSortByColumnName(boolean pSortByColumnName)
	{
		sortByColumnName = pSortByColumnName;
	}
	
	/**
	 * Does nothing, the header of a table can't be hidden.
	 * 
	 * @param pTableHeaderVisible ignored.
	 */
	public void setTableHeaderVisible(boolean pTableHeaderVisible)
	{
		autoTableHeaderVisibility = false;
		
		tableHeaderVisible = pTableHeaderVisible;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setTableReadonly(boolean pTableReadonly)
	{
		tableReadOnly = pTableReadonly;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void setValidationEnabled(boolean pValidationEnabled)
	{
		validationEnabled = pValidationEnabled;
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isDirectCellEditor()
	{
		return false;
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Gets the correct display value for the given row and column.
	 * 
	 * @param pDataRow the data row
	 * @param pColumnName the column
	 * @return the display value.
	 * @throws ModelException if it fails.
	 */
	protected String getDisplayValue(IDataRow pDataRow, String pColumnName) throws ModelException
	{
		IDataType dataType = pDataRow.getRowDefinition().getColumnDefinition(pColumnName).getDataType();
		
		Object value = pDataRow.getValue(pColumnName);
		String stringValue = dataType.convertToString(value);
		
		if (displayReferencedColumnName != null && linkReference != null)
		{
			Integer index = displayValueCache.get(stringValue);

			IDataBook refDataBook = linkReference.getReferencedDataBook();
			IDataPage page = refDataBook.getDataPage();
			
			String refColumn = linkReference.getReferencedColumnName(pColumnName);
			IDataType refDataType = refDataBook.getRowDefinition().getColumnDefinition(refColumn).getDataType();
			
			IDataRow refDataRow = null;
			String refStringValue = null;
			
			if (refDataBook.getFilter() != null) // ensure, that the filter is null, otherwise value will not be found
			{
				refDataBook.setFilter(null);
			}

			if (index != null)
			{
				refDataRow = refDataBook.getDataRow(index.intValue());
				if (refDataRow != null)
				{
					Object refValue = refDataRow.getValue(refColumn);
					try
					{
						refStringValue = dataType.convertToString(dataType.convertToTypeClass(refValue));
					}
					catch (Exception ex)
					{
						refStringValue = refDataType.convertToString(refValue);
					}
				}
			}
			
			if (lastPageForSearch == null 
					|| lastPageForSearch.get() != page
					|| !CommonUtil.equals(refColumn, lastColumnForSearch)
					|| !CommonUtil.equals(displayReferencedColumnName, lastDisplayColumn)
					|| (index != null && (refDataRow == null || !CommonUtil.equals(stringValue, refStringValue))))
			{
				displayValueCache.clear();
				
				lastPageForSearch = new WeakReference<IDataPage>(page);
				lastColumnForSearch = refColumn;
				lastDisplayColumn = displayReferencedColumnName;
				
				lastSearchRow = 0;
				index = null;
			}
			
			if (index == null)
			{
				if (refDataBook.getFilter() != null) // ensure, that the filter is null, otherwise value will not be found
				{
					refDataBook.setFilter(null);
				}
				refDataRow = refDataBook.getDataRow(lastSearchRow);
				
				while (refDataRow != null) // step through the data book beginning with the last search row
				{
					Object refValue = refDataRow.getValue(refColumn);
					try
					{
						refStringValue = dataType.convertToString(dataType.convertToTypeClass(refValue));
					}
					catch (Exception ex)
					{
						refStringValue = refDataType.convertToString(refValue);
					}

					String refDisplayValue = refDataRow.getValueAsString(displayReferencedColumnName);
					
					displayValueCache.put(refStringValue, Integer.valueOf(lastSearchRow));
					
					if (CommonUtil.equals(stringValue, refStringValue))
					{
						if (refDisplayValue != null)
						{
							return refDisplayValue;
						}
						
						return stringValue;
					}
					
					lastSearchRow++;
					refDataRow = refDataBook.getDataRow(lastSearchRow);
				}
			}
			else if (refDataRow != null)
			{
				String displayValue = refDataRow.getValueAsString(displayReferencedColumnName);
				
				if (displayValue != null)
				{
					return displayValue;
				}
			}
		}

		return stringValue;
	}
	
}	// AbstractLinkedCellEditor
