/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 * 
 * 26.05.2009 - [JR] - creation
 * 04.10.2009 - [JR] - changePassword: old password is required
 * 13.10.2009 - [JR] - changePassword: memory/disk password problem when exception during save [BUGFIX]
 * 14.11.2009 - [JR] - #7
 *                     changePassword: validatePassword called   
 * 06.06.2010 - [JR] - #132: encryption support
 *                   - #133: more than one user [BUGFIX] 
 * 07.06.2010 - [JR] - #49: access control method implemented
 * 20.06.2010 - [JR] - refactoring: removed getDirectory from IConfiguration
 * 11.05.2011 - [JR] - release implemented  
 * 17.05.2015 - [TK] - #1389: JNDI support                 
 */
package com.sibvisions.rad.server.security;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;

import javax.naming.InitialContext;
import javax.rad.remote.IConnectionConstants;
import javax.rad.server.IConfiguration;
import javax.rad.server.ISession;

import com.sibvisions.rad.server.config.ApplicationZone;
import com.sibvisions.rad.server.config.Configuration;
import com.sibvisions.util.log.LoggerFactory;
import com.sibvisions.util.type.CommonUtil;
import com.sibvisions.util.type.ResourceUtil;
import com.sibvisions.util.xml.XmlNode;
import com.sibvisions.util.xml.XmlWorker;

/**
 * The <code>XmlSecurityManager</code> uses a xml file to authenticate users. It requires
 * the following information:
 * <ul>
 *   <li>usersfile (the file or JNDI resource with all users)</li>
 * </ul>
 * 
 * @author Ren Jahn
 */
public class XmlSecurityManager extends AbstractSecurityManager
{
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Class members
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    /** the current user list. */
    private XmlNode xmnUsers = null;
    
    /** the user file. */
    private File fiUsers = null;
    
    /** whether, the user file was loaded as virtual resource. */
    private boolean bVirtual = false;

    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Interface implementation
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * {@inheritDoc}
     */
    public synchronized void validateAuthentication(ISession pSession) throws Exception
    {
        //Exception will be thrown when the credentials are not valid!
        getAuthenticatedUserNode(pSession);
    }

    /**
     * {@inheritDoc}
     */
    public synchronized void changePassword(ISession pSession) throws Exception
    {
        if (bVirtual)
        {
            throw new SecurityException("Can't change password because the user file was loaded as virtual resource!");
        }
        
        String sOldPwd = (String)pSession.getProperty(IConnectionConstants.OLDPASSWORD);
        String sNewPwd = (String)pSession.getProperty(IConnectionConstants.NEWPASSWORD);

        IConfiguration cfgSession = pSession.getConfig();
        
        //-------------------------------------------------
        // Password validation
        //-------------------------------------------------
        
        validatePassword(pSession, sOldPwd, sNewPwd);
        
        //-------------------------------------------------
        // Change password
        //-------------------------------------------------

        XmlNode xmnUser = getAuthenticatedUserNode(pSession);
        
        //check old password, because the session password is identical to the users password!
        if (comparePassword(cfgSession, sOldPwd, xmnUser.getNode("/password").getValue()))
        {
            xmnUser.setNode("/password", getEncryptedPassword(cfgSession, sNewPwd));

            FileOutputStream fosUsers = null;
            
            try
            {
                fosUsers = new FileOutputStream(fiUsers);

                XmlWorker worker = new XmlWorker();
                
                worker.write(fosUsers, xmnUsers);
            }
            catch (Throwable th)
            {
                //ensure that the memory contains the correct password!
                xmnUser.setNode("/password", getEncryptedPassword(cfgSession, sOldPwd));
            }
            finally
            {
                if (fosUsers != null)
                {
                    try
                    {
                        fosUsers.close();
                    }
                    catch (Exception e)
                    {
                        //nothing to be done
                    }
                }
            }
        }
        else
        {
            throw new SecurityException("Invalid password for '" + pSession.getUserName() + "' and application '" + pSession.getApplicationName() + "'");
        }
    }
    
    /**
     * {@inheritDoc}
     */
    public synchronized void logout(ISession pSession)
    {
    }   
    
    /**
     * {@inheritDoc}
     */
    public synchronized IAccessController getAccessController(ISession pSession)
    {
        return null;
    }
    
    /**
     * {@inheritDoc}
     */
    public synchronized void release()
    {
    }
    
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // User-defined methods
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * Loads all users from the configured userfile.
     * 
     * @param pSession the accessing session
     * @throws Exception if the userfile can not be loaded or the configuration is invalid
     */
    private void loadUsers(ISession pSession) throws Exception
    {
        if (xmnUsers == null)
        {
            IConfiguration config = pSession.getConfig();
            
            String sFile = config.getProperty("/application/securitymanager/userfile");
            
            if (sFile == null)
            {
                throw new SecurityException("Parameter 'userfile' is missing for application '" + pSession.getApplicationName() + "'");
            }
            
            //first: try absolute path
            File fiUserList = new File(sFile);
            
            InputStream isUsers = null;

            if (!fiUserList.exists())
            {
                ApplicationZone zone = Configuration.getApplicationZone(pSession.getApplicationName());
                
                fiUserList = new File(zone.getDirectory(), sFile);
                    
                if (fiUserList.exists())
                {
                    isUsers = new FileInputStream(fiUserList);
                }
                else
                {
                    //try to load the user file as JNDI binding
                    try
                    {
                        InitialContext ctxt = new InitialContext();
                        
                        try
                        {
                            Object objInstance = ctxt.lookup(sFile);
                            
                            if (objInstance instanceof String)
                            {
                                isUsers = ResourceUtil.getResourceAsStream((String)objInstance);
                                
                                if (isUsers == null)
                                {
                                    try
                                    {
                                        isUsers = new URL((String) objInstance).openStream();
                                    }
                                    catch (MalformedURLException exc)
                                    {
                                        // do nothing
                                    }
                                }
                                
                                if (isUsers == null)
                                {
                                    isUsers = new ByteArrayInputStream(((String)objInstance).getBytes("UTF-8"));
                                    
                                    //maybe XML (no special node checks)
                                    XmlWorker.readNode(isUsers);
                                    
                                    isUsers.reset();
                                }
                            }
                            else if (objInstance instanceof URL)
                            {
                                isUsers = ((URL)objInstance).openStream();
                            }
                            else if (objInstance instanceof InputStream)
                            {
                                isUsers = (InputStream)objInstance;
                            }
                        }
                        finally
                        {
                            ctxt.close();
                        }
                    }
                    catch (Exception ex)
                    {
                        LoggerFactory.getInstance(Configuration.class).debug("Couldn't load user file '", sFile, "' via JNDI!", ex);
                    }
                    
                    //try to load the user file as resource
                    if (isUsers == null)
                    {
                        if (Configuration.isSearchClassPath())
                        {                        
                            isUsers = ResourceUtil.getResourceAsStream("/rad/apps/" + pSession.getApplicationName() + "/" + sFile);
                        }
                    }
                    
                    bVirtual = isUsers != null;
                }
            }
            else
            {
                isUsers = new FileInputStream(fiUserList);
            }
            
            if (isUsers == null)
            {
                throw new SecurityException("Userfile '" + sFile + "' doesn't exist!");
            }

            //read users from xml file
            
            XmlWorker worker = new XmlWorker();

            try
            {
                xmnUsers = worker.read(isUsers);
                
                fiUsers = fiUserList;
            }
            finally
            {
                try
                {
                    isUsers.close();
                }
                catch (Exception e)
                {
                    //nothing to be done
                }
            }
        }
    }
    
    /**
     * Gets the xml node from the usersfile with the username and password specified through the
     * session.
     * 
     * @param pSession the accessing session
     * @return the xml node for the session user
     * @throws Exception if the credentials for the user are invalid
     */
    private XmlNode getAuthenticatedUserNode(ISession pSession) throws Exception
    {
        String sUserName = pSession.getUserName();

        if (sUserName != null)
        {
            loadUsers(pSession);
        
            List<XmlNode> liUsers = xmnUsers.getNodes("/users/user");
        
            IConfiguration cfgSession = pSession.getConfig();
            
            for (XmlNode xmnUser : liUsers)
            {
                if (sUserName.equals(xmnUser.getNode("/name").getValue()))
                {
                    if (comparePassword(cfgSession, pSession.getPassword(), xmnUser.getNode("/password").getValue()))
                    {
                        return xmnUser;
                    }
                    else
                    {
                        throw new SecurityException("Invalid password for '" + pSession.getUserName() + "' and application '" + pSession.getApplicationName() + "'");
                    }
                }
            }
        }
        
        throw new SecurityException("User '" + CommonUtil.nvl(sUserName, "<undefined>") + "' was not found for application '" + pSession.getApplicationName() + "'");
    }

}   // XmlSecurityManager
