/*
 * Copyright 2014 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 * 
 * 22.11.2014 - [JR] - creation
 * 27.05.2015 - [JR] - #1397: ICallHandler support
 */
package com.sibvisions.rad.server;

import java.io.File;

import javax.rad.remote.IConnectionConstants;
import javax.rad.server.ICallHandler;
import javax.rad.server.IServer;
import javax.rad.server.ISession;
import javax.rad.server.ServerContext;
import javax.servlet.http.HttpServletRequest;

import com.sibvisions.rad.IPackageSetup;
import com.sibvisions.rad.server.http.HttpContext;
import com.sibvisions.util.log.LoggerFactory;
import com.sibvisions.util.type.StringUtil;

/**
 * The <code>ServerContextImpl</code> is an internal {@link javax.rad.server.ServerContext} implementation.
 * 
 * @author Ren Jahn
 */
public class ServerContextImpl extends ServerContext
{
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Class members
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    /** the server. */
    private Server server;
    
    /** the associated session. */
    private WrappedSession wsessCurrent;
    
    /** the system identifier. */
    private String sIdentifier;
    
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Initialization
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    /**
     * Creates a new instance of <code>ServerContextImpl</code>.
     * 
     * @param pServer the server
     */
    ServerContextImpl(Server pServer)
    {
        server = pServer;
        
        sIdentifier = createSystemIdentifier();
        
        setInstance(this);
    }

    /**
     * Creates a new instance of <code>ServerContextImpl</code> for a given
     * system.
     * 
     * @param pServer the server
     * @param pIdentifier the system identifier
     */
    ServerContextImpl(Server pServer, String pIdentifier)
    {
        server = pServer;
        sIdentifier = pIdentifier;
        setInstance(this);
    }
    
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Abstract methods implementation
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * {@inheritDoc}
     */
    @Override
    protected void destroy()
    {
        setInstance(null);
    }
    
    /**
     * {@inheritDoc}
     */
    @Override
    public ISession getSession()
    {
        return wsessCurrent;
    }
    
    /**
     * {@inheritDoc}
     */
    @Override
    public String getSystemIdentifier()
    {
        return sIdentifier;
    }

    /**
     * {@inheritDoc}
     */
    public String getServerIdentifier()
    {
        return server.getInstanceKey();
    }
    
    /**
     * {@inheritDoc}
     */
    public ICallHandler getCallHandler()
    {
        if (wsessCurrent != null)
        {
            AbstractSession session = wsessCurrent.session;
         
            //ServerContext always returns the callhandler of the master session
            if (session instanceof SubSession)
            {
                return ((SubSession)session).getMasterSession().getCallHandler();
            }
            else
            {
                return wsessCurrent.session.getCallHandler();
            }
        }
        
        return null;
    }
    
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // User-defined methods
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * Sets the session.
     * 
     * @param pSession the session
     */
    void setSession(ISession pSession)
    {
        if (pSession == null)
        {
            wsessCurrent = null;
        }
        else
        {
            if (pSession instanceof WrappedSession)
            {
                wsessCurrent = (WrappedSession)pSession;
            }
            else
            {
                wsessCurrent = new WrappedSession((AbstractSession)pSession);
            }
            
            if (StringUtil.isEmpty(sIdentifier))
            {
                //don't use getProperty, because it changes the last access time and this means that it won't timeout
                Object obj = wsessCurrent.getProperties().get(IConnectionConstants.PREFIX_SERVER + IConnectionConstants.PREFIX_REQUEST + "systemIdentifier");
                
                if (obj != null)
                {
                    sIdentifier = obj.toString();
                }
            }
        }
    }
    
    /**
     * Gets the server.
     * 
     * @return the server
     */
    public IServer getServer()
    {
        return server;
    }

    /**
     * Creates the system identifier.
     * 
     * @return the identifier
     */
    private String createSystemIdentifier()
    {
        //use configured system identifier, if available
        String sTempIdent = System.getProperty(IPackageSetup.SERVER_SYSTEMIDENTIFIER);
        
        if (!StringUtil.isEmpty(sTempIdent))
        {
            LoggerFactory.getInstance(ServerContext.class).debug("Using temporary server identifier: ", sTempIdent);

            return sTempIdent;
        }

        //detect system identifier
        
        String sIdent = "";
        
        HttpContext ctxt = HttpContext.getCurrentInstance();
        
        if (ctxt != null)
        {
            Object oRequest = ctxt.getRequest();
            
            if (oRequest instanceof HttpServletRequest)
            {
                sIdent = ((HttpServletRequest)oRequest).getRequestURL().toString();
            }
            else
            {
                //e.g. PortletRequest 
            }
        }
        else
        {
            //no http context available -> check working directory
            
            File fiDir = new File("");
            
            sIdent = fiDir.getAbsolutePath();
        }

        LoggerFactory.getInstance(ServerContext.class).debug("Using server identifier: ", sIdent);
        
        return sIdent;
    }
    
}   // ServerContextImpl
