/*
 * Copyright 2017 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 01.08.2017 - [JR] - creation
 */
package com.sibvisions.rad.persist.jdbc;

import java.sql.CallableStatement;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import javax.rad.model.condition.ICondition;
import javax.rad.persist.DataSourceException;

import com.sibvisions.rad.persist.jdbc.param.AbstractParam;
import com.sibvisions.rad.persist.jdbc.param.AbstractParam.ParameterType;
import com.sibvisions.rad.persist.jdbc.param.OutParam;
import com.sibvisions.rad.server.protocol.ICategoryConstants;
import com.sibvisions.rad.server.protocol.ICommandConstants;
import com.sibvisions.rad.server.protocol.ProtocolFactory;
import com.sibvisions.rad.server.protocol.Record;
import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.log.ILogger.LogLevel;
import com.sibvisions.util.type.CommonUtil;
/**
 * The <code>AbstractOracleDBAccess</code> is the base implementation for Oracle databases.<br>
 *  
 * @see com.sibvisions.rad.persist.jdbc.DBAccess
 * 
 * @author Roland Hrmann
 */
public abstract class AbstractOracleDBAccess extends DBAccess
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Constants
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	// The ultimate Oracle Meta Data Optimization!
	
	/** The select statement to get the Unique keys in Oracle. */
	private static String sUKSelect = "SELECT c.constraint_name " +
									    "FROM all_constraints c " +
									   "WHERE c.constraint_type = 'U' " +
	                                     "AND c.table_name = ? " + 
	                                     "AND c.owner = ?";
			
	/** The select statement to get the columns from an constraint in Oracle. */
	private static String sConstraintColumnsSelect = "SELECT c.column_name " +
	                                                       ",c.table_name " +
													   "FROM all_cons_columns c " +
													  "WHERE c.owner = ? " +
													    "AND c.constraint_name = ? " +
													  "ORDER BY POSITION"; 
	
	/** The select statement to get the Primary key in Oracle. */
	private static String sPKSelect = "SELECT c.constraint_name " +
									    "FROM all_constraints c " +
									   "WHERE c.constraint_type = 'P' " +
									     "AND c.table_name = ? " +
										 "AND c.owner = ?"; 
	
	/** The select statement to get the Foreign keys in Oracle. */
	private static String sFKSelect =    "SELECT c.constraint_name fk_name " +
	                                           ",c.r_owner pktable_schem " + 
											   ",c.r_constraint_name pk_name " +
			                               "FROM all_constraints c " +
			                              "WHERE c.constraint_type = 'R' " +
			                                "AND c.table_name = ? " +
			                                "AND c.owner = ?";
    
	/** the select statement to get the Check constraints in Oracle. */
	private static String sCheckSelect = "select c.search_condition " +
	                                       "from all_constraints c " +
	                                      "where c.constraint_type = 'C' " +
	                                        "and c.generated = 'USER NAME' " +
	                                        "and c.table_name = ? " +
	                                        "and c.owner = ?";
	
	/** the select statement to get the Check constraints in Oracle. */
	private static String sDefaultValueSelect = "select c.column_name, c.data_type, c.data_default " +
	                                       "from all_tab_columns c " +
	                                      "where c.data_default is not null " +
	                                        "and c.table_name = ? " +
	                                        "and c.owner = ?";
    
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Abstract methods
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * Converts arrays to {@link List} of {@link javax.rad.type.bean.IBean}.
     * @param pParam the param to check
     * @return the param or a list in case of array.
     * @throws SQLException the exception
     */
    protected abstract Object convertArrayToList(Object pParam) throws SQLException;
    
    /**
     * Converts list or array to oracle arrays.
     * @param pParam the param to check
     * @return the param or a list in case of array.
     * @throws SQLException the exception
     */
    protected abstract Object convertToArray(AbstractParam pParam) throws SQLException;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
			
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDatabaseSpecificLockStatement(String pWritebackTable, ServerMetaData pServerMetaData, ICondition pPKFilter) throws DataSourceException
	{
		return super.getDatabaseSpecificLockStatement(pWritebackTable, pServerMetaData, pPKFilter) + " NOWAIT";											
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object[] insertDatabaseSpecific(String pWriteBackTable, String pInsertStatement, ServerMetaData pServerMetaData, 
                                           Object[] pNewDataRow, String pDummyColumn)  throws DataSourceException
    {
		return insertOracle(pWriteBackTable, pInsertStatement, pServerMetaData, pNewDataRow, pDummyColumn);
    }
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void open() throws DataSourceException
	{
		super.open();

		Statement stmt = null;
		
		try
		{
			stmt = getConnectionIntern().createStatement();
			
			stmt.executeUpdate("ALTER SESSION SET NLS_COMP='BINARY'");
			stmt.executeUpdate("ALTER SESSION SET NLS_SORT='BINARY'");
		}
		catch (SQLException ex)
		{
			// Try silent to change nls_sort, nls_comp
		}
		finally
		{
		    CommonUtil.close(stmt);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<Key> getUniqueKeysIntern(String pCatalog, 
        					  			    String pSchema, 
        					  			    String pTable) throws DataSourceException
	{
		ResultSet 		  rsResultSet = null;
		PreparedStatement psResult = null;
		ResultSet 		  rsResultSetColumns = null;
		PreparedStatement psResultColumns = null;

		try
		{	
			ArrayUtil<Key>  auResult           = new ArrayUtil<Key>();
			ArrayUtil<Name> auUniqueKeyColumns = new ArrayUtil<Name>();
			
			long lMillis = System.currentTimeMillis();
			
			psResult = getPreparedStatement(sUKSelect);
			psResultColumns = getPreparedStatement(sConstraintColumnsSelect);

			psResult.setString(1, removeQuotes(pTable));
			psResult.setString(2, removeQuotes(pSchema));
			
			rsResultSet = psResult.executeQuery();

			while (rsResultSet.next())
			{
				String sUKName = rsResultSet.getString("CONSTRAINT_NAME");
				
				psResultColumns.setString(1, removeQuotes(pSchema));
				psResultColumns.setString(2, sUKName);
				
				rsResultSetColumns = psResultColumns.executeQuery();
					
				while (rsResultSetColumns.next())
				{
					auUniqueKeyColumns.add(new Name(rsResultSetColumns.getString("COLUMN_NAME"), 
							                        quote(rsResultSetColumns.getString("COLUMN_NAME"))));
				}
				
				CommonUtil.close(rsResultSetColumns);

				Key uk = new Key(sUKName, auUniqueKeyColumns.toArray(new Name[auUniqueKeyColumns.size()]));
				auResult.add(uk);
				auUniqueKeyColumns.clear();
			}
			
            if (isLogEnabled(LogLevel.DEBUG))
            {
                debug("getUKs(", pTable, ") in ", Long.valueOf((System.currentTimeMillis() - lMillis)), "ms");
            }
			
			return auResult;
		}
		catch (SQLException sqlException)
		{
			throw new DataSourceException("Unique Keys couldn't determined from database! - " + pTable, formatSQLException(sqlException));
		}		
		finally
		{
		    CommonUtil.close(rsResultSet, psResult, rsResultSetColumns, psResultColumns);
		}			
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Key getPrimaryKeyIntern(String pCatalog, String pSchema, String pTable) throws DataSourceException
	{
		ResultSet rsResultSet = null;
		PreparedStatement psResult = null;
		try
		{	
			Key result = null;
			ArrayUtil<Name> auPrimaryKeyColumns = new ArrayUtil<Name>();
			
			long lMillis  = System.currentTimeMillis();
			psResult = getPreparedStatement(sPKSelect);
			psResult.setString(1, removeQuotes(pTable));
			psResult.setString(2, removeQuotes(pSchema));
			rsResultSet = psResult.executeQuery();
			
			if (rsResultSet.next())
			{
				String sPKName = rsResultSet.getString("CONSTRAINT_NAME");
				
				CommonUtil.close(rsResultSet, psResult);
				
				psResult = getPreparedStatement(sConstraintColumnsSelect);
				psResult.setString(1, removeQuotes(pSchema));
				psResult.setString(2, sPKName);
				rsResultSet = psResult.executeQuery();
				
				if (rsResultSet.next())
				{				
					do
					{
						auPrimaryKeyColumns.add(new Name(rsResultSet.getString("COLUMN_NAME"), quote(rsResultSet.getString("COLUMN_NAME"))));
					}
					while (rsResultSet.next());
	
					if (auPrimaryKeyColumns.size() > 0)
					{
						result = new Key(sPKName, auPrimaryKeyColumns.toArray(new Name[auPrimaryKeyColumns.size()]));
					}
				}
			}
			
            if (isLogEnabled(LogLevel.DEBUG))
            {
                debug("getPK(", pTable, ") in ", Long.valueOf((System.currentTimeMillis() - lMillis)), "ms");
            }
			
			return result;
		}
		catch (SQLException sqlException)
		{
			throw new DataSourceException("Primary Key couldn't determined from database! - " + pTable, formatSQLException(sqlException));
		}		
		finally
		{
		    CommonUtil.close(rsResultSet, psResult);
		}			
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<ForeignKey> getForeignKeysIntern(String pCatalog, String pSchema, String pTable) throws DataSourceException
	{
		PreparedStatement psFKs = null;
		ResultSet         rsFKs = null;
		ResultSet 		  rsResultSetColumns = null;
		PreparedStatement psResultColumns = null;
		try
		{
			ArrayUtil<ForeignKey> auForeignKeys = new ArrayUtil<ForeignKey>();

			long lMillis = System.currentTimeMillis();

			String sCatalog = getConnectionIntern().getCatalog();
			
			psFKs = getPreparedStatement(sFKSelect);
			psResultColumns = getPreparedStatement(sConstraintColumnsSelect);
			
			psFKs.setString(1, removeQuotes(pTable));
			psFKs.setString(2, removeQuotes(pSchema));

			rsFKs = psFKs.executeQuery();			
			while (rsFKs.next())
			{
				String sFKName  = rsFKs.getString("FK_NAME");
				String sPKName  = rsFKs.getString("PK_NAME");
				String sPKTableSchema = rsFKs.getString("PKTABLE_SCHEM");
				
				String sPKTable = null;
				
				ArrayUtil<Name> auPKColumns = new ArrayUtil<Name>();
				ArrayUtil<Name> auFKColumns = new ArrayUtil<Name>();

				psResultColumns.setString(1, removeQuotes(pSchema));
				psResultColumns.setString(2, sFKName);
				rsResultSetColumns = psResultColumns.executeQuery();
				
				while (rsResultSetColumns.next())
				{
					auFKColumns.add(new Name(rsResultSetColumns.getString("COLUMN_NAME"), 
							                 quote(rsResultSetColumns.getString("COLUMN_NAME"))));
				}
				
				CommonUtil.close(rsResultSetColumns);
				
				psResultColumns.setString(1, sPKTableSchema);
				psResultColumns.setString(2, sPKName);				
				rsResultSetColumns = psResultColumns.executeQuery();
				
				while (rsResultSetColumns.next())
				{
					if (sPKTable ==  null)
					{
						sPKTable = rsResultSetColumns.getString("TABLE_NAME");
					}
					auPKColumns.add(new Name(rsResultSetColumns.getString("COLUMN_NAME"), 
							                 quote(rsResultSetColumns.getString("COLUMN_NAME"))));
				}
				
				CommonUtil.close(rsResultSetColumns);

				if (sPKTable != null)
				{
					ForeignKey fkForeignKey = new ForeignKey(
							new Name(sPKTable, quote(sPKTable)), 
							new Name(sCatalog, quote(sCatalog)), 
							new Name(sPKTableSchema, quote(sPKTableSchema)));
	
					fkForeignKey.setFKName(sFKName);
					fkForeignKey.setFKColumns(auFKColumns.toArray(new Name[auFKColumns.size()]));
					fkForeignKey.setPKColumns(auPKColumns.toArray(new Name[auPKColumns.size()]));
					auForeignKeys.add(fkForeignKey);
				}

				auPKColumns.clear();
				auFKColumns.clear();				
			}

            if (isLogEnabled(LogLevel.DEBUG))
            {
                debug("getFKs(", pTable, ") in ", Long.valueOf(System.currentTimeMillis() - lMillis), "ms");
            }

			return auForeignKeys;
		}
		catch (SQLException sqlException)
		{
			throw new DataSourceException("Foreign Keys couldn't determined from database! - " + pTable, formatSQLException(sqlException));
		}
		finally
		{
		    CommonUtil.close(rsFKs, psFKs, rsResultSetColumns, psResultColumns);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Map<String, Object> getDefaultValuesIntern(String pCatalog, String pSchema, String pTable) throws DataSourceException
	{
		PreparedStatement psDefaultValues = null;
		
		ResultSet resDefaultValues = null;
		
		Hashtable<String, Object> htDefaultValues = null;
		
		try
		{
			long lMillis = System.currentTimeMillis();

			psDefaultValues = getPreparedStatement(sDefaultValueSelect);
			psDefaultValues.setString(1, removeQuotes(pTable));
			psDefaultValues.setString(2, removeQuotes(pSchema));
			
			resDefaultValues = psDefaultValues.executeQuery();
			
			//detect all possible values
			
			while (resDefaultValues.next())
			{
				if (htDefaultValues == null)
				{
					htDefaultValues = new Hashtable<String, Object>();
				}
				String columnName = resDefaultValues.getString(1);
				String dataType = resDefaultValues.getString(2);
				int type = Types.VARCHAR;
				if (dataType.contains("DATE") || dataType.contains("TIME") || dataType.contains("INTERVAL"))
				{
					type = Types.DATE;
				}
				else if (dataType.contains("NUMBER") || dataType.contains("FLOAT") || dataType.contains("INTEGER"))
				{
					type = Types.DECIMAL;
				}
				String value = resDefaultValues.getString(3);
					
				try
				{
					Object objValue = translateDefaultValue(columnName, type, value.trim());

					if (objValue != null)
					{
						htDefaultValues.put(columnName, objValue);
					}
				}
				catch (Exception e)
				{
					//no default value
					//debug(value, e);
				}
			}			
			
			if (isLogEnabled(LogLevel.DEBUG))
            {
                debug("getDefaultValuesIntern(", pTable, ") in ", Long.valueOf((System.currentTimeMillis() - lMillis)), "ms");
            }
			
			return htDefaultValues;
		}
		catch (SQLException sqle)
		{
			throw new DataSourceException("Can't access check constraints for: '" + pTable + "'", formatSQLException(sqle));
		}
		finally
		{
		    CommonUtil.close(resDefaultValues, psDefaultValues);
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Map<String, Object[]> getAllowedValuesIntern(String pCatalog, String pSchema, String pTable) throws DataSourceException
	{
		PreparedStatement psCheck = null;
		
		ResultSet resCheck = null;
		
		Hashtable<String, Object[]> htAllowed = null;
		
		Hashtable<String, List<String>> htFoundValues = null;
		
		try
		{
			long lMillis = System.currentTimeMillis();

			psCheck = getPreparedStatement(sCheckSelect);
			psCheck.setString(1, removeQuotes(pTable));
			psCheck.setString(2, removeQuotes(pSchema));
			
			resCheck = psCheck.executeQuery();
			
			//detect all possible values
			
			while (resCheck.next())
			{
				htFoundValues = CheckConstraintSupport.parseCondition(resCheck.getString(1), htFoundValues, true);
			}			
			
			//interpret values
			
			htAllowed = CheckConstraintSupport.translateValues(this, pCatalog, pSchema, pTable, htFoundValues);

			if (isLogEnabled(LogLevel.DEBUG))
            {
                debug("getAllowedValuesIntern(", pTable, ") in ", Long.valueOf((System.currentTimeMillis() - lMillis)), "ms");
            }
			
			return htAllowed;
		}
		catch (SQLException sqle)
		{
			throw new DataSourceException("Can't access check constraints for: '" + pTable + "'", formatSQLException(sqle));
		}
		finally
		{
		    CommonUtil.close(resCheck, psCheck);
		}
	}
  
    /** 
     * {@inheritDoc}
     */
    @Override
    protected TableInfo getTableInfoIntern(String pWriteBackTable) throws DataSourceException
    {   
        TableInfo tableInfo = super.getTableInfoIntern(pWriteBackTable);

        String schema = tableInfo.getSchema();
        if (schema == null)
        {
            schema = getUsername().toUpperCase();
        }
        String table = tableInfo.getTable();
        if (table != null && !table.startsWith(QUOTE) && !table.endsWith(QUOTE))
        {
            table = table.toUpperCase();
        }
        
        return new TableInfo(tableInfo.getCatalog(), schema, table);
    }
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected boolean detectModified()
	{
		try 
		{
			return executeSql("select dbms_transaction.local_transaction_id from dual").get(0) != null;
		}
		catch (SQLException e)
		{
			return super.detectModified();
		}
	}
	
    /**
     * Executes a DB function with the specified parameters and return the result.
     * 
     * @param pFunctionName the function (optional with package) name. 
     * @param pReturnOutParam the return SQL Type (see {@link Types}
     * @param pParameters the parameters to use with the correct and corresponding java type.
     * @return the result of the DB function call.
     * @throws SQLException if the call failed.
     */
    @Override
    public Object executeFunction(String pFunctionName, OutParam pReturnOutParam, Object...pParameters) throws SQLException
    {
        Record record = ProtocolFactory.openRecord(ICategoryConstants.DATABASE, ICommandConstants.DB_EXEC_FUNCTION);
        
        try
        {
            if (record != null)
            {
                if (pParameters != null && pParameters.length > 0)
                {
                    record.setParameter(pFunctionName, pParameters);
                }
                else
                {
                    record.setParameter(pFunctionName);
                }
            }
            
            StringBuilder sqlDeclare = new StringBuilder();
            StringBuilder sqlStatement = new StringBuilder("begin ? := ");
            StringBuilder sqlRetrieve = new StringBuilder();
            
            if (pReturnOutParam.getSqlType() == Types.BOOLEAN)
            {
                sqlStatement.append("sys.diutil.bool_to_int(");
            }
            sqlStatement.append(pFunctionName);
            int[] outBoolIndexes = new int[0];
            
            if (pParameters != null && pParameters.length > 0)
            {
                sqlStatement.append("(");
                for (int i = 0; i < pParameters.length; i++)
                {
                    if (i > 0)
                    {
                        sqlStatement.append(", ");          
                    }
                    if (pParameters[i] instanceof AbstractParam)
                    {
                        AbstractParam apParam = (AbstractParam)pParameters[i];
                        
                        ParameterType type = apParam.getType();
                        if ((type == ParameterType.Out || type == ParameterType.InOut)
                                && apParam.getSqlType() == Types.BOOLEAN)
                        {
                            if (sqlDeclare.length() == 0)
                            {
                                sqlDeclare.append("declare ");
                            }
                            String boolName = "bool" + outBoolIndexes.length;

                            sqlStatement.append(boolName);
                            sqlDeclare.append(boolName).append(" boolean := sys.diutil.int_to_bool(?); ");
                            sqlRetrieve.append("? := sys.diutil.bool_to_int(").append(boolName).append(");");
                            outBoolIndexes = ArrayUtil.add(outBoolIndexes, i);
                        }
                        else if (apParam.getSqlType() == Types.BOOLEAN || apParam.getValue() instanceof Boolean)
                        {
                            sqlStatement.append("sys.diutil.int_to_bool(?)");
                        }
                        else
                        {
                            sqlStatement.append("?");
                        }
                    }
                    else if (pParameters[i] instanceof Boolean)
                    {
                        sqlStatement.append("sys.diutil.int_to_bool(?)");
                    }
                    else
                    {
                        sqlStatement.append("?");
                    }
                }
                sqlStatement.append(")");
            }
            if (pReturnOutParam.getSqlType() == Types.BOOLEAN)
            {
                sqlStatement.append(")");
            }
            sqlStatement.append("; ");
            
            sqlStatement.insert(0, sqlDeclare);
            sqlStatement.append(sqlRetrieve);
            sqlStatement.append(" end;");
            
            CallableStatement call = null;
            
            try
            {
                debug("executeFunction -> ", sqlStatement);

                call = getConnection().prepareCall(translateQuotes(sqlStatement.toString()));
                if (pReturnOutParam.getSqlType() == Types.BOOLEAN)
                {
                    call.registerOutParameter(outBoolIndexes.length + 1, Types.INTEGER);
                }
                else
                {
                    if (pReturnOutParam.getTypeName() != null)
                    {
                        call.registerOutParameter(outBoolIndexes.length + 1, pReturnOutParam.getSqlType(), pReturnOutParam.getTypeName());
                    }
                    else
                    {
                        call.registerOutParameter(outBoolIndexes.length + 1, pReturnOutParam.getSqlType());
                    }
                }
                
                AbstractParam apParam;
                
                ParameterType type;
                
                int boolIndex = 0;
                for (int i = 0; pParameters != null && i < pParameters.length; i++)
                {
                    int index = i + outBoolIndexes.length - boolIndex + 2; 
                    if (boolIndex < outBoolIndexes.length && outBoolIndexes[boolIndex] == i)
                    {
                        call.registerOutParameter(pParameters.length + boolIndex + 2, Types.INTEGER);
                        Object val = ((AbstractParam)pParameters[i]).getValue();
                        if (val == null)
                        {
                            call.setNull(boolIndex + 1, Types.INTEGER);
                        }
                        else
                        {
                            int bool = 0;
                            if ((val instanceof Boolean && ((Boolean)val).booleanValue())
                                    || (val instanceof Number && ((Number)val).intValue() != 0)
                                    || (val instanceof String && Boolean.valueOf((String)val).booleanValue()))
                            {
                                bool = 1;
                            }
                                    
                            call.setObject(boolIndex + 1, Integer.valueOf(bool));
                        }

                        boolIndex++;
                    }
                    else if (pParameters[i] == null)
                    {
                        call.setNull(index, Types.VARCHAR);
                    }
                    else
                    {
                        if (pParameters[i] instanceof AbstractParam)
                        {
                            apParam = (AbstractParam)pParameters[i];
    
                            type = apParam.getType();
                            
                            if (type == ParameterType.Out || type == ParameterType.InOut)
                            {
                                if (apParam.getTypeName() != null)
                                {
                                    call.registerOutParameter(index, apParam.getSqlType(), apParam.getTypeName());
                                }
                                else
                                {
                                    call.registerOutParameter(index, apParam.getSqlType());
                                }
                            }
                            
                            if (apParam.getValue() == null)
                            {
                                if (apParam.getTypeName() != null)
                                {
                                    call.setNull(index, apParam.getSqlType(), apParam.getTypeName());
                                }
                                else if (apParam.getSqlType() == Types.BOOLEAN)
                                {
                                    call.setNull(index, Types.INTEGER);
                                }
                                else
                                {
                                    call.setNull(index, apParam.getSqlType());
                                }
                            }
                            else if (apParam.getSqlType() == Types.BOOLEAN || apParam.getValue() instanceof Boolean)
                            {
                                Object val = apParam.getValue();

                                int bool = 0;
                                if ((val instanceof Boolean && ((Boolean)val).booleanValue())
                                        || (val instanceof Number && ((Number)val).intValue() != 0)
                                        || (val instanceof String && Boolean.valueOf((String)val).booleanValue()))
                                {
                                    bool = 1;
                                }

                                call.setObject(index, Integer.valueOf(bool), Types.INTEGER);
                            }
                            else
                            {
                                call.setObject(index, convertValueToDatabaseSpecificObject(convertToArray(apParam)), apParam.getSqlType());
                            }
                        }
                        else if (pParameters[i] instanceof Boolean)
                        {
                            call.setObject(index, ((Boolean)pParameters[i]).booleanValue() ? Integer.valueOf(1) : Integer.valueOf(0));
                        }
                        else
                        {
                            call.setObject(index, convertValueToDatabaseSpecificObject(pParameters[i]));
                        }
                    }
                }

                if (call.execute())
                {
                    CommonUtil.close(call.getResultSet());
                }
                
                Object oResult = call.getObject(outBoolIndexes.length + 1);
                
                if (pReturnOutParam.getSqlType() == Types.BOOLEAN && oResult != null)
                {
                    oResult = Boolean.valueOf(((Number)oResult).intValue() != 0);
                }
                
                boolIndex = 0;
                for (int i = 0; pParameters != null && i < pParameters.length; i++)
                {
                    int index = i + outBoolIndexes.length - boolIndex + 2; 
                    if (boolIndex < outBoolIndexes.length && outBoolIndexes[boolIndex] == i)
                    {
                        apParam = (AbstractParam)pParameters[i];

                        Object val = call.getObject(pParameters.length + boolIndex + 2);
                        
                        if (val instanceof Number)
                        {
                            apParam.setValue(Boolean.valueOf(((Number)val).intValue() != 0));
                        }
                        else
                        {
                            apParam.setValue(null);
                        }

                        boolIndex++;
                    }
                    else if (pParameters[i] instanceof AbstractParam)
                    {
                        apParam = (AbstractParam)pParameters[i];
                        
                        type = apParam.getType();
                        
                        if (type == ParameterType.Out || type == ParameterType.InOut)
                        {
                            apParam.setValue(convertArrayToList(call.getObject(index)));
                        }
                    }
                }
                
                pReturnOutParam.setValue(convertArrayToList(oResult));
                
                return pReturnOutParam.getValue();
            }
            finally 
            {
                CommonUtil.close(call);
            }
        }
        finally
        {
            CommonUtil.close(record);
        }
    }
    
    /**
     * {@inheritDoc}
     */
    @Override
    public void executeProcedure(String pProcedureName, Object...pParameters) throws SQLException
    {
        Record record = ProtocolFactory.openRecord(ICategoryConstants.DATABASE,  ICommandConstants.DB_EXEC_PROCEDURE);
        
        try
        {
            if (record != null)
            {
                if (pParameters != null && pParameters.length > 0)
                {
                    record.setParameter(pProcedureName, pParameters);
                }
                else
                {
                    record.setParameter(pProcedureName);
                }
            }
            
            StringBuilder sqlDeclare = new StringBuilder();
            StringBuilder sqlStatement = new StringBuilder("begin ");
            StringBuilder sqlRetrieve = new StringBuilder(); 
            sqlStatement.append(pProcedureName);
            int[] outBoolIndexes = new int[0];
            
            if (pParameters != null && pParameters.length > 0)
            {
                sqlStatement.append("(");
                for (int i = 0; i < pParameters.length; i++)
                {
                    if (i > 0)
                    {
                        sqlStatement.append(", ");          
                    }
                    if (pParameters[i] instanceof AbstractParam)
                    {
                        AbstractParam apParam = (AbstractParam)pParameters[i];
                        
                        ParameterType type = apParam.getType();
                        if ((type == ParameterType.Out || type == ParameterType.InOut)
                                && apParam.getSqlType() == Types.BOOLEAN)
                        {
                            if (sqlDeclare.length() == 0)
                            {
                                sqlDeclare.append("declare ");
                            }
                            String boolName = "bool" + outBoolIndexes.length;

                            sqlStatement.append(boolName);
                            sqlDeclare.append(boolName).append(" boolean := sys.diutil.int_to_bool(?); ");
                            sqlRetrieve.append("? := sys.diutil.bool_to_int(").append(boolName).append(");");
                            outBoolIndexes = ArrayUtil.add(outBoolIndexes, i);
                        }
                        else if (apParam.getSqlType() == Types.BOOLEAN || apParam.getValue() instanceof Boolean)
                        {
                            sqlStatement.append("sys.diutil.int_to_bool(?)");
                        }
                        else
                        {
                            sqlStatement.append("?");
                        }
                    }
                    else if (pParameters[i] instanceof Boolean)
                    {
                        sqlStatement.append("sys.diutil.int_to_bool(?)");
                    }
                    else
                    {
                        sqlStatement.append("?");
                    }
                }
                sqlStatement.append(")");
            }
            sqlStatement.append("; ");

            sqlStatement.insert(0, sqlDeclare);
            sqlStatement.append(sqlRetrieve);
            sqlStatement.append(" end;");
            
            CallableStatement call = null;
            
            try
            {
                debug("executeProcedure -> ", sqlStatement);
                
                AbstractParam apParam;
                
                ParameterType type;
    
                call = getConnection().prepareCall(translateQuotes(sqlStatement.toString()));
                
                int boolIndex = 0;
                for (int i = 0; pParameters != null && i < pParameters.length; i++)
                {
                    int index = i + outBoolIndexes.length - boolIndex + 1; 
                    if (boolIndex < outBoolIndexes.length && outBoolIndexes[boolIndex] == i)
                    {
                        call.registerOutParameter(pParameters.length + boolIndex + 1, Types.INTEGER);
                        Object val = ((AbstractParam)pParameters[i]).getValue();
                        if (val == null)
                        {
                            call.setNull(boolIndex + 1, Types.INTEGER);
                        }
                        else
                        {
                            int bool = 0;
                            if ((val instanceof Boolean && ((Boolean)val).booleanValue())
                                    || (val instanceof Number && ((Number)val).intValue() != 0)
                                    || (val instanceof String && Boolean.valueOf((String)val).booleanValue()))
                            {
                                bool = 1;
                            }
                                    
                            call.setObject(boolIndex + 1, Integer.valueOf(bool));
                        }

                        boolIndex++;
                    }
                    else if (pParameters[i] == null)
                    {
                        call.setNull(index, Types.VARCHAR);
                    }
                    else
                    {
                        if (pParameters[i] instanceof AbstractParam)
                        {
                            apParam = (AbstractParam)pParameters[i];
    
                            type = apParam.getType();
                            
                            if (type == ParameterType.Out || type == ParameterType.InOut)
                            {
                                if (apParam.getTypeName() != null)
                                {
                                    call.registerOutParameter(index, apParam.getSqlType(), apParam.getTypeName());
                                }
                                else
                                {
                                    call.registerOutParameter(index, apParam.getSqlType());
                                }
                            }
                            
                            if (apParam.getValue() == null)
                            {
                                if (apParam.getTypeName() != null)
                                {
                                    call.setNull(index, apParam.getSqlType(), apParam.getTypeName());
                                }
                                else if (apParam.getSqlType() == Types.BOOLEAN)
                                {
                                    call.setNull(index, Types.INTEGER);
                                }
                                else
                                {
                                    call.setNull(index, apParam.getSqlType());
                                }
                            }
                            else if (apParam.getSqlType() == Types.BOOLEAN || apParam.getValue() instanceof Boolean)
                            {
                                Object val = apParam.getValue();

                                int bool = 0;
                                if ((val instanceof Boolean && ((Boolean)val).booleanValue())
                                        || (val instanceof Number && ((Number)val).intValue() != 0)
                                        || (val instanceof String && Boolean.valueOf((String)val).booleanValue()))
                                {
                                    bool = 1;
                                }

                                call.setObject(index, Integer.valueOf(bool), Types.INTEGER);
                            }
                            else
                            {
                                call.setObject(index, convertValueToDatabaseSpecificObject(convertToArray(apParam)), apParam.getSqlType());
                            }
                        }
                        else if (pParameters[i] instanceof Boolean)
                        {
                            call.setObject(index, ((Boolean)pParameters[i]).booleanValue() ? Integer.valueOf(1) : Integer.valueOf(0));
                        }
                        else
                        {
                            call.setObject(index, convertValueToDatabaseSpecificObject(pParameters[i]));
                        }
                    }
                }
                
                call.execute();
                
                boolIndex = 0;
                for (int i = 0; pParameters != null && i < pParameters.length; i++)
                {
                    int index = i + outBoolIndexes.length - boolIndex + 1; 
                    if (boolIndex < outBoolIndexes.length && outBoolIndexes[boolIndex] == i)
                    {
                        apParam = (AbstractParam)pParameters[i];

                        Object val = call.getObject(pParameters.length + boolIndex + 1);
                        
                        if (val instanceof Number)
                        {
                            apParam.setValue(Boolean.valueOf(((Number)val).intValue() != 0));
                        }
                        else
                        {
                            apParam.setValue(null);
                        }

                        boolIndex++;
                    }
                    else if (pParameters[i] instanceof AbstractParam)
                    {
                        apParam = (AbstractParam)pParameters[i];
                        
                        type = apParam.getType();
                        
                        if (type == ParameterType.Out || type == ParameterType.InOut)
                        {
                            apParam.setValue(convertArrayToList(call.getObject(index)));
                        }
                    }
                }
            }
            finally 
            {
                CommonUtil.close(call);
            }
        }
        finally
        {
            CommonUtil.close(record);
        }
    }   

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Returns the newly inserted row from an Oracle Database. <br>
	 * It uses RETURNING .. INTO to get the primary key values back from the database. 
	 * 
	 * @param pTablename		the table to use for the insert
	 * @param pInsertStatement	the SQL Statement to use for the insert
	 * @param pServerMetaData	the meta data to use.
	 * @param pNewDataRow		the new IDataRow with the values to insert
	 * @param pDummyColumn		true, if all writeable columns are null, but for a correct INSERT it have
	 *                          to be minimum one column to use in the syntax.
	 * @return the newly inserted row from an Oracle Database.
	 * @throws DataSourceException
	 *             if an <code>Exception</code> occur during insert the <code>IDataRow</code> 
	 *             to the storage
	 */	
	protected Object[] insertOracle(String pTablename, String pInsertStatement, ServerMetaData pServerMetaData, 
			                        Object[] pNewDataRow, String pDummyColumn)  throws DataSourceException
	{
		int[] pPKColumnIndices = pServerMetaData.getPrimaryKeyColumnIndices();

		boolean returnPK = false;
		
		if (pPKColumnIndices != null)
		{
			// If there is at least one empty (null) PK column, we will return
			// all PK values.
			for (int i = 0; i < pPKColumnIndices.length && !returnPK; i++)
			{
				returnPK = (pNewDataRow[pPKColumnIndices[i]] == null);
			}
		}
	
		if (returnPK)
		{
			StringBuffer sInsertStatement = new StringBuffer("BEGIN " + pInsertStatement);
			
			// use RETURNING to get all PK column values filled in in from the trigger
			sInsertStatement.append(" RETURNING ");
			
			for (int i = 0; pPKColumnIndices != null && i < pPKColumnIndices.length; i++)
			{
				if (i > 0)
				{
					sInsertStatement.append(", ");
				}
				
				sInsertStatement.append(pServerMetaData.getServerColumnMetaData(pPKColumnIndices[i]).getColumnName().getQuotedName());
			}
			sInsertStatement.append(" INTO ");
			
			for (int i = 0; pPKColumnIndices != null && i < pPKColumnIndices.length; i++)
			{
				if (i > 0)
				{
					sInsertStatement.append(", ");
				}
				sInsertStatement.append("?");
			}
		
			sInsertStatement.append("; END;");
			
			pInsertStatement = sInsertStatement.toString();
		}

		CallableStatement csInsert = null;
		try
		{
			// #436 - OracleDBAccess and PostgresDBAccess should translate JVx quotes in specific insert
			String sSQL = translateQuotes(pInsertStatement);
			debug("executeSQL->", sSQL);
			csInsert = getConnectionIntern().prepareCall(sSQL);
		
			ServerColumnMetaData[] cmdServerColumnMetaData = pServerMetaData.getServerColumnMetaData();
			int[] iaWriteables = pServerMetaData.getWritableColumnIndices();
			
			int iLastIndex = 0;
			if (pDummyColumn == null)
			{
				iLastIndex = setColumnsToStore(csInsert, cmdServerColumnMetaData, iaWriteables, pNewDataRow, null);
			}
			else
			{
				for (int i = 0; i < cmdServerColumnMetaData.length; i++)
				{
					if (cmdServerColumnMetaData[i].getColumnName().getQuotedName().equals(pDummyColumn))
					{
						csInsert.setObject(1, null, cmdServerColumnMetaData[i].getSQLType());
						break;
					}
				}					
				iLastIndex = 1;
			}
			
			if (returnPK)
			{
				for (int i = 0; pPKColumnIndices != null && i < pPKColumnIndices.length; i++)
				{
					int iSQLType = cmdServerColumnMetaData[pPKColumnIndices[i]].getColumnMetaData().getTypeIdentifier();
	
					csInsert.registerOutParameter(iLastIndex + i + 1, iSQLType);
				}
			}
			if (csInsert.executeUpdate() == 1)
			{
				// use RETURNING to get the PK column values filled in by the trigger
				// get the out parameters, and set the PK columns
				if (returnPK)
				{
					for (int i = 0; pPKColumnIndices != null && i < pPKColumnIndices.length; i++)
					{
						pNewDataRow[pPKColumnIndices[i]] = csInsert.getObject(iLastIndex + i + 1);
					}
				}
				return pNewDataRow;
			}
			throw new DataSourceException("Insert failed ! - Result row count != 1 ! - " +  pInsertStatement);
		}
		catch (SQLException sqlException)
		{			
			throw new DataSourceException("Insert failed! - " + pInsertStatement, formatSQLException(sqlException));
		}
		finally
		{
		    CommonUtil.close(csInsert);
		}
	}
	
	/**
	 * Gets the table name for a synonym.
	 * 
	 * @param pStatement the synonym metadata statement
	 * @param pSynomyn the name of the synonym
	 * @return the name of the table or the <code>pSynonym</code> name if no synonym was found
	 * @throws DataSourceException if synonym detection failed
	 */
    protected String getTableForSynonymIntern(String pStatement, String pSynomyn) throws DataSourceException
    {       
        PreparedStatement psResult = null;
        ResultSet rsResultSet = null;
        try
        {   
            long lMillis = System.currentTimeMillis();
            
            psResult = getPreparedStatement(pStatement);
            
            psResult.setString(1, removeQuotes(pSynomyn));
            rsResultSet = psResult.executeQuery();
            if (!rsResultSet.next())
            {
                return pSynomyn;
            }
            
            // Create schema.table@db_link
            String sSchema = rsResultSet.getString("TABLE_OWNER");
            String sTable  = rsResultSet.getString("TABLE_NAME");
            String sDBLink = rsResultSet.getString("DB_LINK");

            StringBuilder sRealTable = new StringBuilder();
            
            if (sSchema != null)
            {
                sRealTable.append(sSchema);                 
                sRealTable.append('.');                 
            }

            sRealTable.append(sTable);                  

            if (sDBLink != null)
            {
                sRealTable.append('@');                 
                sRealTable.append(sDBLink);                 
            }
            
            if (isLogEnabled(LogLevel.DEBUG))
            {
                debug("getTableForSynonym(", pSynomyn, ") in ", Long.valueOf((System.currentTimeMillis() - lMillis)), "ms");
            }
            
            return sRealTable.toString();
        }
        catch (SQLException sqlException)
        {
            throw new DataSourceException("Synonyms couldn't determined from database! - " + pSynomyn, formatSQLException(sqlException));
        }       
        finally
        {
            CommonUtil.close(rsResultSet, psResult);
        }           
    }
	
} 	// OracleDBAccess
