/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 19.11.2008 - [RH] - creation
 * 20.11.2008 - [RH] - createFilter() for typical default Filters
 * 04.05.2009 - [RH] - createEqualsFilter(Object[],...) added
 * 14.07.2009 - [JR] - createEqualsFilter: meta data added and used for column positions
 * 18.10.2009 - [RH] - createEqualsFilter - if the ColumnNames size == 0 then return null, instead of an index out of bounds exception. [BUGFIX]
 * 11.12.2012 - [JR] - createFullTextFilter: pIncludedColumns added
 * 23.02.2013 - [JR] - toString implemented
 *                   - createCondition with removabel column names implemented
 * 24.11.2013 - [JR] - createEqualsFilter with additional parameter to ignore/allow null values
 * 25.04.2014 - [JR] - #1022: check mem filter                   
 */
package com.sibvisions.rad.model;

import javax.rad.model.ColumnDefinition;
import javax.rad.model.IDataBook;
import javax.rad.model.IDataRow;
import javax.rad.model.ModelException;
import javax.rad.model.condition.And;
import javax.rad.model.condition.CompareCondition;
import javax.rad.model.condition.Equals;
import javax.rad.model.condition.GreaterEquals;
import javax.rad.model.condition.ICondition;
import javax.rad.model.condition.LessEquals;
import javax.rad.model.condition.Like;
import javax.rad.model.condition.LikeIgnoreCase;
import javax.rad.model.condition.Not;
import javax.rad.model.condition.OperatorCondition;
import javax.rad.model.condition.Or;
import javax.rad.model.datatype.BigDecimalDataType;
import javax.rad.model.datatype.BinaryDataType;
import javax.rad.model.datatype.BooleanDataType;
import javax.rad.model.datatype.IDataType;
import javax.rad.model.datatype.StringDataType;
import javax.rad.model.datatype.TimestampDataType;
import javax.rad.persist.ColumnMetaData;
import javax.rad.ui.celleditor.ILinkedCellEditor;

import com.sibvisions.rad.model.mem.MemDataBook;
import com.sibvisions.rad.model.remote.RemoteDataBook;
import com.sibvisions.util.ArrayUtil;

/**
 * The <code>Filter</code> class provide helper functions to create combined  <code>ICondition</code>s.<br> 
 * 
 * @see javax.rad.model.condition.ICondition
 * @author Roland Hrmann, Martin Handsteiner
 */
public class Filter
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
		
	/**
	 * Should not be used.
	 */
	protected Filter() 
	{ 
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creates a Filter with AND combined LIKE conditions over all filter column names. 
	 * 
	 * @param pFilterDataRow		the IDataRow used for the Filter
	 * @param pFilterColumnNames	the Columns to use in the Filter
	 * @return an Filter with AND combined LIKE conditions over all filter column names.
	 */
	public static ICondition createLikeFilter(IDataRow pFilterDataRow, String[] pFilterColumnNames)
	{
		if (pFilterColumnNames == null || pFilterColumnNames.length == 0)
		{
			return null;
		}
		ICondition cResult = new Like(pFilterDataRow, pFilterColumnNames[0]);
		
		for (int i = 1; i < pFilterColumnNames.length; i++)
		{
			cResult = cResult.and(new Like(pFilterDataRow, pFilterColumnNames[i]));
		}
		return cResult;
	}

	/**
	 * Creates a Filter with AND combined LikeIgnoreCase conditions over all filter column names. 
	 * 
	 * @param pFilterDataRow		the IDataRow used for the Filter
	 * @param pFilterColumnNames	the Columns to use in the Filter
	 * @return an Filter with AND combined LikeIgnoreCase conditions over all filter column names.
	 */
	public static ICondition createLikeIgnoreCaseFilter(IDataRow pFilterDataRow, String[] pFilterColumnNames)
	{
		if (pFilterColumnNames == null || pFilterColumnNames.length == 0)
		{
			return null;
		}
		ICondition cResult = new LikeIgnoreCase(pFilterDataRow, pFilterColumnNames[0]);
		
		for (int i = 1; i < pFilterColumnNames.length; i++)
		{
			cResult = cResult.and(new LikeIgnoreCase(pFilterDataRow, pFilterColumnNames[i]));
		}
		return cResult;
	}

	/**
	 * Creates a Filter with AND combined EQUALS conditions over all filter column names. The new filter ignores
	 * NULL values.
	 * 
	 * @param pFilterDataRow		the IDataRow used for the Filter
	 * @param pFilterColumnNames	the Columns to use in the Filter
	 * @return an Filter with AND combined EQUALS conditions over all filter column names.
	 */
	public static ICondition createEqualsFilter(IDataRow pFilterDataRow, String[] pFilterColumnNames)
	{
		return createEqualsFilter(pFilterDataRow, pFilterColumnNames, true);
	}
	
	/**
	 * Creates a Filter with AND combined EQUALS conditions over all filter column names. 
	 * 
	 * @param pFilterDataRow		the IDataRow used for the Filter
	 * @param pFilterColumnNames	the Columns to use in the Filter
	 * @param pIgnoreNull           <code>true</code> to ignore null values (don't use columns with null value in filter)
	 * @return an Filter with AND combined EQUALS conditions over all filter column names.
	 */
	public static ICondition createEqualsFilter(IDataRow pFilterDataRow, String[] pFilterColumnNames, boolean pIgnoreNull)
	{
		if (pFilterColumnNames == null || pFilterColumnNames.length == 0)
		{
			return null;
		}
		ICondition cResult = new Equals(pFilterDataRow, pFilterColumnNames[0]);
		
		for (int i = 1; i < pFilterColumnNames.length; i++)
		{
			cResult = cResult.and(new Equals(pFilterDataRow, pFilterColumnNames[i], pIgnoreNull));
		}
		return cResult;
	}
	
	/**
	 * Creates a Filter with AND combined EQUALS conditions over all filter column names. 
	 * 
	 * @param pColumnNames		the column names to use
	 * @param pDataRow			the values Object[] of the row to use.
	 * @param pColumnMetaData   the meta data for the columns used by the <code>pDataRow</code>
	 * @return an Filter with AND combined EQUALS conditions over all filter column names.
	 */
	public static ICondition createEqualsFilter(String[] pColumnNames, Object[] pDataRow, ColumnMetaData[] pColumnMetaData)
	{
		if (pColumnNames.length == 0)
		{
			return null;
		}
		int[] iColumnPos = new int[pColumnNames.length];
		
		int iPos;

		//search the positions of the columns in the meta data
		for (int i = 0, anz = pColumnMetaData.length; i < anz; i++)
		{
			iPos = ArrayUtil.indexOf(pColumnNames, pColumnMetaData[i].getName());
			
			if (iPos >= 0)
			{
				iColumnPos[iPos] = i;
			}
		}

		//use the correct column positions
		ICondition cPKFilter = new Equals(pColumnNames[0], pDataRow[iColumnPos[0]]);		
		for (int i = 1; i < pColumnNames.length; i++)
		{
			cPKFilter = cPKFilter.and(new Equals(pColumnNames[i], pDataRow[iColumnPos[i]]));
		}
		
		return cPKFilter;
	}	
	
	/**
	 * Creates an Filter with AND combined and for each <br>
	 * - column with the DataType String with LikeIgnoreCase condition and <br>
	 * - column with the DataType BigDecimal &amp; Boolean with Equals condition and <br>
	 * - for two column in order with the DataType Timestamp the first with GreaterEquals 
	 *   and the second with LessEquals condition and <br>
	 * - one column with the DataType Timestamp with Equals condition and <br>
	 * - column with an ILinkedCellEditor.setValidationEnabled(true) with Equals otherwise with a LikeIgnoreCase condition<br>
	 * over all columns. 
	 * 
	 * @param pFilterDataRow		the IDataRow used for the Filter
	 * @return an Filter with AND combined with a guessed default conditions over all columns.
	 */
	public static ICondition createFilter(IDataRow pFilterDataRow)
	{
		return createFilter(pFilterDataRow, null);
	}
	
	/**
	 * Creates an Filter with AND combined and for each <br>
	 * - column with the DataType String with LikeIgnoreCase condition and <br>
	 * - column with the DataType BigDecimal &amp; Boolean with Equals condition and <br>
	 * - for two column in order with the DataType Timestamp the first with GreaterEquals 
	 *   and the second with LessEquals condition and <br>
	 * - one column with the DataType Timestamp with Equals condition and <br>
	 * - column with an ILinkedCellEditor.setValidationEnabled(true) with Equals otherwise with a LikeIgnoreCase condition<br>
	 * over all filter column names. 
	 * 
	 * @param pFilterDataRow		the IDataRow used for the Filter
	 * @param pFilterColumnNames	the Columns to use in the Filter
	 * @return an Filter with AND combined with a guessed default conditions over all filter column names.
	 */
	public static ICondition createFilter(IDataRow pFilterDataRow, String[] pFilterColumnNames)
	{
		boolean columnsAreSet = pFilterColumnNames != null && pFilterColumnNames.length > 0;
		if (!columnsAreSet)
		{
			pFilterColumnNames = pFilterDataRow.getRowDefinition().getColumnNames();
		}

		ICondition cResult  = null;
		ICondition cCurrent = null;
				
		for (int i = 0; i < pFilterColumnNames.length; i++)
		{
			int columnIndex = pFilterDataRow.getRowDefinition().getColumnDefinitionIndex(pFilterColumnNames[i]);
			ColumnDefinition cdColumn = pFilterDataRow.getRowDefinition().getColumnDefinition(columnIndex);
			
			if (columnsAreSet || cdColumn.isFilterable()) // in generic mode, only filter filterable columns
			{
				IDataType       dtDataType = cdColumn.getDataType();
				
				if (dtDataType instanceof TimestampDataType
				    && i + 1 < pFilterColumnNames.length
				    && pFilterDataRow.getRowDefinition().getColumnDefinition(i + 1).getDataType() instanceof TimestampDataType)
				{
					// - for two column in order with the DataType Timestamp the first with GreaterEquals 
					//   and the second with LessEquals condition
					cCurrent = new GreaterEquals(pFilterDataRow, pFilterColumnNames[i]);
				}
				else if (dtDataType instanceof TimestampDataType
					     && cCurrent instanceof GreaterEquals)
				{
					// - for two column in order with the DataType Timestamp the first with GreaterEquals 
					//   and the second with LessEquals condition
					cCurrent = new LessEquals(pFilterDataRow, pFilterColumnNames[i]);
				}
				else if (dtDataType instanceof BigDecimalDataType
				    || dtDataType instanceof BooleanDataType
				    // - column with an ILinkedCellEditor.setValidationEnabled(true) with Equals
				    || dtDataType.getCellEditor() instanceof ILinkedCellEditor
				       && ((ILinkedCellEditor)dtDataType.getCellEditor()).isValidationEnabled()
				    // - one column with the DataType Timestamp with Equals condition
				    || dtDataType instanceof TimestampDataType)
				{
					cCurrent = new Equals(pFilterDataRow, pFilterColumnNames[i]);
				}
				else if (dtDataType instanceof StringDataType)
				{
					cCurrent = new LikeIgnoreCase(pFilterDataRow, pFilterColumnNames[i]);		
				}
				else
				{
					cCurrent = null;
				}
	
				if (cCurrent != null)
				{
					if (cResult == null)
					{
						cResult = cCurrent;
					}
					else
					{
						cResult = cResult.and(cCurrent);
					}
				}
			}
		}
		return cResult;
	}
	
	/**
	 * Full text filter.
	 * 
	 * @param pDataBook the databook to search.
	 * @param pSearchString the search string.
	 * @param pIncludedColumns the list of column names for filtering or <code>null</code> to use all column names
	 * @return The full text filter
	 */
	public static ICondition createFullTextFilter(IDataBook pDataBook, String pSearchString, String... pIncludedColumns)
	{
		if (pSearchString == null || pSearchString.trim().length() == 0)
		{
			return null;
		}
		else
		{
			String[] columnNames;
			boolean columnsAreSet = pIncludedColumns != null && pIncludedColumns.length > 0;
			
			if (columnsAreSet)
			{
				columnNames = pIncludedColumns;
			}
			else
			{
				columnNames = pDataBook.getRowDefinition().getColumnNames();
			}
			
			boolean bIsMemFilter = false;
			
			if (pDataBook instanceof MemDataBook)
			{
			    if (pDataBook instanceof RemoteDataBook)
			    {
			        bIsMemFilter = ((RemoteDataBook)pDataBook).isMemFilter();
			    }
			    else
			    {
			        bIsMemFilter = true;
			    }
			}
			
			pSearchString = "*" + pSearchString.trim() + "*";
				
			Or orCondition = new Or();
	
			for (int i = 0; i < columnNames.length; i++)
			{
				try
				{
					ColumnDefinition cdColumn = pDataBook.getRowDefinition().getColumnDefinition(columnNames[i]);
					
					if (!(cdColumn.getDataType() instanceof BinaryDataType)
						&& (cdColumn.isWritable() || bIsMemFilter)
						&& (columnsAreSet || cdColumn.isFilterable())) // in generic mode, only filter filterable columns
					{
						orCondition.add(new LikeIgnoreCase(columnNames[i], pSearchString));
					}
				}
				catch (ModelException e)
				{
					// do nothing
				}
			}
			
			return orCondition;
		}
	}

	/**
	 * Creates a new {@link ICondition} from the given {@link ICondition} and ignores or uses specific columns.
	 * 
	 * @param pCondition the condition
	 * @param pInclude <code>true</code> to include only given column, <code>false</code> to exclude given columns
	 * @param pColumns columns which should be ignored or used
	 * @return the condition without removed columns
	 */
	public static ICondition createCondition(ICondition pCondition, boolean pInclude, String... pColumns)
	{
		ICondition condition = null;
		
		if (pCondition instanceof CompareCondition)
		{
			CompareCondition cCompare = (CompareCondition)pCondition;
			
			String sColumnName = cCompare.getColumnName();
			
			if (pInclude)
			{
				if (ArrayUtil.contains(pColumns, sColumnName))
				{
					condition = cCompare;
				}
			}
			else
			{
				//ignore given columns!
				if (!ArrayUtil.contains(pColumns, sColumnName))
				{
					condition = cCompare;
				}
			}
		}
		else if (pCondition instanceof OperatorCondition)
		{			
			OperatorCondition cOperator = (OperatorCondition)pCondition;
						
			ICondition cond;
			
			ICondition[] caConditions = cOperator.getConditions();
			
			for (int i = 0; i < caConditions.length; i++)
			{
				cond = createCondition(caConditions[i], pInclude, pColumns);
							
				if (cond != null)
				{
					if (i > 0 && condition != null)
					{
						if (cOperator instanceof And)
						{
							condition = condition.and(cond);
						}
						else if (cOperator instanceof Or)
						{
							condition = condition.or(cond);
						}
					}
					else 
					{
						condition = cond;
					}
				}
			}
		}
		else if (pCondition instanceof Not)
		{
			ICondition cCond = ((Not)pCondition).getCondition();
			
			ICondition cond = createCondition(cCond, pInclude, pColumns);
		
			if (cond != null)
			{
				condition = new Not(cond);
			}
		}
		
		return condition;
	}     

    /**
     * Gets the value from the first {@link Equals} condition of the given filter.
     * 
     * @param pFilter the filter
     * @param pColumn the column name 
     * @return the value or <code>null</code> if the column was not found
     */
    public static Object getEqualsValue(ICondition pFilter, String pColumn)
    {
        if (pFilter instanceof OperatorCondition)
        {
            for (ICondition cond : ((OperatorCondition)pFilter).getConditions())
            {
                if (cond instanceof Equals)
                {
                    if (pColumn.equals(((Equals)cond).getColumnName()))
                    {
                        return ((Equals)cond).getValue();
                    }
                }
            }
        }
        else if (pFilter instanceof Equals)
        {
            if (pColumn.equals(((Equals)pFilter).getColumnName()))
            {
                return ((Equals)pFilter).getValue();
            }
        }
        
        return null;
    }	
	
}	// Filter
