/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 26.10.2009 - [HM] - creation
 * 13.02.2010 - [JR] - format with Date and long parameter implemented
 * 24.02.2013 - [JR] - convert timezone implemented
 */
package com.sibvisions.util.type;

import java.text.DateFormat;
import java.text.DateFormatSymbols;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Locale;
import java.util.TimeZone;

/**
 * The <code>DateUtil</code> is a utility class for date conversion and for formatting dates
 * as string.
 *  
 * @author Martin Handsteiner
 */
public class DateUtil
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
	/** The date format. */
	private SimpleDateFormat dateFormat;
	
	/** The locale base for creation. */
	private Locale creationLocale = null;
	/** The pattern base for creation. */
	private String creationPattern = null;
	
	/** Performance tuning, pattern. */
	private List<String>[] parsedPattern = null;
	/** Performance tuning, reference. */
    private List<String>[] parsedReference = null;
	/** Performance tuning, months. */
    private String[] months = null;
	/** Performance tuning, shortMonths. */
    private String[] shortMonths = null;
	/** Performance tuning, months. */
    private String[] lowerMonths = null;
	/** Performance tuning, shortMonths. */
    private String[] lowerShortMonths = null;

    /** True, if the format should be checked strict. */
    private boolean strictFormatCheck = false;
	
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** 
	 * Constructs a new instance of <code>DateUtil</code> with default date format.
	 */
	public DateUtil()
	{
		setDateFormat(null);
	}

	/** 
	 * Constructs a new instance of <code>DateUtil</code> that supports empty Strings and null values.
	 * 
	 * @param pDateFormat the formatter that should support empty Strings and null values
	 */
	public DateUtil(DateFormat pDateFormat)
	{
		setDateFormat(pDateFormat);
	}

	/** 
	 * Constructs a new instance of <code>DateUtil</code> that supports empty Strings and null values.
	 * 
	 * @param pDatePattern the pattern that should support empty Strings and null values
	 */
	public DateUtil(String pDatePattern)
	{
		setDatePattern(pDatePattern);
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Parses the date from text.
	 * 
	 * @param pText the text.
	 * @return the parsed date.
     * @throws ParseException if there is an error in the conversion
	 */
	public Date parse(String pText) throws ParseException
	{
		if (pText == null || pText.length() == 0)
		{
			return null;
		}
		else
		{
			if (creationLocale != null)
			{
				setDateFormatIntern(LocaleUtil.getDefault(), creationPattern);
			}
			
			return parseStringIntern(pText);
		}
	}

	/**
	 * Formats the date to text.
	 * 
	 * @param pDate the date.
	 * @return the formatted text.
	 */
	public String format(Date pDate)
	{
		if (pDate == null)
		{
			return null;
		}
		else
		{
			if (creationLocale != null)
			{
				setDateFormatIntern(LocaleUtil.getDefault(), creationPattern);
			}
			
			return dateFormat.format(pDate);
		}
	}

	/**
	 * Gets the date format.
	 * 
	 * @return the date format.
	 */
	public DateFormat getDateFormat()
	{
		return dateFormat;
	}

	/**
	 * Sets the new date format, if something was changed.
	 * @param pLocale the locale
	 * @param pDatePattern the pattern
	 */
	private void setDateFormatIntern(Locale pLocale, String pDatePattern)
	{
		if (pDatePattern == null)
		{
			if (pLocale != creationLocale || creationPattern != null)
			{
				dateFormat = (SimpleDateFormat)DateFormat.getDateTimeInstance(DateFormat.LONG, DateFormat.SHORT, LocaleUtil.getDefault());
				dateFormat.setLenient(false);
				
				creationLocale = pLocale;
				creationPattern = pDatePattern;

				createReferenceDate();
			}
		}
		else
		{
			if (pLocale != creationLocale || !pDatePattern.equals(creationPattern))
			{
				dateFormat = new SimpleDateFormat(pDatePattern, LocaleUtil.getDefault());
				dateFormat.setLenient(false);
				
				creationLocale = pLocale;
				creationPattern = pDatePattern;

				createReferenceDate();
			}
		}
	}
	
	/**
	 * Creates a Reference Date.
	 */
	private void createReferenceDate()
	{
		GregorianCalendar cal = new GregorianCalendar();
	    cal.set(Calendar.HOUR_OF_DAY, 0);
	    cal.set(Calendar.MINUTE, 0);
	    cal.set(Calendar.SECOND, 0);
	    cal.set(Calendar.MILLISECOND, 0);

	    String referenceDate = dateFormat.format(cal.getTime());
		String pattern = dateFormat.toPattern();
		
		parsedPattern = getParsed(pattern, true);
	    parsedReference = getParsed(referenceDate, false);

	    DateFormatSymbols symbols = dateFormat.getDateFormatSymbols();
		months = symbols.getMonths();
		shortMonths = symbols.getShortMonths();
		lowerMonths = new String[months.length];
		lowerShortMonths = new String[months.length];

		for (int i = 0; i < months.length; i++)
		{
			lowerMonths[i] = toLowerAndWithoutUmlauts(months[i]);
			lowerShortMonths[i] = toLowerAndWithoutUmlauts(shortMonths[i]);
		}
	}

	/**
	 * To lower case and replaces umlauts.
	 * @param pMonth the original month
	 * @return the converted month.
	 */
	private String toLowerAndWithoutUmlauts(String pMonth)
	{
		return pMonth.toLowerCase().replace('', 'a').replace('', 'u').replace('', 'o');
	}
	
	/**
	 * Gets the date format.
	 * 
	 * @param pDateFormat the date format.
	 */
	public void setDateFormat(DateFormat pDateFormat)
	{
		if (pDateFormat == null)
		{
			setDateFormatIntern(LocaleUtil.getDefault(), null);
		}
		else if (pDateFormat instanceof SimpleDateFormat)
		{
			dateFormat = (SimpleDateFormat)pDateFormat;
			dateFormat.setLenient(false);

			creationLocale = null;
			creationPattern = null;
			
			createReferenceDate();
		}
		else
		{
			throw new IllegalArgumentException("Only SimpleDateFormat is supported!");
		}
	}

	/**
	 * Gets the date format pattern.
	 * 
	 * @return the date format pattern.
	 */
	public String getDatePattern()
	{
		return dateFormat.toPattern();
	}

	/**
	 * Gets the date format pattern.
	 * 
	 * @param pDatePattern the date format pattern.
	 */
	public void setDatePattern(String pDatePattern)
	{
		if (pDatePattern == null)
		{
			setDateFormat(null);
		}
		else
		{
			setDateFormatIntern(LocaleUtil.getDefault(), pDatePattern);
		}
	}

	/**
	 * Strict format check defines whether the set pattern should be exactly by format, or be more flexibel in analysing the given date.
	 * This has nothing to do with lenient in SimpleDateFormat. Lenient is set to false anyway.
	 * Only correct dates are allowed.
	 * The following will be allowed without strict check, with locale de_AT and pattern: dd. MMMM.yyyy HH:mm
	 *   01.01.2016
	 *   01.Januar.2016 00:00
	 *   01.Jnner.2016
	 *   01.01
	 *   01.01.16
	 *   01.01
	 *   01 01 2016 00 00
	 *   010116
	 *   01012016
	 *   
	 * The result will always be 01.Januar.2016 00:00
	 *  
	 * @return true, if format should be checked strict.
	 */
	public boolean isStrictFormatCheck()
	{
		return strictFormatCheck;
	}
	
	/**
	 * Strict format check defines whether the set pattern should be exactly by format, or be more flexibel in analysing the given date.
	 * This has nothing to do with lenient in SimpleDateFormat. Lenient is set to false anyway.
	 * Only correct dates are allowed.
	 * The following will be allowed without strict check, with locale de_AT and pattern: dd. MMMM.yyyy HH:mm
	 *   01.01.2016
	 *   01.Januar.2016 00:00
	 *   01.Jnner.2016
	 *   01.01
	 *   01.01.16
	 *   01.01
	 *   01 01 2016 00 00
	 *   010116
	 *   01012016
	 *   
	 * The result will always be 01.Januar.2016 00:00
	 *  
	 * @param pStrictFormatCheck true, if format should be checked strict.
	 */
	public void setStrictFormatCheck(boolean pStrictFormatCheck)
	{
		strictFormatCheck = pStrictFormatCheck;
		
	}
	
	/**
	 * Gets the best transformation of the date part for the pattern.
	 * 
	 * @param pPart the date part
	 * @param pPattern the pattern
	 * @param pReference the the reference date part
	 * @return the best transformation of the date part for the pattern
	 */
	private String getPart(String pPart, String pPattern, String pReference)
	{
		try
		{
			if (pPattern.startsWith("MMM"))
			{
				int number = Integer.parseInt(pPart);
				if (pPattern.length() == 3)
				{
					return shortMonths[(number - 1) % 12];
				}
				else
				{
					return months[(number - 1) % 12];
				}
			}
			else if (pPattern.toLowerCase().startsWith("yy"))
			{
				if (pReference.length() > pPart.length())
				{
					int year = Integer.parseInt(pReference.substring(0, pReference.length() - pPart.length()) + pPart);
					int refYear = Integer.parseInt(pReference);
					if (pPart.length() == 2 && year >= refYear + 50)
					{
						year -= 100;
					}

					return String.valueOf(year);
				}
			}
		}
		catch (Exception ex)
		{
			// Do nothing
		}
		
		if (pPattern.startsWith("M"))
		{
			String lowerPart = toLowerAndWithoutUmlauts(pPart);
			for (int i = 0; i < 12; i++)
			{
				if (lowerMonths[i].startsWith(lowerPart) || (lowerPart.length() <= lowerMonths[i].length() + 2 && lowerPart.startsWith(lowerShortMonths[i])))
				{
					int mon = i + 1;
					if (pPattern.length() == 2 && mon < 10)
					{
						return "0" + mon;
					}
					else if (pPattern.length() <= 2)
					{
						return String.valueOf(mon);
					}
					else if (pPattern.length() == 3)
					{
						return shortMonths[i];
					}
					else
					{
						return months[i];
					}
				}
			}
		}

		return pPart;
	}

	/**
	 * Gets the stric transformation of the date part for the pattern.
	 * 
	 * @param pPart the date part
	 * @param pPattern the pattern
	 * @param pReference the the reference date part
	 * @return the best transformation of the date part for the pattern
	 */
	private String getStrictPart(String pPart, String pPattern, String pReference)
	{
		if (pPattern.startsWith("MMM"))
		{
			String lowerPart = toLowerAndWithoutUmlauts(pPart);
			for (int i = 0; i < 12; i++)
			{
				if (lowerMonths[i].equals(lowerPart) || (lowerPart.length() <= lowerMonths[i].length() + 2 && lowerPart.startsWith(lowerShortMonths[i])))
				{
					if (pPattern.length() == 3)
					{
						return shortMonths[i];
					}
					else
					{
						return months[i];
					}
				}
			}
		}

		return pPart;
	}
	
	/**
	 * Gets the character type depending on format or date parsing.
	 * @param pCharacter the character
	 * @param pFormat the format
	 * @return the character type
	 */
	private char getCharacterType(char pCharacter, boolean pFormat)
	{
		if (Character.isLetterOrDigit(pCharacter))
		{
			if (pFormat)
			{
				return pCharacter;
			}
			else
			{
				return Character.isDigit(pCharacter) ? '0' : 'a';
			}
		}
		else
		{
			return ' ';
		}
	}
	
	
	/**
	 * Parses the date parts and the seperator parts.
	 * 
	 * @param pText the date
	 * @param pFormat true, if it is a format
	 * @return the date parts and the seperator parts
	 */
	private List<String>[] getParsed(String pText, boolean pFormat)
	{
		List<String> datePart = new ArrayList<String>();
		List<String> seperatorPart = new ArrayList<String>();
		
		int len = pText.length();
		int pos = 0;
		boolean isLetterOrDigit = Character.isLetterOrDigit(pText.charAt(0));
		char characterType = getCharacterType(pText.charAt(0), pFormat);
		for (int i = 1; i < len; i++)
		{
			char ch = pText.charAt(i);
			
			boolean newIsLetterOrDigit = Character.isLetterOrDigit(ch);
			char newCharacterType = getCharacterType(ch, pFormat);

			String part = pText.substring(pos, i);
			if (isLetterOrDigit != newIsLetterOrDigit
					|| characterType != newCharacterType)
			{
				if (isLetterOrDigit)
				{
					datePart.add(part);
					if (isLetterOrDigit == newIsLetterOrDigit && characterType != newCharacterType)
					{
						seperatorPart.add("");
					}
				}
				else
				{
					seperatorPart.add(part);
				}
				pos = i;
				isLetterOrDigit = newIsLetterOrDigit;
				characterType = newCharacterType;
			}
			else if (isLetterOrDigit && !pFormat && newCharacterType == '0')
			{
				if (datePart.size() < parsedPattern[0].size())
				{
					int maxSize = parsedPattern[0].get(datePart.size()).startsWith("y") ? 4 : 2;
					if (part.length() == maxSize)
					{
						datePart.add(part);
						seperatorPart.add("");

						pos = i;
					}
				}
			}
		}
		String part = pText.substring(pos);
		if (isLetterOrDigit)
		{
			datePart.add(part);
		}
		else
		{
			seperatorPart.add(part);
		}

		return new List[] {datePart, seperatorPart};
	}
	
    /**
     * Parses <code>text</code> returning an Date. Some
     * formatters may return null.
     *
     * @param pText String to convert
     * @return Date representation of text
     * @throws ParseException if there is an error in the conversion
     */
	private Date parseStringIntern(String pText) throws ParseException
	{
		try
		{
			return dateFormat.parse(pText);
		}
		catch (ParseException parseException)
		{
			try
			{
			    List<String>[] parsedDate = getParsed(pText, false);
			    
			    int datePartCount = parsedDate[0].size();
			    int patternSeperatorCount = strictFormatCheck ? parsedDate[1].size() : parsedPattern[1].size();
			    
			    StringBuffer buffer = new StringBuffer(48);
			    for (int i = 0, size = parsedPattern[0].size(); i < size; i++)
			    {
			    	if (i < datePartCount)
			    	{
			    		if (strictFormatCheck)
			    		{
			    			buffer.append(getStrictPart(parsedDate[0].get(i), parsedPattern[0].get(i), parsedReference[0].get(i)));
			    		}
			    		else
			    		{
			    			buffer.append(getPart(parsedDate[0].get(i), parsedPattern[0].get(i), parsedReference[0].get(i)));
			    		}
			    	}
			    	else if (!strictFormatCheck)
			    	{
			    		buffer.append(parsedReference[0].get(i));
			    	}
			    	if (i < patternSeperatorCount)
			    	{
			    		if (strictFormatCheck)
			    		{
			    			buffer.append(parsedDate[1].get(i));
			    		}
			    		else
			    		{
			    			buffer.append(parsedPattern[1].get(i));
			    		}
			    	}
			    }
			    
			    return dateFormat.parse(buffer.toString());
			}
			catch (ParseException exception)
			{
				throw (ParseException)exception;
			}
			catch (Exception exception)
			{
				throw new ParseException("Wrong Dateformat", 0);
			}
		}
	}

	/**
	 * Formats a time string.
	 * 
	 * @param pDate the time (in millis since 01.01.1970 00:00) value to be formatted into a time string
	 * @param pFormat the format 
	 * @return the formatted date/time string
	 * @see SimpleDateFormat
	 */
	public static String format(long pDate, String pFormat)
	{
		return format(new Date(pDate), pFormat);
	}
	
	/**
	 * Formats a Date into a date/time string.
	 * 
	 * @param pDate the time value to be formatted into a time string
	 * @param pFormat the format 
	 * @return the formatted date/time string
	 * @see SimpleDateFormat
	 */
	public static String format(Date pDate, String pFormat)
	{
		SimpleDateFormat sdf = new SimpleDateFormat(pFormat);
		
		return sdf.format(pDate);
	}
	
	/**
	 * Creates a new date instance.
	 * 
	 * @param pDay the day of month
	 * @param pMonth the month
	 * @param pYear the year
	 * @param pHour the hour of day
	 * @param pMinutes the minutes
	 * @param pSeconds the seconds
	 * @return the date
	 */
	public static Date getDate(int pDay, int pMonth, int pYear, int pHour, int pMinutes, int pSeconds)
	{
		Calendar cal = GregorianCalendar.getInstance();
		
		cal.set(Calendar.DAY_OF_MONTH, pDay);
		cal.set(Calendar.MONTH, pMonth - 1);
		cal.set(Calendar.YEAR, pYear);
		cal.set(Calendar.HOUR_OF_DAY, pHour);
		cal.set(Calendar.MINUTE, pMinutes);
		cal.set(Calendar.SECOND, pSeconds);
		cal.set(Calendar.MILLISECOND, 0);
		
		return cal.getTime();
	}
	
	/**
	 * Converts a date from one timezone to another timezone.
	 * 
	 * @param pDate the date
	 * @param pFromTimeZone the timezone of the date
	 * @param pToTimeZone the expected timezone
	 * @return the date converted to the expected timezone
	 */
	public static Date convert(Date pDate, String pFromTimeZone, String pToTimeZone)
	{
		return convert(pDate, TimeZone.getTimeZone(pFromTimeZone), TimeZone.getTimeZone(pToTimeZone));
	}

	/**
	 * Converts a date from one timezone to another timezone.
	 * 
	 * @param pDate the date
	 * @param pFromTimeZone the timezone of the date
	 * @param pToTimeZone the expected timezone
	 * @return the date converted to the expected timezone
	 */
	public static Date convert(Date pDate, TimeZone pFromTimeZone, TimeZone pToTimeZone)
	{
		if (pDate == null)
		{
			return null;
		}
		
		Calendar fromCal = Calendar.getInstance(pFromTimeZone);
		fromCal.setTime(pDate);
		   
		Calendar toCal = Calendar.getInstance(pToTimeZone);
		toCal.setTime(pDate);

		int iFromOffset = fromCal.get(Calendar.ZONE_OFFSET) + fromCal.get(Calendar.DST_OFFSET);
		int iToOffset = toCal.get(Calendar.ZONE_OFFSET) + toCal.get(Calendar.DST_OFFSET);

		return new Date(toCal.getTimeInMillis() + iToOffset - iFromOffset);		
	}
	
}	// DateUtil
