/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 14.10.2009 - [JR] - creation
 * 27.10.2009 - [JR] - send: parameter validation [BUGFIX]
 * 02.11.2012 - [JR] - set charset=UTF-8 for html mails
 * 21.12.2016 - [JR] - #1690: properties and TLS support
 */
package com.sibvisions.util;

import java.io.InputStream;
import java.util.Date;
import java.util.Properties;

import javax.activation.DataHandler;
import javax.activation.DataSource;
import javax.mail.Authenticator;
import javax.mail.BodyPart;
import javax.mail.Message;
import javax.mail.Multipart;
import javax.mail.PasswordAuthentication;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import javax.mail.internet.MimeUtility;

/**
 * The <code>Mail</code> class is a utility class for sending mails.
 * 
 * @author Ren Jahn
 */
public final class Mail extends Authenticator
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the plain text content. */
	public static final String CONTENT_TEXT = "text/plain; charset=UTF-8";
	
	/** the html content. */
	public static final String CONTENT_HTML = "text/html; charset=UTF-8";

	
	/** the hostname/ip of the mailserver. */
	private String sHost;
	
	/** the smtp port of the mailserver. */
	private String sPort;
	
	/** the username for smtp authentication. */
	private String sUserName;
	
	/** the password for smtp authentication. */
	private String sPassword;
	
	/** the content type. */
	private String sContentType = CONTENT_TEXT;
	
    /** the additional mail properties. */
    private Properties props;

    /** whether TLS is enabled. */
    private boolean bTLSEnabled = false;

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creates a new instance of <code>Mail</code> with a specific mailserver and smtp port.
	 * 
	 * @param pHost the hostname or ip address of the mailserver
	 * @param pSmtpPort the smtp port of the mailserver
	 */
	public Mail(String pHost, String pSmtpPort)
	{
		this(pHost, pSmtpPort, null, null);
	}
	
	/**
	 * Creates a new instance of <code>Mail</code> with a specific mailserver, smtp port and
	 * authentication credentials.
	 * 
	 * @param pHost the hostname or ip address of the mailserver
	 * @param pSmtpPort the smtp port of the mailserver
	 * @param pUserName the username for smtp authentication
	 * @param pPassword the password for smtp authentication
	 */
	public Mail(String pHost, String pSmtpPort, String pUserName, String pPassword)
	{
		sHost = pHost;
		sPort = pSmtpPort;
		
		sUserName = pUserName;
		sPassword = pPassword;
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Interface implementation
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	protected PasswordAuthentication getPasswordAuthentication() 
	{
        return new PasswordAuthentication(sUserName, sPassword);
    }
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	
	
	/**
	 * Sets the content type.
	 * 
	 * @param pContentType the content type
	 */
	public void setContentType(String pContentType)
	{
		sContentType = pContentType;
	}
	
	/**
	 * Gets the content type.
	 * 
	 * @return the content type
	 */
	public String getContentType()
	{
		return sContentType;
	}
	
	/**
	 * Sets or removes a property.
	 * 
	 * @param pName the name
	 * @param pValue the value or <code>null</code> to remove the property
	 */
	public void setProperty(String pName, String pValue)
	{
	    if (pValue != null)
	    {
    	    if (props == null)
    	    {
    	        props = new Properties();
    	    }
    	    
    	    props.put(pName, pValue);
	    }
	    else if (props != null)
	    {
	        props.remove(pName);
	        
	        if (props.isEmpty())
	        {
	            props = null;
	        }
	    }
	}
	
	/**
	 * Gets the value of a property.
	 * 
	 * @param pName the name
	 * @return the value
	 */
	public Object getProperty(String pName)
	{
	    if (props != null)
	    {
	        return props.get(pName);
	    }
	    else
	    {
	        return null;
	    }
	}
	
	/**
	 * Sets whether TLS is enabled.
	 * 
	 * @param pEnable <code>true</code> to enable, <code>false</code> otherwise
	 */
	public void setTLSEnabled(boolean pEnable)
	{
	    bTLSEnabled = pEnable;
	}
	
	/**
	 * Gets whether TLS is enabled.
	 * 
	 * @return <code>true</code> if enabled, <code>false</code> otherwise
	 */
	public boolean isTLSEnabled()
	{
	    return bTLSEnabled;
	}

    /**
     * Sends a plain text mail.
     * 
     * @param pFrom the sender
     * @param pTo the recipient(s) comma separated
     * @param pSubject the subject of the message
     * @param pText the text of the message
     * @throws Exception if the send operation failed
     */
    public void send(String pFrom, String pTo, String pSubject, String pText) throws Exception
    {
        send(pFrom, pTo, null, pSubject, pText, null, null);
    }
	
	/**
	 * Sends a plain text mail.
	 * 
	 * @param pFrom the sender
	 * @param pTo the recipient(s) comma separated
	 * @param pCc the carbon copy recipient(s) comma separated
	 * @param pSubject the subject of the message
	 * @param pText the text of the message
	 * @throws Exception if the send operation failed
	 */
	public void send(String pFrom, String pTo, String pCc, String pSubject, String pText) throws Exception
	{
		send(pFrom, pTo, pCc, pSubject, pText, null, null);
	}
	
	/**
	 * Sends a plain text mail.
	 * 
	 * @param pFrom the sender
	 * @param pTo the recipient(s) comma separated
	 * @param pCc the carbon copy recipient(s) comma separated
	 * @param pSubject the subject of the message
	 * @param pText the text of the message
	 * @param pFilename the file name
	 * @param pContent the content
	 * @throws Exception if the send operation failed
	 */
	public void send(String pFrom, String pTo, String pCc, String pSubject, String pText, String pFilename, Object pContent) throws Exception
	{
		send(pFrom, pTo, pCc, null, pSubject, pText, pFilename, pContent);
	}
	
	/**
	 * Sends a plain text mail.
	 * 
	 * @param pFrom the sender
	 * @param pTo the recipient(s) comma separated
	 * @param pCc the carbon copy recipient(s) comma separated
	 * @param pBcc the carbon copy recipient(s) comma separated
	 * @param pSubject the subject of the message
	 * @param pText the text of the message
	 * @param pFilename the file name
	 * @param pContent the content
	 * @throws Exception if the send operation failed
	 */
	public void send(String pFrom, String pTo, String pCc, String pBcc, String pSubject, String pText, String pFilename, Object pContent) throws Exception
	{
		//Validateion
		if (sHost == null || sHost.trim().length() == 0)
		{
			throw new IllegalArgumentException("Invalid hostname: ''");
		}

		if (sPort == null || sPort.trim().length() == 0)
		{
			throw new IllegalArgumentException("Invalid port: ''");
		}

		if (pFrom == null || pFrom.trim().length() == 0)
		{
			throw new IllegalArgumentException("Invalid sender: ''");
		}
		
		if (pTo == null || pTo.trim().length() == 0)
		{
			throw new IllegalArgumentException("Invalid recipient: ''");
		}
		
		Properties prop = new Properties();
		
    	prop.put("mail.smtp.host", sHost);
        prop.put("mail.smtp.port", sPort);
        
        Authenticator auth = null;
        
        if (sUserName != null)
        {
            auth = this;
            
            prop.put("mail.smtp.auth", "true");
        }
        else
        {
        	prop.put("mail.smtp.auth", "false");
        }
        
        if (bTLSEnabled)
        {
            prop.put("mail.smtp.starttls.enable", "true");
        }

        //custom properties - allows overriding
        if (props != null)
        {
            prop.putAll(props);
        }

        //comma separated!
        String sTo = pTo.replace(";", ",");
        
        Session sess = Session.getInstance(prop, auth);

        InternetAddress iaTo = new InternetAddress(pFrom, false);
        iaTo.setPersonal(iaTo.getPersonal(), "UTF-8");
        
        MimeMessage mess = new MimeMessage(sess);
  		mess.setFrom(iaTo);
  		mess.setRecipients(Message.RecipientType.TO, InternetAddress.parse(sTo, false));

  		if (pCc != null)
  		{
  			String sCc = pCc.replace(";", ",");
  			
      		mess.setRecipients(Message.RecipientType.CC, InternetAddress.parse(sCc, false));
  		}
  		
  		if (pBcc != null)
  		{
  			String sBcc = pBcc.replace(";", ",");
  			
      		mess.setRecipients(Message.RecipientType.BCC, InternetAddress.parse(sBcc, false));
  		}
  		
  		mess.setSubject(MimeUtility.encodeText(pSubject, "UTF-8", "Q"));
  		
  		DataSource source = null;
  		if (pContent instanceof DataSource)
  		{
  			source = (DataSource)pContent;
  			
  			if (pFilename == null || pFilename.length() == 0)
  			{
  				pFilename = source.getName();
  			}
  		}
  		else if (pFilename != null && pFilename.length() > 0)
  		{
  			if (pContent instanceof InputStream)
	  		{
  				source = new ByteArrayDataSource(pFilename, (InputStream)pContent);
	  		}
	  		else if (pContent instanceof byte[])
	  		{
	  			source = new ByteArrayDataSource(pFilename, (byte[])pContent);
	  		}
  		}
  		
  		if (source == null)
  		{
  		    mess.setContent(pText, sContentType);
  		}
  		else
  		{
	  		BodyPart messageBodyPart = new MimeBodyPart();
		  	messageBodyPart.setContent(pText, sContentType);
		
		  	Multipart multipart = new MimeMultipart();
		  	multipart.addBodyPart(messageBodyPart);
		
		  	messageBodyPart = new MimeBodyPart();
		  	messageBodyPart.setDataHandler(new DataHandler(source));
		  	messageBodyPart.setFileName(pFilename);
		  	multipart.addBodyPart(messageBodyPart);
		  	
		  	mess.setContent(multipart);
  		}
  		
  		mess.setHeader("X-Mailer", Mail.class.getName());
  		mess.setSentDate(new Date());

		Transport.send(mess);
	}
	
}	// Mail
