/*
 * Copyright 2012 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 20.07.2012 - [JR] - creation
 */
package com.sibvisions.rad.persist.jdbc.param;

import java.sql.Types;

/**
 * The <code>AbstractParam</code> defines a parameter for JDBC calls.
 * 
 * @author Ren Jahn
 */
public abstract class AbstractParam
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** Delegate the type decision to the JDBC driver. */
	public static final int SQLTYPE_AUTOMATIC = Integer.MAX_VALUE;

	/** VARCHAR SQL Type {@link Types#VARCHAR}. */
	public static final int SQLTYPE_VARCHAR = Types.VARCHAR;
	
	/** VARCHAR SQL Type {@link Types#DECIMAL}. */
	public static final int SQLTYPE_DECIMAL = Types.DECIMAL;
	
	/** VARCHAR SQL Type {@link Types#TIMESTAMP}. */
	public static final int SQLTYPE_TIMESTAMP = Types.TIMESTAMP;
	
	
	/** The possible parameter types. */
	public enum ParameterType
	{
		/** Input parameter. */
		In,
		/** Output parameter. */
		Out,
		/** Input and Output parameter. */
		InOut
	}
	
	/** the value. */
	private Object oValue;

	/** the parameter type. */
	private ParameterType type;
	
	/** the SQL type. */
	private int iSqlType;
	
	/** the SQL type. */
	private String sArrayType;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Creates a new instance of <code>AbstractParam</code>.
	 * 
	 * @param pType the parameter type
	 * @param pSqlType the SQL type
	 * @param pValue the value
	 */
	protected AbstractParam(ParameterType pType, int pSqlType, Object pValue)
	{
		type = pType;
		iSqlType = pSqlType;
		oValue = pValue;
	}

	/**
	 * Creates a new instance of <code>AbstractParam</code>.
	 * 
	 * @param pType the parameter type
	 * @param pArrayType the array type
	 * @param pValue the value
	 */
	protected AbstractParam(ParameterType pType, String pArrayType, Object pValue)
	{
		type = pType;
		iSqlType = Types.ARRAY;
		sArrayType = pArrayType;
		oValue = pValue;
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Gets the parameter type.
	 * 
	 * @return the type
	 */
	public ParameterType getType()
	{
		return type;
	}
	
	/**
	 * Gets the SQL type.
	 * 
	 * @return the SQL Type
	 * @see java.sql.Types
	 */
	public int getSqlType()
	{
		if (oValue == null && iSqlType == SQLTYPE_AUTOMATIC)
		{
			//no specific SQL Type and the null as value -> try VARCHAR
			return Types.VARCHAR;
		}
		else
		{
			return iSqlType;
		}
	}
	
	/**
	 * Gets the SQL type.
	 * 
	 * @return the SQL Type
	 * @see java.sql.Types
	 */
	public String getArrayType()
	{
		return sArrayType;
	}
	
	/**
	 * Gets the current value.
	 * 
	 * @return the value
	 */
	public Object getValue()
	{
		return oValue;
	}

	/**
	 * Sets the current value.
	 * 
	 * @param pValue the value
	 */
	public void setValue(Object pValue)
	{
		oValue = pValue;
	}
	
}	// AbstractParam
