/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 * 
 * 08.05.2009 - [JR] - creation
 * 04.10.2009 - [JR] - setProperty: security for CLIENT properties
 * 18.11.2009 - [JR] - #33: WrappedSession: put implemented
 * 02.03.2011 - [JR] - #297: implemented addObject, removeObject, getObject and renamed addObject to putObject
 * 10.07.2013 - [JR] - #725: extended AbstractSessionContext instead of SessionContext
 */
package com.sibvisions.rad.server;

import javax.rad.remote.IConnection;
import javax.rad.server.ICallBackBroker;
import javax.rad.server.ICallHandler;
import javax.rad.server.IConfiguration;
import javax.rad.server.ISession;
import javax.rad.server.InjectObject;
import javax.rad.server.SessionContext;

import com.sibvisions.rad.server.config.ServerZone;

/**
 * The <code>SessionContextImpl</code> is an internal {@link javax.rad.server.SessionContext} implementation for
 * the {@link AbstractSession}.
 * 
 * @author Ren Jahn
 */
final class SessionContextImpl extends SessionContext
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /** the master session. */
    private WrappedSession wsessMaster;

    /** the associated session. */
	private WrappedSession wsessCurrent;

	/** the callback broker. */
	private ICallBackBroker broker;
	
    /** the name of the object from which the method will be called. */
	private String sObjectName = null;
	
	/** the name of the method which will be called. */
	private String sMethodName = null;

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creates a new instance of <code>SessionContextImpl</code> for a specific
	 * {@link ISession}.
	 * 
	 * @param pSession the associated session for this {@link javax.rad.server.SessionContext}
	 */
	SessionContextImpl(AbstractSession pSession)
	{
		wsessCurrent = new WrappedSession(pSession);
		
		setCurrentInstance(this);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Abstract methods implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void destroy()
	{
		setCurrentInstance(null);

		wsessMaster = null;
		wsessCurrent = null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ISession getSession()
	{
		return wsessCurrent;
	}
	
    /**
     * {@inheritDoc}
     */
    @Override
    public ISession getMasterSession()
    {
        if (wsessMaster != null)
        {
            return wsessMaster;
        }
        else
        {
            if (wsessCurrent == null)
            {
                return null;
            }
            
            AbstractSession sessorig = wsessCurrent.session;
                    
            if (sessorig instanceof MasterSession)
            {
                wsessMaster = wsessCurrent;
                
                return wsessMaster;
            }
            else if (sessorig instanceof SubSession)
            {
                wsessMaster = new WrappedSession(((SubSession)sessorig).getMasterSession());
                
                return wsessMaster;
            }
        }
        
        return null;
    }
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public IConfiguration getSessionConfig()
	{
		if (wsessCurrent == null)
		{
			return null;
		}
		
		return wsessCurrent.getConfig();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IConfiguration getServerConfig()
	{
		if (wsessCurrent == null)
		{
			return null;
		}
		
        ServerZone zone = wsessCurrent.session.getApplicationZone().getServerZone();
        
        if (zone != null)
        {
            return zone.getConfig();
        }
        
        return null;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public IConnection getServerConnection()
	{
		if (wsessCurrent == null)
		{
			return null;
		}
		
		return new DirectServerConnection((IDirectServer)wsessCurrent.session.getSessionManager().getServer());
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getObjectName()
	{
		return sObjectName;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getMethodName()
	{
		return sMethodName;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public InjectObject putObject(InjectObject pObject)
	{
		return wsessCurrent.session.putObject(pObject);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public InjectObject removeObject(InjectObject pObject)
	{
		return wsessCurrent.session.removeObject(pObject);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public InjectObject getObject(String pName)
	{
		return wsessCurrent.session.getObject(pName);
	}
	
    /**
     * {@inheritDoc}
     */
	@Override
    public ICallHandler getCallHandler()
    {
        return wsessCurrent.session.getCallHandler();
    }
	
    /**
     * {@inheritDoc}
     */
    @Override
	public ICallBackBroker getCallBackBroker()
	{
        if (broker == null)
        {
            broker = new SessionCallBackBroker(this);
        }
        
        return broker;
	}
    
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // User-defined methods
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Sets the name of the current object from which the method will be called.
	 * 
	 * @param pObjectName the object name or <code>null</code> if the object is unknown
	 * @see #getObjectName()
	 */
	void setObjectName(String pObjectName)
	{
		sObjectName = pObjectName;
	}
	
	/**
	 * Sets the name of the method which will be called.
	 * 
	 * @param pMethodName the method name or <code>null</code> if the method is unknown
	 * @see #getMethodName()
	 */
	void setMethodName(String pMethodName)
	{
		sMethodName = pMethodName;
	}
	
}	// SessionContextImpl
