/*
 * Copyright 2016 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 08.06.2016 - [JR] - creation
 */
package com.sibvisions.rad.server;

import javax.rad.remote.IConnection;
import javax.rad.server.ICallBackBroker;
import javax.rad.server.ISession;
import javax.rad.server.ResultObject;
import javax.rad.server.SessionContext;

import com.sibvisions.util.ArrayUtil;

/**
 * The <code>SessionCallBackBroker</code> is the {@link ICallBackBroker} implementation for {@link javax.rad.server.ISession}.
 * 
 * @author Ren Jahn
 */
final class SessionCallBackBroker implements ICallBackBroker
{
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Class members
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /** the session. */
    private WrappedSession session;
    
    /** the master session. */
    private WrappedSession master;
    
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Initialization
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * Creates a new instance of <code>SessionCallBackBroker</code>.
     * 
     * @param pContext the session context
     */
    SessionCallBackBroker(SessionContext pContext)
    {
        session = (WrappedSession)pContext.getSession();
        master = (WrappedSession)pContext.getMasterSession();
    }

    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Interface implementation
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * {@inheritDoc}
     */
    public PublishState publish(String pInstruction, Object pObject)
    {
        if (master != null)
        {
            if (session.isValid() && master.isValid())
            {
                master.session.addCallBackResult(new ResultObject(IConnection.TYPE_CALLBACKRESULT_RESULT, pObject, pInstruction + "@" + session.getId()));

                return PublishState.Completed;
            }
        }
        
        return PublishState.Failed;
    }
    
    /**
     * {@inheritDoc}
     */
    public PublishState publishAll(String pInstruction, Object pObject)
    {
        if (master != null)
        {
            String sName = session.getApplicationName();
            
            DefaultSessionManager sman = master.session.getSessionManager();
            
            Monitoring monitor = ((Server)sman.getServer()).getMonitoring();
            
            ArrayUtil<Object> liMasterIds = monitor.getMasterSessionIds(sName);
    
            if (liMasterIds != null)
            {
                boolean bPartial = false;
                
                ISession sessMaster;
                ISession sessNotify;
                
                for (int i = 0, cnt = liMasterIds.size(); i < cnt; i++)
                {
                    try
                    {
                        sessMaster = sman.get(liMasterIds.get(i));
                        sessNotify = sessMaster;
                        
                        //We send the object to every master session, but if the found master session is the same session as the current master session, we try
                        //to notify the triggering session and not the master session!
                        
                        if (sessMaster == master.session)
                        {
                            if (session.session != master.session)
                            {
                                sessNotify = session;
                            }
                        }

                        ((AbstractSession)sessMaster).addCallBackResult(new ResultObject(IConnection.TYPE_CALLBACKRESULT_RESULT, pObject, pInstruction + "@" + sessNotify.getId()));
                    }
                    catch (Exception e)
                    {
                        bPartial = true;
                    }
                }
                
                if (bPartial)
                {
                    return PublishState.Partial;
                }
                else
                {
                    return PublishState.Completed;
                }
            }
        }
        
        return PublishState.Failed;
    }

}   // SessionCallBackBroker
