/*
 * Copyright 2016 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 07.06.2016 - [RZ] - creation.                          
 */
package com.sibvisions.rad.persist.jdbc;

import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.List;
import java.util.Map;

import javax.rad.persist.DataSourceException;

import com.sibvisions.util.type.StringUtil;

/**
 * The {@link H2DBAccess} is the implementation for the H2 database.
 * 
 * @see com.sibvisions.rad.persist.jdbc.DBAccess
 * 
 * @author Robert Zenz
 */
public class H2DBAccess extends DBAccess
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Creates a new instance of {@link H2DBAccess}.
	 */
	public H2DBAccess()
	{
		super();
		
		setDriver("org.h2.Driver");
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getColumnName(ResultSetMetaData pMetaData, int pColumn) throws SQLException
	{
		// We need to return the column label because H2 is JDBC compliant and
		// does set the column name as name, and not the label.
		// If we have "select A "B" from C" then "A" is the column name and
		// "B" is the column label.
		return pMetaData.getColumnLabel(pColumn);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Map<String, Object> getDefaultValuesIntern(String pCatalog, String pSchema, String pTable) throws DataSourceException
	{
		// We must convert the table name to uppercase to make sure that the
		// metadata for that table is loaded.
		return super.getDefaultValuesIntern(pCatalog, pSchema, pTable.toUpperCase());
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<ForeignKey> getForeignKeysIntern(String pCatalog, String pSchema, String pTable) throws DataSourceException
	{
		// We must convert the table name to uppercase to make sure that the
		// metadata for that table is loaded.
		return super.getForeignKeysIntern(pCatalog, pSchema, pTable.toUpperCase());
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Object translateDefaultValue(String pColumnName, int pDataType, String pDefaultValue) throws Exception
	{
		// H2 does deliver the default values quoted.
		return super.translateDefaultValue(pColumnName, pDataType, StringUtil.removeQuotes(pDefaultValue, "'"));
	}
	
}	// H2DBAccess
