/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 * 
 * 06.10.2008 - [JR] - creation
 * 06.11.2008 - [JR] - extends WorkScreenBase instead of AbstractApplicationBase
 * 22.02.2009 - [JR] - doClose implemented
 * 04.08.2009 - [JR] - QUESTION icon supported
 * 12.05.2010 - [JR] - setEventHandler
 * 04.08.2011 - [JR] - setFont for TextArea because the default font is monospaced
 * 28.04.2016 - [JR] - #2938: use margin anchors for message
 */
package com.sibvisions.rad.application;

import javax.rad.application.IContent;
import javax.rad.application.IMessageConstants;
import javax.rad.genui.UIColor;
import javax.rad.genui.UIDimension;
import javax.rad.genui.UIFont;
import javax.rad.genui.UIImage;
import javax.rad.genui.component.UIButton;
import javax.rad.genui.component.UIIcon;
import javax.rad.genui.component.UITextArea;
import javax.rad.genui.container.UIDesktopPanel;
import javax.rad.genui.container.UIInternalFrame;
import javax.rad.genui.container.UIPanel;
import javax.rad.genui.layout.UIBorderLayout;
import javax.rad.genui.layout.UIFlowLayout;
import javax.rad.genui.layout.UIFormLayout;
import javax.rad.ui.IAlignmentConstants;
import javax.rad.ui.event.ActionHandler;
import javax.rad.ui.event.IActionListener;
import javax.rad.ui.event.UIActionEvent;
import javax.rad.ui.event.UIEvent;
import javax.rad.ui.event.UIWindowEvent;

import com.sibvisions.util.type.StringUtil;

/**
 * The <code>Message</code> class is an {@link UIInternalFrame} to
 * display messages on the screen. The message contains an icon, 
 * the message and message dependent buttons.
 *  
 * @author Ren Jahn
 */
public class Message extends UIInternalFrame
                     implements IContent,
                                IMessageConstants
{
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Class members
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    /** the eventhandler for eventdispatching. */
    private static ActionHandler eventMessage = new ActionHandler();

    /** the message opener. */
    private Object          opener;
    
    /** the area for displaying the message. */
    private UITextArea      taMessage   = new UITextArea();
    
    /** the ok/yes button. */
    private UIButton        butOK       = null;
    
    /** the cancel/no button. */
    private UIButton        butCancel   = null;
    
    /** the ok method name. */
    private String          sOkAction;
    
    /** the cancel method name. */
    private String          sCancelAction;
    
    /** the message type. */
    private int             iIconType;
    
    /** the button display type. */
    private int             iButtonType;
    
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Initialization
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * Creates a new instance of <code>Message</code>.
     * 
     * @param pDesktop the desktop for showing the frame
     * @param pIconType the message type
     * @param pButtonType the type for the visible buttons
     * @param pMessage the message
     * @param pOkAction the action to call when ok or yes was clicked
     * @param pCancelAction the action to call when cancel or close was clicked
     */
    public Message(UIDesktopPanel pDesktop, 
                   int pIconType, 
                   int pButtonType, 
                   String pMessage, 
                   String pOkAction, 
                   String pCancelAction)
    {
        super(pDesktop);
        
        iIconType   = pIconType;
        iButtonType = pButtonType;
        
        sOkAction     = pOkAction;
        sCancelAction = pCancelAction;
        
        taMessage.setText(pMessage);
        taMessage.setFont(UIFont.getDefaultFont());
        taMessage.setRows(0);
        taMessage.setColumns(0);
        
        init();
    }
    
    /**
     * Initializes the UI components.
     */
    protected void init()
    {
        UIPanel panCenter = new UIPanel();
        
        UIIcon icon = null;
        
        UIFlowLayout fllButtons = new UIFlowLayout();
        fllButtons.setMargins(8, 0, 8, 0);
        
        UIPanel panButtons = new UIPanel(fllButtons);
        
        UIFormLayout flCenter = new UIFormLayout();
        flCenter.setHorizontalGap(10);
        
        panCenter.setLayout(flCenter);
        panCenter.setBackground(UIColor.white);
        
        //Handle Message Type
        
        String sName;

        UIImage image = null;
        
        switch (iIconType)
        {
            case MESSAGE_ICON_INFO:
                sName = "Information";
                image = UIImage.getImage(UIImage.MESSAGE_INFO_LARGE);
                break;
            case MESSAGE_ICON_WARNING:
                sName = "Warning";
                image = UIImage.getImage(UIImage.MESSAGE_WARNING_LARGE);
                break;
            case MESSAGE_ICON_ERROR:
                sName = "Error";
                image = UIImage.getImage(UIImage.MESSAGE_ERROR_LARGE);
                break;
            case MESSAGE_ICON_QUESTION:
                sName = "Question";
                image = UIImage.getImage(UIImage.MESSAGE_QUESTION_LARGE);
                break;
            default:
                sName = "";
                break;
        }
        
        if (image != null)
        {
            icon = new UIIcon();
            icon.setImage(image);
            icon.setVerticalAlignment(IAlignmentConstants.ALIGN_TOP);
            
            panCenter.add(icon);
        }
        
        setTitle(sName);
        setIconImage(null);
        setMaximizable(false);
        setIconifiable(false);
        setResizable(false);
        setModal(true); 
        
        //Message
        
        taMessage.setEditable(false);
        taMessage.setBorderVisible(false);
        taMessage.setWordWrap(true);

        UIIcon ico = new UIIcon();
        ico.setBackground(UIColor.gray);

        //centered looks better for "short" messages (#2938)
        if (StringUtil.countCharacter(taMessage.getText(), '\n') < 3)
        {
            panCenter.add(taMessage, flCenter.getVCenterConstraints(1,  0, -1, -1));
        }
        else
        {
            panCenter.add(taMessage, flCenter.getConstraints(1, 0, -1, -1));
        }
        
        panCenter.add(ico, flCenter.getConstraints(flCenter.createAnchor(flCenter.getBottomAnchor(), -1),
                                                   flCenter.getLeftAnchor(),
                                                   flCenter.getBottomAnchor(),
                                                   flCenter.getRightAnchor()));

        //Create Buttons
        
        switch (iButtonType)
        {
            case IMessageConstants.MESSAGE_BUTTON_OK:
                butOK     = createButton("OK", "doOk");
                butOK.setDefaultButton(true);
                break;
            case IMessageConstants.MESSAGE_BUTTON_OK_CANCEL:
                butOK     = createButton("OK", "doOk");
                butCancel = createButton("Cancel", "doCancel");
                butCancel.setDefaultButton(true);
                break;
            case IMessageConstants.MESSAGE_BUTTON_YES_NO:
                butOK     = createButton("Yes", "doOk");
                butCancel = createButton("No", "doCancel");
                butCancel.setDefaultButton(true);
                break;
            default:
                //do nothing
                break;
        }
        
        if (butOK != null)
        {
            panButtons.add(butOK);
        }
        
        if (butCancel != null)
        {
            panButtons.add(butCancel);
        }
        
        //add Components
        
        UIBorderLayout blMain = new UIBorderLayout();
        blMain.setVerticalGap(1);
        
        setLayout(blMain);
        
        add(panCenter, UIBorderLayout.CENTER);
        add(panButtons, UIBorderLayout.SOUTH);
        
        eventWindowClosing().addListener(this, "doClose");

        panCenter.setMinimumSize(250, 50);
        panCenter.setMaximumSize(800, 600);
    }   

    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Interface implementation
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    /**
     * {@inheritDoc}
     */
    public void notifyDestroy()
    {
    }

    /**
     * {@inheritDoc}
     */
    public void notifyVisible()
    {
    }
    
    /**
     * {@inheritDoc}
     */
    public <OP> void setOpener(OP pOpener)
    {
        opener = pOpener;
    }
    
    /**
     * {@inheritDoc}
     */
    public <OP> OP getOpener()
    {
        return (OP)opener;
    }

    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // User-defined methods
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    /**
     * Creates an action button for this message.
     * 
     * @param pText the button text
     * @param pAction the redispatch action
     * @return the created button
     */
    protected UIButton createButton(String pText, String pAction)
    {
        UIButton button = new UIButton();
        
        button.setText(pText);
        button.eventAction().addListener(this, pAction);
        button.setPreferredSize(new UIDimension(80, 25));
        
        return button;
    }
    
    /**
     * Calls an action from the the opener if the opener is known.
     * 
     * @param pEvent the source event
     * @param pAction the specified action name
     * @throws Throwable if the call throws an error
     */
    private void callAction(UIEvent pEvent, String pAction) throws Throwable
    {
        if (pAction != null && opener != null)
        {
            IActionListener listener = eventMessage.createListener(opener, pAction);
            listener.action(new UIActionEvent(this,
                                              UIActionEvent.ACTION_PERFORMED,
                                              pEvent.getWhen(),
                                              pEvent.getModifiers(),
                                              null));
        }
    }
    
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Actions
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    /**
     * Closes the message when the OK/Yes button was clicked and redispatches
     * the event to the {@link #getOpener()}, if an action is defined for the 
     * OK/Yes button.
     * 
     * @param pEvent the event from the button
     * @throws Throwable if the an action is configured but the call throws an error
     */
    public void doOk(UIActionEvent pEvent) throws Throwable
    {
        dispose();
        
        callAction(pEvent, sOkAction);
    }
    
    /**
     * Closes the message when the Cancel/No button was clicked and redispatches
     * the event to the {@link #getOpener()}, if an action is defined for the 
     * Cancel/No button.
     * 
     * @param pEvent the event from the button
     * @throws Throwable if the a cancel action is configured but the call throws an error
     */
    public void doCancel(UIActionEvent pEvent) throws Throwable
    {
        dispose();

        callAction(pEvent, sCancelAction);
    }
    
    /**
     * Closes the message when the x button in the title was clicked. If a cancel
     * action was defined, it will be called.
     * 
     * @param pEvent the event from the frame
     * @throws Throwable if the a cancel action is configured but the call throws an error
     */
    public void doClose(UIWindowEvent pEvent) throws Throwable
    {
        dispose();
        
        callAction(pEvent, sCancelAction);
    }
    
}   // Message
