/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 10.11.2008 - [HM] - creation
 * 27.03.2009 - [JR] - uninstallEditor: unset databook
 * 04.08.2009 - [JR] - popupMenuWillBecomeVisible, focusGained: changed selection (see REV 2838)
 *                     (focusGained should not always select because you can't input text without overwriting)
 * 24.03.2011 - [JR] - #317: cancelEditing checks parents enabled state
 * 31.03.2011 - [JR] - #161: forward translation change to combobase
 * 11.04.2014 - [JR] - #1006: automatically show/hide header
 * 22.04.2014 - [RZ] - #1014: implemented displayReferencedColumnName
 * 18.09.2014 - [RZ] - #1111: fixed that the cell editor will become empty upon invalid input
 */
package com.sibvisions.rad.ui.swing.ext.celleditor;

import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.EventQueue;
import java.awt.Font;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.List;

import javax.rad.model.ColumnDefinition;
import javax.rad.model.IDataBook;
import javax.rad.model.IDataPage;
import javax.rad.model.IDataRow;
import javax.rad.model.IRowDefinition;
import javax.rad.model.ModelException;
import javax.rad.model.SortDefinition;
import javax.rad.model.condition.CompareCondition;
import javax.rad.model.condition.Equals;
import javax.rad.model.condition.ICondition;
import javax.rad.model.condition.Like;
import javax.rad.model.condition.LikeIgnoreCase;
import javax.rad.model.condition.OperatorCondition;
import javax.rad.model.reference.ColumnMapping;
import javax.rad.model.reference.ReferenceDefinition;
import javax.rad.model.ui.ICellEditor;
import javax.rad.model.ui.ICellEditorHandler;
import javax.rad.model.ui.ICellEditorListener;
import javax.rad.model.ui.ICellRenderer;
import javax.rad.model.ui.IEditorControl;
import javax.rad.ui.IAlignmentConstants;
import javax.rad.ui.IColor;
import javax.rad.ui.celleditor.ILinkedCellEditor;
import javax.swing.AbstractListModel;
import javax.swing.BorderFactory;
import javax.swing.ComboBoxModel;
import javax.swing.JComponent;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.PopupMenuEvent;
import javax.swing.event.PopupMenuListener;
import javax.swing.table.DefaultTableCellRenderer;

import com.sibvisions.rad.ui.awt.impl.AwtDimension;
import com.sibvisions.rad.ui.celleditor.AbstractLinkedCellEditor;
import com.sibvisions.rad.ui.swing.ext.ICellFormatterEditorListener;
import com.sibvisions.rad.ui.swing.ext.JVxComboBase;
import com.sibvisions.rad.ui.swing.ext.JVxTable;
import com.sibvisions.rad.ui.swing.ext.JVxUtil;
import com.sibvisions.rad.ui.swing.ext.cellrenderer.JVxIconRenderer;
import com.sibvisions.rad.ui.swing.ext.cellrenderer.JVxRendererContainer;
import com.sibvisions.rad.ui.swing.ext.format.CellFormat;
import com.sibvisions.rad.ui.swing.ext.layout.JVxBorderLayout;
import com.sibvisions.rad.ui.swing.impl.SwingFactory;
import com.sibvisions.util.ArrayUtil;

/**
 * The <code>JVxLinkedCellEditor</code> provides the generation of the 
 * physical linked editor component, handles correct all events, and 
 * gives standard access to edited values.
 * 
 * @author Martin Handsteiner
 */
public class JVxLinkedCellEditor extends AbstractLinkedCellEditor 
                                 implements ICellRenderer<JComponent>
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** The default Renderer anyway!. */
	private JVxRendererContainer cellRenderer = null;
	/** The text renderer. */
	private DefaultTableCellRenderer textRenderer = null;
	/** The text renderer. */
	private JVxIconRenderer iconRenderer = null;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Constructs a new JVxLinkedCellEditor.
	 */
	public JVxLinkedCellEditor()
	{
		this(null);
	}
	
	/**
	 * Constructs a new JVxLinkedCellEditor with the given link reference.
	 * @param pLinkReference the link reference.
	 */
	public JVxLinkedCellEditor(ReferenceDefinition pLinkReference)
	{
		popupSize = new AwtDimension(400, 200);
		
		setLinkReference(pLinkReference);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	public ICellEditorHandler<JComponent> createCellEditorHandler(ICellEditorListener pCellEditorListener, 
			                                                      IDataRow pDataRow, String pColumnName)
	{
		return new CellEditorHandler(this, (ICellFormatterEditorListener)pCellEditorListener, pDataRow, pColumnName, displayReferencedColumnName);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public JComponent getCellRendererComponent(JComponent pParentComponent, IDataPage pDataPage, int pRowNumber, IDataRow pDataRow, String pColumnName, boolean pIsSelected,
			boolean hasFocus)
	{
		if (cellRenderer == null)
		{
			cellRenderer = new JVxRendererContainer();
			
			textRenderer = new DefaultTableCellRenderer();
			textRenderer.setFont(null);
			textRenderer.setOpaque(false);
			
			iconRenderer = new JVxIconRenderer();
			iconRenderer.setImage(JVxUtil.getImage("/com/sibvisions/rad/ui/swing/ext/images/combobox.png"));
			iconRenderer.setOpaque(false);
			
			cellRenderer.add(textRenderer, JVxBorderLayout.CENTER);
			cellRenderer.add(iconRenderer, JVxBorderLayout.EAST);
		}
		
		textRenderer.setHorizontalAlignment(SwingFactory.getHorizontalSwingAlignment(getHorizontalAlignment()));
		textRenderer.setVerticalAlignment(SwingFactory.getVerticalSwingAlignment(getVerticalAlignment()));

		IDataBook dataBook = pDataPage.getDataBook();
		
		try
		{
			iconRenderer.setVisible(pIsSelected && !dataBook.isReadOnly() && dataBook.isUpdateAllowed()
					&& !dataBook.getRowDefinition().getColumnDefinition(pColumnName).isReadOnly());
		}
		catch (Exception ex)
		{
			iconRenderer.setVisible(false);
		}
		
		try
		{
			textRenderer.setText(getDisplayValue(pDataRow, pColumnName));
		}
		catch (Exception pException)
		{
			textRenderer.setText(null);
		}

		return cellRenderer;
	}
	
	//****************************************************************
	// Subclass definition
	//****************************************************************

	/**
     * Sets the internal changed Flag, and informs the CellEditorListener 
     * if editing is completed.
     * 
     * @author Martin Handsteiner
     */
    public static class CellEditorHandler extends AbstractListModel
                                          implements ICellEditorHandler<JComponent>,
                                          			 ComboBoxModel,
                                          			 DocumentListener,
                                          			 KeyListener,
                                        			 PopupMenuListener,
                                                     FocusListener,
                                                     MouseListener,
                                                     Runnable
    {
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	// Class members
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    	/** The CellEditor, that created this handler. */
    	private JVxLinkedCellEditor cellEditor;
    	
    	/** The CellEditorListener to inform, if editing is started or completed. */
    	private ICellFormatterEditorListener cellEditorListener;
    	
    	/** The data row that is edited. */
    	private IDataRow dataRow;
    	
    	/** The column index. */
    	private int columnIndex;

    	/** The column names. */
    	private String[] columnNames;

    	/** The column names referenced. */
    	private String[] referencedColumnNames;

    	/** The column name of the edited column. */
    	private String columnName;

    	/** The column name referenced by the edited column. */
    	private String referencedColumnName;
    	
    	/** The search Columns. */
    	private String[] searchColumns;

    	/** The search Columns. */
    	private String[] additionalClearColumns;

    	/** The referenced search Columns. */
    	private String[] referencedSearchColumns;

    	/** The DataBook referenced by the edited column. */
    	private IDataBook referencedDataBook;

    	/** The physical component that is added to the parent container. */
    	private JVxComboBase cellEditorComponent;
    	
    	/** Dynamic alignment. */
    	private IAlignmentConstants dynamicAlignment = null;
    	
    	/** The table shown in the popup. */
    	private JVxTable table;
    	
    	/** Tells the listener to ignore the events. */
    	private boolean ignoreEvent = false;
    	
    	/** True, if it's the first editing started event. */
    	private boolean firstEditingStarted = true;
    	
    	/** True, if it's the first editing started event. */
    	private boolean popupChanged = false;
    	
    	/** The name of the displayed column. */
    	private String displayColumnName;
    	
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	// Initialization
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	
    	/**
    	 * Constructs a new CellEditorHandler.
    	 * 
    	 * @param pCellEditor the CellEditor that created this handler.
    	 * @param pCellEditorListener CellEditorListener to inform, if editing is started or completed.
    	 * @param pDataRow the data row that is edited.
    	 * @param pColumnName the column name of the edited column.
    	 * @param pDisplayColumnName the name of the displayed column.
    	 */
    	public CellEditorHandler(JVxLinkedCellEditor pCellEditor, ICellFormatterEditorListener pCellEditorListener,
    			                 IDataRow pDataRow, String pColumnName, String pDisplayColumnName)
    	{
    		cellEditor = pCellEditor;
    		cellEditorListener = pCellEditorListener;
    		dataRow = pDataRow;
    		columnName = pColumnName;
    		displayColumnName = pDisplayColumnName;
    		
    		columnNames = cellEditor.linkReference.getColumnNames();
    		referencedColumnNames = cellEditor.linkReference.getReferencedColumnNames();
    		if (columnNames.length == 0 && referencedColumnNames.length == 1)
    		{
    			columnNames = new String[] {pColumnName};
    		}

    		columnIndex = ArrayUtil.indexOf(columnNames, columnName);
    		if (columnIndex < 0)
    		{
    			throw new IllegalArgumentException("The edited column " + columnName + " has to be part of in column names list in LinkReference!");
    		}
    		else
    		{
        		referencedColumnName = referencedColumnNames[columnIndex];
    		}
    		
    		if (cellEditor.searchColumnMapping == null)
    		{
    			searchColumns = null;
    			referencedSearchColumns = null;
    		}
    		else
    		{
    			searchColumns = cellEditor.searchColumnMapping.getColumnNames();
    			referencedSearchColumns = cellEditor.searchColumnMapping.getReferencedColumnNames();
    		}
    		
       		referencedDataBook = cellEditor.linkReference.getReferencedDataBook();
    		try
			{
    			additionalClearColumns = searchForAdditionalClearColumns(dataRow.getRowDefinition(), columnNames, 
    					getConditionColumns(cellEditor.searchColumnMapping, cellEditor.additionalCondition));

    			referencedDataBook.setReadOnly(cellEditor.isTableReadonly());
			}
			catch (ModelException pModelException)
			{
				// Do nothing, it only tries to set data book readonly.
			}
    		referencedDataBook.setSelectionMode(IDataBook.SelectionMode.DESELECTED);
    		
    		table = new JVxTable();
    		table.setDataBook(referencedDataBook);
    		table.setBorder(BorderFactory.createEmptyBorder());
    		table.setAutoResize(true);
    		table.getJTable().setFocusable(false);
    		
    		cellEditorComponent = new JVxComboBase();
    		cellEditorComponent.setPopupComponent(table);
    		cellEditorComponent.setModel(this);
    		if (cellEditor.getPopupSize() != null)
    		{
    			cellEditorComponent.setPopupSize((Dimension)cellEditor.getPopupSize().getResource());
    		}
    		
    		if (cellEditorComponent.getEditorComponent() instanceof JTextField)
    		{ 
    			if (cellEditorListener.getControl() instanceof IAlignmentConstants && cellEditorListener.getControl() instanceof IEditorControl)
    			{	// use alignment of editors, if possible.
    				dynamicAlignment = (IAlignmentConstants)cellEditorListener.getControl();
    			}
    			else
    			{
	    			((JTextField)cellEditorComponent.getEditorComponent()).setHorizontalAlignment(SwingFactory.getHorizontalSwingAlignment(cellEditor.getHorizontalAlignment()));
    			}
        	}

    		cellEditorComponent.getEditorComponent().getDocument().addDocumentListener(this);
    		cellEditorComponent.getEditorComponent().addKeyListener(this);
    		cellEditorComponent.getEditorComponent().addFocusListener(this);
    		cellEditorComponent.getEditorComponent().setFocusTraversalKeysEnabled(false);
    		cellEditorComponent.addPopupMenuListener(this);
    		table.getJTable().addMouseListener(this);
    	}

    	/**
    	 * Searches the condition columns.
    	 * 
    	 * @param pColumnMapping the columnMapping.
    	 * @param pCondition the additional condition.
    	 * @return the list of all found condition columns.
    	 */
    	private String[] getConditionColumns(ColumnMapping pColumnMapping, ICondition pCondition)
    	{
    		ArrayUtil<String> conditionColumns = new ArrayUtil<String>();
    		
    		if (pColumnMapping != null)
    		{
    			conditionColumns.addAll(pColumnMapping.getColumnNames());
    		}
    		
    		fillInConditionColumns(pCondition, conditionColumns);
    		
    		int size = conditionColumns.size();
    		if (size == 0)
    		{
    			return null;
    		}
    		else
    		{
    			return conditionColumns.toArray(new String[size]);
    		}
    	}
    	
    	/**
    	 * Searches the condition columns.
    	 * 
    	 * @param pCondition the condition.
    	 * @param pConditionColumns the list of all found condition columns.
    	 */
    	private void fillInConditionColumns(ICondition pCondition, List<String> pConditionColumns)
    	{
    		if (pCondition instanceof CompareCondition)
    		{
    			String colName = ((CompareCondition)pCondition).getDataRowColumnName();
    			if (!pConditionColumns.contains(colName))
    			{
    				pConditionColumns.add(colName);
    			}
    		}
    		else if (pCondition instanceof OperatorCondition)
    		{
    			ICondition[] conditions = ((OperatorCondition)pCondition).getConditions();
    			
    			for (int i = 0; i < conditions.length; i++)
    			{
    				fillInConditionColumns(conditions[i], pConditionColumns);
    			}
    		}
    	}
    	
    	/**
    	 * Searches for additional columns to be cleared on save.
    	 * 
    	 * @param pRowDef the row definition.
    	 * @param pRefDefColumns the columns of this reference definition.
    	 * @param pSearchColumns the own search columns.
    	 * @return additional clear columns.
    	 * @throws ModelException if it fails.
    	 */
    	private String[] searchForAdditionalClearColumns(IRowDefinition pRowDef, String[] pRefDefColumns, String[] pSearchColumns) throws ModelException
    	{
    		ArrayUtil<String> columns = new ArrayUtil<String>();
    		
    		String[] allColumns = pRowDef.getColumnNames();
    		
    		for (String colName : allColumns)
    		{
    			ICellEditor ce = pRowDef.getColumnDefinition(colName).getDataType().getCellEditor();
    			
    			if (ce instanceof ILinkedCellEditor)
    			{
    				ILinkedCellEditor linkCe = (ILinkedCellEditor)ce;
    				ReferenceDefinition refDef = linkCe.getLinkReference();
    				String[] searchCols = ArrayUtil.removeAll(
    						getConditionColumns(linkCe.getSearchColumnMapping(), linkCe.getAdditionalCondition()), pSearchColumns);
    				
    				if (refDef != null && searchCols != null
    						&& ArrayUtil.intersect(pRefDefColumns, searchCols).length > 0)
    				{
   						String[] addCols = ArrayUtil.removeAll(refDef.getColumnNames(), pRefDefColumns);
	    						
						for (String col : addCols)
						{
							if (!columns.contains(col))
							{
								columns.add(col);
							}
						}
    				}
    			}
    		}
    		return columns.toArray(new String[columns.size()]);
    	}
    	
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	// Interface implementation
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    	// ICellEditorHandler
    	
    	/**
    	 * {@inheritDoc}
    	 */
    	public void uninstallEditor()
    	{
    		cellEditorComponent.getEditorComponent().getDocument().removeDocumentListener(this);
    		cellEditorComponent.getEditorComponent().removeKeyListener(this);
    		cellEditorComponent.getEditorComponent().removeFocusListener(this);
    		cellEditorComponent.removePopupMenuListener(this);
    		table.getJTable().removeKeyListener(this);
    		table.getJTable().removeMouseListener(this);
    		//important, to remove the table-control from the databook!
    		table.setDataBook(null);
    	}

    	/**
    	 * {@inheritDoc}
    	 */
    	public ICellEditor getCellEditor()
    	{
    		return cellEditor;
    	}

    	/**
    	 * {@inheritDoc}
    	 */
    	public ICellEditorListener getCellEditorListener()
    	{
    		return cellEditorListener;
    	}

    	/**
    	 * {@inheritDoc}
    	 */
    	public IDataRow getDataRow()
    	{
    		return dataRow;
    	}

    	/**
    	 * {@inheritDoc}
    	 */
    	public String getColumnName()
    	{
    		return columnName;
    	}

    	/**
    	 * {@inheritDoc}
    	 */
    	public JComponent getCellEditorComponent()
    	{
    		return cellEditorComponent;
    	}

    	/**
    	 * Creates a search string.
    	 * 
    	 * @param pItem item
    	 * @return a search string.
    	 */
    	private String getWildCardString(Object pItem)
    	{
    		if (cellEditor.isSearchTextAnywhere())
			{
				return "*" + pItem + "*";
			}
			else
			{
				return pItem + "*";
			}
    	}
    	
    	/**
    	 * Creates a Condition including the search columns.
    	 * 
    	 * @param pCondition the base condition.
    	 * @return a Condition including the search columns.
    	 */
    	private ICondition getSearchCondition(ICondition pCondition)
    	{
			if (pCondition == null)
			{
				pCondition = cellEditor.additionalCondition;
			}
			else if (cellEditor.additionalCondition != null)
			{
				pCondition = pCondition.and(cellEditor.additionalCondition);
			}
			
    		if (searchColumns != null)
    		{
    			for (int i = 0; i < searchColumns.length; i++)
    			{
    				ICondition condition = new Equals(dataRow, searchColumns[i], referencedSearchColumns[i]);
    				
    				if (pCondition == null)
    				{
    					pCondition = condition;
    				}
    				else
    				{
    					pCondition = pCondition.and(condition);
    				}
    			}
    		}
    		return pCondition;
    	}
    	
    	/**
    	 * Sets the values and clears the additionalClearColumns, if values are changed.
    	 * @param pNewValues the new values.
    	 * @throws ModelException if it fails.
    	 */
    	private void setValuesAndClearIfNecessary(Object[] pNewValues) throws ModelException
    	{
    		IDataRow oldRow = dataRow.createDataRow(columnNames);
    		
    		dataRow.setValues(columnNames, pNewValues);
    		
    		if (!dataRow.equals(oldRow, columnNames))
    		{
    			dataRow.setValues(additionalClearColumns, null);
    		}
    	}
    	
    	/**
    	 * {@inheritDoc}
    	 */
    	public void saveEditing() throws ModelException
    	{
    		if (popupChanged && referencedDataBook.getSelectedRow() >= 0)
			{
    			setValuesAndClearIfNecessary(referencedDataBook.getValues(referencedColumnNames));
			}
			else
			{
	    		Object item = cellEditorComponent.getEditor().getItem();
	    		
	    		if (item == null || "".equals(item))
	    		{
					Object[] result = new Object[referencedColumnNames.length];
					
	    			setValuesAndClearIfNecessary(result);
	    		}
	    		else
	    		{
		    		referencedDataBook.setFilter(getSearchCondition(new Like(getRelevantSearchColumnName(), item)));
		    		if (referencedDataBook.getDataRow(0) != null && referencedDataBook.getDataRow(1) == null)
		    		{
		    			setValuesAndClearIfNecessary(referencedDataBook.getDataRow(0).getValues(referencedColumnNames));
		    		}
		    		else
		    		{
	    				if (cellEditor.isValidationEnabled())
	    				{
			        		referencedDataBook.setFilter(getSearchCondition(new LikeIgnoreCase(getRelevantSearchColumnName(), getWildCardString(item))));
			        		if (referencedDataBook.getDataRow(0) != null && referencedDataBook.getDataRow(1) == null)
			        		{
				    			setValuesAndClearIfNecessary(referencedDataBook.getDataRow(0).getValues(referencedColumnNames));
			    			}
			    			else
			    			{
				    			setValuesAndClearIfNecessary(dataRow.getValues(columnNames));
		    				}
		    			}
	    				else
	    				{
	    					Object[] result = new Object[referencedColumnNames.length];
	    					result[columnIndex] = item;
			    			setValuesAndClearIfNecessary(result);
	    				}
		    		}
	    		}
			}
			popupChanged = false;
    	}

    	/**
    	 * {@inheritDoc}
    	 */
    	public void cancelEditing() throws ModelException
    	{
    		if (!ignoreEvent)
    		{
	    		ignoreEvent = true;
	    		try
	    		{
	    			ColumnDefinition columnDef = dataRow.getRowDefinition().getColumnDefinition(columnName);
	    			
	        		cellEditorComponent.setTranslation(cellEditorListener.getControl().getTranslation());
	        		cellEditorComponent.setTranslationEnabled(cellEditorListener.getControl().isTranslationEnabled());

	        		cellEditorComponent.setSelectedItem(cellEditor.getDisplayValue(dataRow, columnName));
	    			
					CellFormat cellFormat = null;
					
					Container conParent = cellEditorComponent.getParent();
					
					boolean bParentEnabled = conParent == null || conParent.isEnabled();
					
		    		if (dataRow instanceof IDataBook)
		    		{
		    			IDataBook dataBook = (IDataBook)dataRow;
		    			boolean editable = bParentEnabled
		    					        && dataBook.isUpdateAllowed() 
		    							&& !columnDef.isReadOnly()
		    							&& dataBook.getSelectedRow() >= 0;
		    			if (editable && dataBook.getReadOnlyChecker() != null)
		    			{
		    				try
							{
		    					editable = !dataBook.getReadOnlyChecker().isReadOnly(dataBook, dataBook.getDataPage(), dataBook, columnName, dataBook.getSelectedRow(), -1);
							}
							catch (Throwable pTh)
							{
								// Ignore
							}
		    			}
		    			cellEditorComponent.setEditorEditable(editable);

						if (cellEditorListener.getCellFormatter() != null)
						{
							try
							{
								cellFormat = cellEditorListener.getCellFormatter().getCellFormat(
										dataBook, dataBook.getDataPage(), dataBook, columnName, dataBook.getSelectedRow(), -1);
							}
							catch (Throwable pThrowable)
							{
								// Do nothing
							}
						}
		    		}
					else
					{
						cellEditorComponent.setEditorEditable(bParentEnabled && !columnDef.isReadOnly());
					}
		    		
					Color background;
					Color foreground;
					Font  font;
					if (cellFormat == null)
					{
						background = null;
						foreground = null;
						font = null;
					}
					else
					{
						background = cellFormat.getBackground();
						foreground = cellFormat.getForeground();
						font = cellFormat.getFont();
					}
					if (font == null)
					{
						font = ((Component)cellEditorListener).getFont();
					}
					if (foreground == null && ((Component)cellEditorListener).isForegroundSet())
					{
						foreground = ((Component)cellEditorListener).getForeground();
					}
					if (background == null && ((Component)cellEditorListener).isBackgroundSet())
					{
						background = ((Component)cellEditorListener).getBackground();
					}

					cellEditorComponent.getEditorComponent().setFont(font);
		    		if (cellEditorComponent.isEditorEditable())
		    		{
		    			if (background == null)
		    			{
			    			if (columnDef.isNullable())
			    			{
			    				background = JVxUtil.getSystemColor(IColor.CONTROL_BACKGROUND);
			    			}
			    			else
			    			{
			    				background = JVxUtil.getSystemColor(IColor.CONTROL_MANDATORY_BACKGROUND);
			    			}
		    			}
		    			if (cellEditorComponent.getEditorComponent().hasFocus())
						{
							cellEditorComponent.getEditorComponent().selectAll();
						}
						else
						{
							cellEditorComponent.getEditorComponent().select(0, 0);
						}
		    		}
		    		else
		    		{
		    			background = JVxUtil.getSystemColor(IColor.CONTROL_READ_ONLY_BACKGROUND);
						cellEditorComponent.getEditorComponent().select(0, 0);
		    		}
	    			cellEditorComponent.getEditorComponent().setBackground(background);
	    			cellEditorComponent.getEditorComponent().setForeground(foreground);
	    			cellEditorComponent.setBackground(background); // Synthetica Look&Feel ignores the editor colors.
	    			cellEditorComponent.setForeground(foreground);
		
	    			if (dynamicAlignment != null)
		    		{
		    			int hAlign = dynamicAlignment.getHorizontalAlignment();
		    			if (hAlign == IAlignmentConstants.ALIGN_DEFAULT)
		    			{
		    				hAlign = cellEditor.getHorizontalAlignment();
		    			}
		    			((JTextField)cellEditorComponent.getEditorComponent()).setHorizontalAlignment(SwingFactory.getHorizontalSwingAlignment(hAlign));
		    		}
		    		
		    		if (conParent instanceof JComponent)
		    		{
		    			cellEditorComponent.getEditorComponent().putClientProperty("tabIndex", ((JComponent)conParent).getClientProperty("tabIndex"));
		    		}

	    			// Forward the Cell Formatter of the underlaying Control to the lookup table.
	           		table.setCellFormatter(cellEditorListener.getCellFormatter());
	    		}
	    		catch (Exception pException)
	    		{
	    			cellEditorComponent.setSelectedItem(null);
	    			cellEditorComponent.setEditorEditable(false);
	    			cellEditorComponent.getEditorComponent().setBackground(JVxUtil.getSystemColor(IColor.CONTROL_READ_ONLY_BACKGROUND));
	    			
	    			throw new ModelException("Editor cannot be restored!", pException);
	    		}
	    		finally
	    		{
	    			firstEditingStarted = true;
	    			ignoreEvent = false;
	    		}
    		}
    	}
    	
    	// ComboBoxModel
    	
		/**
		 * {@inheritDoc}
		 */
		public Object getSelectedItem()
		{
			try
			{
				if (popupChanged && referencedDataBook.getSelectedRow() >= 0)
				{
					return referencedDataBook.getValue(referencedColumnName);
				}
			}
			catch (Exception ex)
			{
				// Cancel
			}
			
	    	//cellEditorComponent.setPopupCanceled(true);
			return cellEditorComponent.getEditor().getItem();
		}

		/**
		 * {@inheritDoc}
		 */
		public void setSelectedItem(Object pItem)
		{
			try
			{
				if (firstEditingStarted)
				{
					referencedDataBook.setFilter(getSearchCondition(null));
					if (cellEditor.isSortByColumnName())
					{
						referencedDataBook.setSort(new SortDefinition(referencedColumnName));
					}
					
					if (pItem != null)
					{
						String[] compareColumns = new String[] {referencedColumnName};
						IDataRow searchRow = referencedDataBook.createEmptyDataRow(null);
						searchRow.setValue(referencedColumnName, pItem);
						
						long start = System.currentTimeMillis();
						
						referencedDataBook.setSelectedRow(-1);
						int i = 0;
						IDataRow row = referencedDataBook.getDataRow(i);
						while (row != null && referencedDataBook.getSelectedRow() < 0 && System.currentTimeMillis() - start < 1000)
						{
							if (searchRow.equals(row, compareColumns))
							{
								referencedDataBook.setSelectedRow(i);
							}
							else
							{
								i++;
								row = referencedDataBook.getDataRow(i);
							}
						}
					}
				}
				else
				{
					if (pItem == null)
					{
						referencedDataBook.setFilter(getSearchCondition(null));
					}
					else
					{
						referencedDataBook.setFilter(getSearchCondition(new LikeIgnoreCase(getRelevantSearchColumnName(), getWildCardString(pItem))));
					}
				}

	    		table.setColumnView(cellEditor.getColumnView());
	    		if (cellEditor.autoTableHeaderVisibility)
	    		{
	    		    table.setTableHeaderVisible(referencedDataBook.getDataRow(0) != null && table.getColumnView().getColumnCount() > 2);
	    		}
	    		else
	    		{
	    			table.setTableHeaderVisible(referencedDataBook.getDataRow(0) != null && cellEditor.isTableHeaderVisible());
	    		}
			}
			catch (ModelException pModelException)
			{
				// Do Nothing
			}
			popupChanged = false;
		}

		/**
		 * {@inheritDoc}
		 */
		public Object getElementAt(int pIndex)
		{
			return null;
		}

		/**
		 * {@inheritDoc}
		 */
		public int getSize()
		{
			return 0;
		}

    	// DocumentListener
    	
    	/**
    	 * {@inheritDoc}
    	 */
        public void insertUpdate(DocumentEvent pDocumentEvent) 
        {
       		if (!(EventQueue.getCurrentEvent() instanceof FocusEvent))
       		{
       			fireEditingStarted();
       			if (cellEditorComponent.isPopupVisible())
       			{
       				cellEditorComponent.setPopupVisible(true);
       			}
        	}
        }
        
    	/**
    	 * {@inheritDoc}
    	 */
        public void removeUpdate(DocumentEvent pDocumentEvent) 
        {
       		if (!(EventQueue.getCurrentEvent() instanceof FocusEvent))
       		{
       			fireEditingStarted();
       			if (cellEditorComponent.isPopupVisible())
       			{
       				cellEditorComponent.setPopupVisible(true);
       			}
        	}
        }
        
    	/**
    	 * {@inheritDoc}
    	 */
        public void changedUpdate(DocumentEvent pDocumentEvent) 
        {
        }

    	// KeyListener
    	
    	/**
    	 * {@inheritDoc}
    	 */
		public void keyPressed(KeyEvent pKeyEvent)
		{
			if (!pKeyEvent.isConsumed())
			{
				switch (pKeyEvent.getKeyCode())
				{
					case KeyEvent.VK_ESCAPE: 
						pKeyEvent.consume(); 
						cellEditorComponent.setPopupCanceled(true);
						cellEditorComponent.setPopupVisible(false);
				        fireEditingComplete(ICellEditorListener.ESCAPE_KEY, true);
				        break;
					case KeyEvent.VK_ENTER: 
						pKeyEvent.consume(); 
						if (cellEditorComponent.isPopupVisible())
						{
							// Force closing, the popup and write back the popup selection.
							popupChanged = true;
							fireEditingStarted();
							cellEditorComponent.setPopupVisible(false);
						}
						if (pKeyEvent.isShiftDown())
						{
					        fireEditingComplete(ICellEditorListener.SHIFT_ENTER_KEY, true);
						}
						else
						{
					        fireEditingComplete(ICellEditorListener.ENTER_KEY, true);
						}
				        break;
					case KeyEvent.VK_TAB: 
						pKeyEvent.consume(); 
						if (pKeyEvent.isShiftDown())
						{
					        fireEditingComplete(ICellEditorListener.SHIFT_TAB_KEY, true);
						}
						else
						{
					        fireEditingComplete(ICellEditorListener.TAB_KEY, true);
						}
				        break;
					case KeyEvent.VK_DOWN: 
					case KeyEvent.VK_UP: 
						if (cellEditorComponent.isPopupVisible())
						{
							table.keyPressed(pKeyEvent);
						}
				        break;
				    default:
				    	// Nothing to do
				}
			}
		}
		
    	/**
    	 * {@inheritDoc}
    	 */
		public void keyReleased(KeyEvent pKeyEvent)
		{
			if (!pKeyEvent.isConsumed())
			{
				switch (pKeyEvent.getKeyCode())
				{
					case KeyEvent.VK_DOWN: 
					case KeyEvent.VK_UP: 
						if (cellEditorComponent.isPopupVisible())
						{
							table.keyReleased(pKeyEvent);
						}
				        break;
				    default:
				    	// Nothing to do
				}
			}
		}
		
    	/**
    	 * {@inheritDoc}
    	 */
		public void keyTyped(KeyEvent pKeyEvent)
		{
		}

    	// PopupMenuListener
    	
    	/**
    	 * {@inheritDoc}
    	 */
		public void popupMenuCanceled(PopupMenuEvent pPopupMenuEvent)
		{
			fireEditingComplete(ICellEditorListener.ESCAPE_KEY, false);
		}

    	/**
    	 * {@inheritDoc}
    	 */
		public void popupMenuWillBecomeInvisible(PopupMenuEvent pPopupMenuEvent)
		{
			cellEditorComponent.getEditorComponent().select(
					cellEditorComponent.getEditorComponent().getSelectionStart(),
					cellEditorComponent.getEditorComponent().getSelectionEnd());
		}

    	/**
    	 * {@inheritDoc}
    	 */
		public void popupMenuWillBecomeVisible(PopupMenuEvent pPopupMenuEvent)
		{
    		try
			{
				referencedDataBook.setReadOnly(cellEditor.isTableReadonly());
			}
			catch (ModelException pModelException)
			{
				// Do nothing, it only tries to set data book readonly.
			}

			if (firstEditingStarted)
			{
				cellEditorComponent.getEditorComponent().selectAll();
			}

       		SwingUtilities.invokeLater(this);
		}
		
    	/**
    	 * {@inheritDoc}
    	 */
		public void run()
		{
	   		table.scrollToSelectedCell();
		}

    	// FocusListener
    	
    	/**
    	 * {@inheritDoc}
    	 */
		public void focusGained(FocusEvent pFocusEvent)
		{
			if (!cellEditorComponent.isPopupVisible() && cellEditorComponent.isEditorEditable())
			{
				cellEditorComponent.getEditorComponent().selectAll();
			}
		}

    	/**
    	 * {@inheritDoc}
    	 */
		public void focusLost(FocusEvent pFocusEvent)
		{
			if (!pFocusEvent.isTemporary() && !cellEditorComponent.isPopupFocusEvent(pFocusEvent))
			{
				fireEditingComplete(ICellEditorListener.FOCUS_LOST, true);
			}
		}
		
    	/**
    	 * {@inheritDoc}
    	 */
		public void mouseClicked(MouseEvent pMouseEvent)
		{
		}

    	/**
    	 * {@inheritDoc}
    	 */
		public void mouseEntered(MouseEvent pMouseEvent)
		{
		}

    	/**
    	 * {@inheritDoc}
    	 */
		public void mouseExited(MouseEvent pMouseEvent)
		{
		}

    	/**
    	 * {@inheritDoc}
    	 */
		public void mousePressed(MouseEvent pMouseEvent)
		{
		}

    	/**
    	 * {@inheritDoc}
    	 */
		public void mouseReleased(MouseEvent pMouseEvent)
		{
			// Force closing, the popup and write back the popup selection.
			popupChanged = true;
			fireEditingStarted();
			fireEditingComplete(ICellEditorListener.ENTER_KEY, true);
		}
    	
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    	// User-defined methods
    	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

		/**
		 * Delegates the event to the ICellEditorListener.
		 * It takes care, that the event occurs only one time.
		 */
		protected void fireEditingStarted()
		{
        	if (firstEditingStarted && !ignoreEvent && cellEditorListener != null)
        	{
           		firstEditingStarted = false;
           		cellEditorListener.editingStarted();
				if (cellEditor.isAutoOpenPopup() && !cellEditorComponent.isPopupVisible())
				{
					cellEditorComponent.setPopupVisible(true);
				}

        	}
		}
		
		/**
		 * Delegates the event to the ICellEditorListener.
		 * It takes care, that editing started will be called before,
		 * if it is not called until jet.
		 * 
		 * @param pCompleteType the editing complete type.
		 * @param pClosePopup try closing the popup.
		 */
		protected void fireEditingComplete(String pCompleteType, boolean pClosePopup)
		{
			if (!ignoreEvent && cellEditorListener != null)
			{
				if (pClosePopup && cellEditorComponent.isPopupVisible())
				{
					ignoreEvent = true;
					cellEditorComponent.setPopupCanceled(true);
					cellEditorComponent.setPopupVisible(false);
					ignoreEvent = false;
				}
				cellEditorListener.editingComplete(pCompleteType);
			}
		}

		/**
		 * Returns the relevant search column. If a {@code displayColumnName} is set,
		 * then that one will be returned, otherwise it will return {@code referencedColumnName}.
		 * 
		 * @return the relevant search column name.
		 */
		private String getRelevantSearchColumnName()
		{
			if (displayColumnName != null)
			{
				return displayColumnName;
			}
			else
			{
				return referencedColumnName;
			}
		}

    }	// CellEditorHandler

}	// JVxLinkedCellEditor
