/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 01.10.2008 - [JR] - creation
 * 10.02.2009 - [JR] - getClassNameMapping: added full qualified class name to mapping [BUGFIX]
 * 11.02.2009 - [JR] - createObject: AExchangeManager included
 * 12.02.2009 - [JR] - TYPE_SESSION splitted to TYPE_SUBSESSION and TYPE_MASTERSESSION
 * 10.05.2009 - [JR] - getObject: support for dot separated object names
 * 18.11.2009 - [JR] - #33: putObject implemented
 * 27.01.2010 - [JR] - invoke implemented
 * 15.02.2010 - [JR] - createInstance: checked GenericBean instance
 * 22.02.2010 - [JR] - #67: getObject: Map support
 *                   - getObject: object not found throws an exception [BUGFIX]
 * 05.03.2010 - [JR] - invoke: user friendly message when the object is null
 * 23.03.2010 - [JR] - #103: getObject now sets the object name and the method
 * 24.03.2010 - [JR] - #105: putObject: dot notation support
 *                   - Map instead of Bean
 * 22.12.2010 - [JR] - initApplicationObject: check ClassNotFoundException to allow missing application objects
 * 25.12.2010 - [JR] - removed final from class 
 * 01.03.2011 - [JR] - initSessionObject: injectObjects after caching the object 
 *                                        (allows LCO access for inject objects constructor)
 * 02.03.2011 - [JR] - #297: updateSessionObject (put object even if it exists -> support object changing)
 * 25.05.2011 - [JR] - getSessionObjectInternal implemented  
 * 26.05.2011 - [JR] - #363: ILifeCycleObject handling  
 * 21.11.2012 - [JR] - #535: check object and method access via IObjectAccessProvider
 * 23.08.2013 - [JR] - #774: check if session is available after method invocation
 * 29.01.2014 - [JR] - #935: session isolation feature    
 * 11.05.2014 - [JR] - #1033: Replacement annotation checked (object access)      
 * 09.09.2014 - [JR] - #1105: implicit LCO support     
 * 12.09.2014 - [JR] - #1106: strict isolation support     
 * 16.10.2014 - [JR] - #1144: Replacement annotation checked (action calls)      
 * 18.12.2014 - [JR] - #1217: removed object id 
 * 09.03.2015 - [JR] - #965: Used existing annotations
 * 29.05.2015 - [JR] - #1397: notify callhandler about object creation
 */
package com.sibvisions.rad.server;

import java.lang.ref.WeakReference;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.WeakHashMap;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.rad.server.AbstractObjectProvider;
import javax.rad.server.ISession;
import javax.rad.server.InjectObject;
import javax.rad.server.SessionContext;
import javax.rad.server.UnknownObjectException;
import javax.rad.server.event.ISessionListener;
import javax.rad.type.bean.Bean;

import com.sibvisions.rad.server.annotation.Accessible;
import com.sibvisions.rad.server.annotation.NotAccessible;
import com.sibvisions.rad.server.annotation.Replacement;
import com.sibvisions.rad.server.annotation.StrictIsolation;
import com.sibvisions.rad.server.config.Configuration;
import com.sibvisions.rad.server.security.IObjectAccessController;
import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.Reflective;
import com.sibvisions.util.log.ILogger;
import com.sibvisions.util.log.LoggerFactory;
import com.sibvisions.util.type.CommonUtil;
import com.sibvisions.util.type.StringUtil;

/**
 * The <code>DefaultObjectProvider</code> manages the remote accessible objects. It compiles
 * source files and offers always the current object.
 * 
 * @author Ren Jahn
 */
public class DefaultObjectProvider extends AbstractObjectProvider
                                   implements ISessionListener
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/** the logger instance. */
	private ILogger log = LoggerFactory.getInstance(getClass());
	
	/** the cache for application life cycle objects. */
	private Hashtable<String, Map> htApplicationObjects = null;
	
	/** the cache for session life cycle objects. */
	private Hashtable<Object, Map> htSessionObjects = null;
	
	/** the object access controller. */
	private IObjectAccessController oaController = null;
	
	/** cache of construct methods. */
	private WeakHashMap<Class<?>, WeakReference<Method>> whmpConstruct;
	
	/** cache of pre-destroy methods. */
	private WeakHashMap<Class<?>, List<WeakReference<Method>>> whmpDestroy;

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Creates an instance of <code>AbstractObjectProvider</code>.
	 * 
	 * @param pServer communication server
	 */
	protected DefaultObjectProvider(Server pServer)
	{
		super(pServer);
		
		pServer.getSessionManager().addSessionListener(this);

		//#535
		try
		{
			String sObjectAccess = Configuration.getServerZone().getProperty("/server/objectprovider/accesscontroller");
		
			if (!StringUtil.isEmpty(sObjectAccess))
			{
				oaController = (IObjectAccessController)Reflective.construct(sObjectAccess);

				log.debug("Use ", sObjectAccess, " as ObjectAccessController");
			}
		}
		catch (Throwable th)
		{
			log.debug("Can't use configured ObjectAccessController!", th);
			
			oaController = null;
		}
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	public void sessionCreated(ISession pSession)
	{
	}

	/**
	 * {@inheritDoc}
	 */
	public void sessionDestroyed(ISession pSession)
	{
		destroySession(pSession);
	}

	/**
	 * {@inheritDoc}
	 */
	public Object getObject(ISession pSession, String pObjectName) throws Throwable
	{
	    synchronized (pSession)
	    {
    		Map mapLifeCycle = getSessionObject(pSession);
    		
    		//#1105
    		if (mapLifeCycle instanceof ImplicitLifeCycleObject)
    		{
    		    throw new ClassNotFoundException("Missing instance name");    		    
    		}
    		
    		//search the object name within the life-cycle object
    		
    		if (pObjectName == null || pObjectName.trim().length() == 0)
    		{
    			if (mapLifeCycle == null)
    			{
    				throw new UnknownObjectException(pSession.getLifeCycleName());
    			}
    			
    			//an action call doesn't need an object name
    			return mapLifeCycle;
    		}
    		else
    		{
    			//search the desired object
    			ArrayUtil<String> auNames = StringUtil.separateList(pObjectName, ".", true);
    			
    			//get the callable object
    			Object oInvoke = mapLifeCycle;
    			Object oResult;
    			String sObjectName;
    			
    			StringBuilder sbCurrentObjectName = new StringBuilder();
    			
    			SessionContextImpl context = ((SessionContextImpl)SessionContext.getCurrentInstance());
    			
    			String sOriginalMethodName = context.getMethodName();
    			
    			context.setMethodName(null);
    			
    			IObjectAccessController controller = getObjectAccessController();
    			
    			for (int i = 0, anz = auNames.size(); i < anz; i++)
    			{
    				sObjectName = auNames.get(i);
    				
    				if (sbCurrentObjectName.length() > 0)
    				{
    					sbCurrentObjectName.append(".");
    				}
    				
    				sbCurrentObjectName.append(sObjectName);
    				
    				context.setObjectName(sbCurrentObjectName.toString());
    				
    				if (i == anz - 1)
    				{
    					context.setMethodName(sOriginalMethodName);
    				}
    
    				if (oInvoke == null)
    				{
    					throw new UnknownObjectException(sbCurrentObjectName.toString());
    				}
    				
    				try
    				{
    				    String sMethodName = StringUtil.formatMethodName("get", sObjectName);
    				    
                        Method met = Reflective.getMethod(oInvoke.getClass(), sMethodName); 

                        if (i > 0)
    				    {
        				    oInvoke = invokeSubMethod(met, oInvoke, sbCurrentObjectName);
    				    }
    				    else
    				    {
                            oInvoke = invokeMethod(met, oInvoke, sbCurrentObjectName);
    				    }
    				}
    				catch (NoSuchMethodException nsme)
    				{
    				    boolean bFound = false;
    				    
                        //first object: check replacements
                        if (i == 0)
                        {
                            Replacement replace;
                        
                            Method[] methods = oInvoke.getClass().getMethods();
                            
                            if (methods != null)
                            {
                                for (int j = 0; j < methods.length && !bFound; j++)
                                {                                    
                                    if (methods[j].getParameterTypes().length == 0 && methods[j].getReturnType() != Void.TYPE)
                                    {
                                        replace = methods[j].getAnnotation(Replacement.class);
                                        
                                        if (replace != null)
                                        {
                                            if (sObjectName.equals(replace.name()))
                                            {
                                                if (i > 0)
                                                {
                                                    oInvoke = invokeSubMethod(methods[j], oInvoke, sbCurrentObjectName);
                                                }
                                                else
                                                {
                                                    oInvoke = invokeMethod(methods[j], oInvoke, sbCurrentObjectName);
                                                }
                                                
                                                bFound = true;
                                            }
                                        }
                                    }
                                }
                            }
                        }

                        if (!bFound)
                        {
                            if (oInvoke instanceof Map)
        					{
        						oResult = ((Map)oInvoke).get(sObjectName);
        						
        						if (oResult == null && !((Map)oInvoke).containsKey(sObjectName))
        						{
        							throw new UnknownObjectException(sObjectName);
        						}
        						
        						//use the result!
        						oInvoke = oResult;
        					}
        					else
        					{
        						throw new UnknownObjectException(sObjectName, nsme);
        					}
                        }
    				}
    				
    				//#535
    				if (controller != null && !controller.isObjectAccessAllowed(this, pSession, mapLifeCycle, sbCurrentObjectName.toString(), oInvoke))
    				{
    					throw new SecurityException("Access to '" + sObjectName + "' is denied!");
    				}
    			}
    			
    			//don't check null, because a getXXX method exists and returns null (maybe expected)!
    			return oInvoke;
    		}
	    }
	}
	
	/**
	 * {@inheritDoc}
	 */
	public Object putObject(ISession pSession, String pObjectName, Object pObject) throws Throwable
	{
		if (pObjectName == null)
		{
			return null;
		}
		
		synchronized (pSession)
		{
    		Object objSession;
    		
    		String sObjectName;
    		
    		int iPos = pObjectName.lastIndexOf(".");
    		
    		if (iPos > 0)
    		{
    			SessionContextImpl context = ((SessionContextImpl)SessionContext.getCurrentInstance()); 
    			
    			//cache the old method name because the getObject method changes the method name, and in that special
    			//case it's not correct to set a method name!
    			String sOldMethodName = context.getMethodName();
    			
    			context.setMethodName(null);
    			
    			objSession = getObject(pSession, pObjectName.substring(0, iPos));
    			
    			context.setMethodName(sOldMethodName);
    			
    			sObjectName = pObjectName.substring(iPos + 1);
    		}
    		else
    		{
    			objSession = getSessionObject(pSession);
    			 
    			sObjectName = pObjectName;
    		}
    		
    		try
    		{
    			return Reflective.call(objSession, StringUtil.formatMethodName("set", sObjectName), pObject);
    		}
    		catch (NoSuchMethodException nsme)
    		{
    			if (objSession instanceof Map)
    			{
    				return ((Map)objSession).put(sObjectName, pObject);
    			}
    			else
    			{
    				throw new RuntimeException("Can't set object '" + pObjectName + "'", nsme);
    			}
    		}
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public Object invoke(ISession pSession, String pObjectName, String pMethodName, Object... pParams) throws Throwable
	{
		Object obj = getObject(pSession, pObjectName);
		
		try
		{
			//it's possible that a null object will be returned as valid object, but that's not a valid call!
			if (obj == null)
			{
				throw new RuntimeException("The Object '" + pObjectName + "' is known but 'null' was returned!");
			}
			
			//#535
			IObjectAccessController controller = getObjectAccessController();
			
			if (controller != null && !controller.isMethodInvocationAllowed(this, pSession, pObjectName, obj, pMethodName, pParams))
			{
				throw new SecurityException("Invocation of '" + pMethodName + "' is not allowed!");
			}
			
			try
			{
    			if (obj instanceof GenericBean)
    			{
    				//try to call the action generic
    				return ((GenericBean)obj).invoke(pMethodName, pParams);
    			}
    			else
    			{
    				//call the action by name, and all methods because we are not a generic bean
    				return Reflective.call(obj, pMethodName, pParams);
    			}
			}
			catch (NoSuchMethodException nsme)
			{
                //check replacements
                Replacement replace;
            
                Method[] methods = obj.getClass().getDeclaredMethods();
                
                if (methods != null)
                {
                    for (int i = 0; i < methods.length; i++)
                    {                                    
                        replace = methods[i].getAnnotation(Replacement.class);
                        
                        if (replace != null)
                        {
                            if (pMethodName.equals(replace.name()))
                            {
                                return Reflective.call(obj, true, methods[i].getName(), pParams);
                            }
                        }
                    }
                }
			    
			    throw nsme;
			}
		}
		finally
		{
			//if this call was an async call and the session is not longer valid, ensure that all objects are removed
			if (!getServer().getSessionManager().isAvailable(pSession))
			{
				destroySession(pSession);
			}
		}
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Server getServer()
	{
		return (Server)super.getServer();
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Gets the life-cycle object for a session. 
	 * 
	 * @param pSession the accessing session 
	 * @return the life-cycle object for the session
	 * @throws Exception if the life-cycle object can not be created
	 */
	protected Map getSessionObject(ISession pSession) throws Exception
	{
	    synchronized (pSession)
	    {
    		Map mapSession = null;
    		
    		AbstractSession session = (AbstractSession)pSession;
    		
    		if (htSessionObjects != null)
    		{
    			mapSession = htSessionObjects.get(session.getId());
    		}
    		
    		if (mapSession == null)
    		{
    			mapSession = initSessionObject(session, true);
    		}
    		else
    		{
    			updateSessionObject(session, mapSession);
    		}
    		
    		return mapSession;
	    }
	}
	
	/**
	 * Creates and initializes a life-cycle object for a session.
	 * 
	 * @param pSession the accessing session 
	 * @param pNotifyCallHandler <code>true</code> to notify Session' call handler
	 * @return the life-cycle object for the session
	 * @throws Exception if the life-cycle object can not be created
	 */
	private Map initSessionObject(AbstractSession pSession, boolean pNotifyCallHandler) throws Exception
	{
	    synchronized (pSession)
	    {
    		if (pSession instanceof SubSession)
    		{
                Map mapSub = createInstance(pSession, pSession.getLifeCycleName());

                if (!isIsolated(mapSub))
                {
                    AbstractSession sessParent = ((SubSession)pSession).getMasterSession();
        			
        			Map mapMaster = null;
        			
    			    synchronized (sessParent)
    			    {
        				if (htSessionObjects != null)
        				{
        					mapMaster = htSessionObjects.get(sessParent.getId());
        				}
        				
        				//If the Master-session didn't execute commands -> the life-cycle object is not
        				//available. In that case it's important to create the object.
        				//But be careful! We need a SessionContext for the MasterSession, otherwise we have
        				//a SessionContext for the SubSession!
        				if (mapMaster == null)
        				{				
        					//object and method name are not known. Of course, we could check if a SessionContext
        					//is available, but it's better to give the life-cycle object always consistent values!
        					SessionContext context = sessParent.createSessionContext(null, null);
        					
        					try
        					{
        					    //DON'T notify call handler, because this would fire events and following postObjectCreation would
        					    //fire same events again
        						mapMaster = initSessionObject(sessParent, false);
        					}
        					finally
        					{
        						context.release();
        					}
        				}
    			    }

    			    setParent(pSession, mapSub, mapMaster);
                }
                
                if (pNotifyCallHandler)
                {
                    pSession.getCallHandler().postObjectCreation();
                }
                
    			if (htSessionObjects == null)
    			{
    				htSessionObjects = new Hashtable<Object, Map>();
    			}
    			
    			htSessionObjects.put(pSession.getId(), mapSub);
    
    			//inject after put to Hashtable -> that makes it possible that injected objects have access to
    			//the life-cycle object in the constructor
    			injectObjects(pSession, mapSub);
    			
    			return mapSub;
    		}
    		else
    		{
                Map mapMaster;

                String sLCOName = pSession.getLifeCycleName();
                
                //#1105
                if (!StringUtil.isEmpty(sLCOName))
                {
                    mapMaster = createInstance(pSession, sLCOName);
                }
                else
                {
                    mapMaster = new ImplicitLifeCycleObject();
                }

                if (!isIsolated(mapMaster))
                {
        			setParent(pSession, mapMaster, getApplicationObject(pSession));
                }    			
    	        
    			if (htSessionObjects == null)
    			{
    				htSessionObjects = new Hashtable<Object, Map>();
    			}
    
    			htSessionObjects.put(pSession.getId(), mapMaster);
    
    			if (pNotifyCallHandler)
    			{
    			    pSession.getCallHandler().postObjectCreation();
    			}
    			
    			//inject after put to Hashtable -> that makes it possible that injected objects have access to
    			//the life-cycle object in the constructor
    			injectObjects(pSession, mapMaster);
    			
    			return mapMaster;
    		}
	    }
	}
	
	/**
	 * Updates an existing life-cycle object with current session information.
	 * 
	 * @param pSession the session
	 * @param pMap the associated life-cycle object
	 * @throws Exception if the session access fails
	 */
	private void updateSessionObject(AbstractSession pSession, Map pMap) throws Exception
	{
		List<Entry<String, InjectObject>> liChanges = pSession.getChangedInjectObjects();
		
		if (liChanges != null)
		{
			Entry<String, InjectObject> entry;
			
			InjectObject injobj;
			
			for (int i = 0, anz = liChanges.size(); i < anz; i++)
			{
				entry = liChanges.get(i);

				injobj = entry.getValue();

				if (injobj == null)
				{
					//if the object was not an injected object -> no problem because it will be createad again
					pMap.remove(entry.getKey());
				}
				else
				{
					pMap.put(injobj.getName(), injobj.getObject());
				}
			}
		}
	}
	
	/**
	 * Gets the life-cycle object for an application.
	 * 
	 * @param pSession the accessing session 
	 * @return the life-cycle object for the application
	 * @throws Exception if the life-cycle object can not be created
	 */
	protected synchronized Map getApplicationObject(AbstractSession pSession) throws Exception
	{
		Map mapApplication = null;

		String sApplicationName = pSession.getApplicationName(); 
		
		if (htApplicationObjects != null)
		{
			mapApplication = htApplicationObjects.get(sApplicationName);
		}
		
		if (mapApplication == null)
		{
			mapApplication = initApplicationObject(pSession);
		}
		
		return mapApplication;
	}
	
	/**
	 * Creates and initializes a life-cycle object for an application.
	 * 
	 * @param pSession the accessing session
	 * @return the new application object
	 * @throws Exception if the life-cycle object can not be created
	 */
	private Map initApplicationObject(AbstractSession pSession) throws Exception
	{
		String sApplication = pSession.getApplicationZone().getProperty("/application/lifecycle/application");
		
		if (!StringUtil.isEmpty(sApplication))
		{
			try
			{
				Map mapApplication = createInstance(pSession, sApplication);
				
				if (htApplicationObjects == null)
				{
					htApplicationObjects = new Hashtable<String, Map>();
				}
				
				htApplicationObjects.put(pSession.getApplicationName(), mapApplication);
				
				return mapApplication;
			}
			catch (ClassNotFoundException cnfe)
			{
				log.error("Application object '" + sApplication + "' was not found!", cnfe);

				//don't create a empty Map, to support application loading for new Master sessions
				//e.g. if we change the configuration
				return null;
			}
		}
		else
		{
			return null;
		}
	}

	/**
	 * Creates a new {@link Map} instance with a specific class name and, if possible, sets a parent object.
	 * 
	 * @param pSession the calling session
	 * @param pInstanceName the full qualified class name for the instance
	 * @return the new instance
	 * @throws Exception if the instance can not be created
	 */
	protected Map createInstance(AbstractSession pSession, String pInstanceName) throws Exception
	{
		if (pInstanceName != null)
		{
			Object objInstance;

			ClassLoader loader = getClassLoader(pSession);
			
			Class<?> clazz;
			
			if (loader == null)
			{
			    clazz = Class.forName(pInstanceName);
			    
				objInstance = clazz.newInstance();
			}
			else
			{
			    clazz = Class.forName(pInstanceName, true, loader);
			    
				objInstance = clazz.newInstance();
			}
			
			if (!(objInstance instanceof Map))
			{
				throw new RuntimeException("The lifecycle object '" + pInstanceName + "' has to be a Map!");
			}

			Method metConstruct = null;
			
			//means: class doesn't have a method with @PostConstruct
			boolean bNullMethod = false;
			
			if (whmpConstruct != null)
	        {
			    WeakReference<Method> wref = whmpConstruct.get(clazz);
			    
			    if (wref == null)
			    {
			        bNullMethod = whmpConstruct.containsKey(clazz);
			    }
			    else
			    {
    	            metConstruct = wref.get();
			    }
	        }
			
			if (!bNullMethod)
			{
    			if (metConstruct == null)
    			{
    			    List<WeakReference<Method>> liDestroy = new ArrayUtil<WeakReference<Method>>(); 
    			            
    			    for (Method method : clazz.getDeclaredMethods())
    			    {
    			        if (method.getParameterTypes().length == 0 
    			            && method.getReturnType().equals(Void.TYPE) 
    			            && !Modifier.isStatic(method.getModifiers()))
    			        {
    			            if (method.isAnnotationPresent(PostConstruct.class))
    			            {
        			            if (metConstruct != null)
        			            {
        			                throw new IllegalStateException("It's not allowed to define @PostConstrut on more than one method!");
        			            }
        			            
                                metConstruct = method;
    			            }
    			            
    			            if (method.isAnnotationPresent(PreDestroy.class))
    			            {
    			                liDestroy.add(new WeakReference<Method>(method));
    			            }
    			        }
    			    }
    			    
                    //@PostConstruct CACHE
    			    
                    if (whmpConstruct == null)
                    {
                        whmpConstruct = new WeakHashMap<Class<?>, WeakReference<Method>>();
                    }
                     
                    if (metConstruct == null)
    			    {
                        whmpConstruct.put(clazz, null);
    			    }
                    else
                    {
                        whmpConstruct.put(clazz, new WeakReference(metConstruct));
                    }
                    
                    //@PreDestroy CACHE
                    
                    if (whmpDestroy == null)
                    {
                        whmpDestroy = new WeakHashMap<Class<?>, List<WeakReference<Method>>>();
                    }
                    
                    if (liDestroy.isEmpty())
                    {
                        whmpDestroy.put(clazz, null);
                    }
                    else
                    {
                        whmpDestroy.put(clazz, liDestroy);
                    }
    			}
    			
    			//Found -> execute
    			if (metConstruct != null)
    			{
    			    try
    			    {
    			        Reflective.invoke(objInstance, metConstruct);
    			    }
    			    catch (Throwable th)
    			    {
    			        if (th instanceof Exception)
    			        {
    			            throw (Exception)th;
    			        }
    			        
    			        throw new RuntimeException(th);
    			    }
    			}
			}
			
			return (Map)objInstance;
		}
		else
		{
			throw new ClassNotFoundException("Missing instance name");
		}
	}
	
	/**
	 * Gets the classloader for loading LCOs.
	 * 
	 * @param pSession the session
	 * @return the class loader or <code>null</code> to use the default class loader
	 */
	protected ClassLoader getClassLoader(AbstractSession pSession)
	{
	    return null;
	}
	
	/**
	 * Gets whether the given object is an isolated object. This means that {@link StrictIsolation}
	 * annotation was added.
	 * 
	 * @param pInstance the object to check
	 * @return <code>true</code> if isolated, <code>false</code> otherwise
	 */
	protected boolean isIsolated(Object pInstance)
	{
	    return pInstance != null && isIsolated(pInstance.getClass());
	}
	
	/**
	 * Gets whether the given session is isolated. This means that the LCO contains the {@link StrictIsolation}
	 * annotation.
	 * 
	 * @param pSession the session to check
	 * @return <code>true</code> if isolated, <code>false</code> otherwise
	 */
	public static boolean isIsolated(AbstractSession pSession)
	{
	    AbstractObjectProvider prov = pSession.getObjectProvider();

	    if (prov instanceof DefaultObjectProvider)
	    {
	        DefaultObjectProvider dprov = (DefaultObjectProvider)prov;
	        
	        synchronized (pSession)
	        {
	            if (dprov.htSessionObjects != null)
	            {
	                Object obj = dprov.htSessionObjects.get(pSession.getId());
	                
	                //LCO of session was created -> check the LCO 
	                if (obj != null)
	                {
	                    return dprov.isIsolated(obj);
	                }
	            }
	            
	            //LCO NOT initialized (don't create it)
	            
	            String sLCOName = pSession.getLifeCycleName();
	            
	            if (sLCOName != null)
	            {
	                Class<?> clazz;
	                
	                try
	                {
	                    ClassLoader loader = dprov.getClassLoader(pSession);
	                    
	                    if (loader == null)
	                    {
	                        clazz = Class.forName(sLCOName);
	                    }
	                    else
	                    {
	                        clazz = Class.forName(sLCOName, true, loader);
	                    }
	                }
	                catch (Exception e)
	                {
	                    dprov.log.equals(e);
	                    
	                    return false;
	                }
	                
	                return dprov.isIsolated(clazz);
	            }
	        }
	    }
	    
	    return false;
	}
	
	/**
	 * Gets whether the class has {@link StrictIsolation} annotation.
	 * 
	 * @param pClass the class
	 * @return <code>true</code> if class is "isolated", <code>false</code> otherwise
	 */
    private boolean isIsolated(Class<?> pClass)
    {
        return pClass != null && pClass.getAnnotation(StrictIsolation.class) != null;
    }
	
	/**
	 * Sets the parent for a LCO. This doesn't work if given instance is not a {@link GenericBean} or given parent
	 * is not a {@link Bean}.
	 * 
	 * @param pSession the current session
	 * @param pInstance the LCO
	 * @param pParent the parent
	 */
	protected void setParent(ISession pSession, Map pInstance, Map pParent)
	{
        //---------------------------------------------------------------------
        // Use the "members" of the parent 
        //---------------------------------------------------------------------
        
        if (pParent != null)
        {
            if (pInstance instanceof GenericBean && pParent instanceof Bean)
            {
                ((GenericBean)pInstance).setParent((Bean)pParent);
            }
            else
            {
                log.info("Can't set parent for: ", pSession.getLifeCycleName(), " because the life-cycle object is not instance of GenericBean");
            }
        }
	}
	
	/**
	 * Injects the available objects from the session context into the sessions life-cycle object.
	 * 
	 * @param pSession the accessing session
	 * @param pLifeCycleObject the life-cycle object
	 * @throws Exception if the injection configuration is invalid
	 */
	private void injectObjects(AbstractSession pSession, Map pLifeCycleObject) throws Exception
	{
		String sName;
		
		InjectObject injobj;
		
		for (Enumeration<InjectObject> en = pSession.getInjectObjects(); en.hasMoreElements();)
		{
			injobj = en.nextElement();
		
			sName = injobj.getName();
			
			pLifeCycleObject.put(sName, injobj.getObject());
		}
	}

	/**
	 * Remove injected objects, if needed.
	 * 
	 * @param pSession the session
	 * @param pLifeCycleObject the life-cycle object
	 * @throws Exception if accessing injected objects failed
	 */
	private void removeInjectedObjects(AbstractSession pSession, Map pLifeCycleObject) throws Exception
	{
        String sName;
        
        InjectObject injobj;
        
        for (Enumeration<InjectObject> en = pSession.getInjectObjects(); en.hasMoreElements();)
        {
            injobj = en.nextElement();
        
            if (injobj.isExternal())
            {
                sName = injobj.getName();
                
                pLifeCycleObject.remove(sName);
            }
        }
	}
	
	/**
	 * Sets the object access controller.
	 * 
	 * @param pController the controller
	 */
	public void setObjectAccessController(IObjectAccessController pController)
	{
		oaController = pController;
	}
	
	/**
	 * Gets the object access controller.
	 * 
	 * @return the controller
	 */
	public IObjectAccessController getObjectAccessController()
	{
		return oaController;
	}

	/**
	 * Removes and destroys the life-cycle object for the given session.
	 * 
	 * @param pSession the session
	 */
	private void destroySession(ISession pSession)
	{
		if (htSessionObjects != null)
		{
			Map map = htSessionObjects.remove(pSession.getId());
			
			if (map != null)
			{
			    if (whmpDestroy != null)
			    {
			        List<WeakReference<Method>> liDestroy = whmpDestroy.get(map.getClass());
			        
			        if (liDestroy != null)
			        {
			            Method met;
			            
			            //Invoke @PreDestroy methods (if found during creation)
			            for (int i = 0, cnt = liDestroy.size(); i < cnt; i++)
			            {
			                met = liDestroy.get(i).get();
			                
			                if (met != null)
			                {
			                    try
			                    {
			                        Reflective.invoke(map, met);
			                    }
			                    catch (Throwable th)
			                    {
			                        log.debug(th);
			                    }
			                }
			            }
			        }
			    }
			    
    			if (map instanceof ILifeCycleObject)
    			{
    				SessionContext context = SessionContext.getCurrentInstance();
    				
    				boolean bNewContext;
    				
    				if (context == null)
    				{
    					context = ((AbstractSession)pSession).createSessionContext(null, "destroy");
    					
    					bNewContext = true;
    				}
    				else
    				{
    					bNewContext = false;
    				}
    					
    				try
    				{
                        //don't close injected objects
    				    removeInjectedObjects((AbstractSession)pSession, map);

                        ((ILifeCycleObject)map).destroy();
    				}
    				catch (Throwable th)
    				{
    					log.debug(th);
    				}
    				finally
    				{
    					if (bNewContext)
    					{
    						context.release();
    					}
    				}
    			}
    			else if (!map.isEmpty())
    			{
                    for (Iterator<Entry<Object, Object>> it = map.entrySet().iterator(); it.hasNext();)
                    {
                        CommonUtil.close(it.next().getValue());
                    }
    			}
			}
		}
	}

    /**
     * Invokes the given method and checks if it's not accesible. A method is not accessible if the {@link NotAccessible}
     * annotation is present.
     * 
     * @param pMethod the method to call
     * @param pObject the object that contains the method
     * @param pObjectName the object name (only for exception handling)
     * @return the result of the method call
     * @throws Exception if method call fails
     * @throws SecurityException if access to method was explicitely denied
     */
    private Object invokeMethod(Method pMethod, Object pObject, StringBuilder pObjectName) throws Exception
    {
        if (pMethod.isAnnotationPresent(NotAccessible.class))
        {
            throw new SecurityException("Access to " + pObjectName + " denied!");
        }
        
        return pMethod.invoke(pObject);
    }

    /**
	 * Invokes the given method and checks if it's accesible. A method is accessible if the {@link Accessible}
	 * annotation is present.
	 * 
	 * @param pMethod the method to call
	 * @param pObject the object that contains the method
	 * @param pObjectName the object name (only for exception handling)
	 * @return the result of the method call
	 * @throws Exception if method call fails
	 * @throws SecurityException if access to method was not explicitely granted
	 */
	private Object invokeSubMethod(Method pMethod, Object pObject, StringBuilder pObjectName) throws Exception
	{
        if (!pMethod.isAnnotationPresent(Accessible.class))
        {
            throw new SecurityException("Access to " + pObjectName + " denied!");
        }
        
        return pMethod.invoke(pObject);
	}
	
    //****************************************************************
    // Subclass definition
    //****************************************************************
	
	/**
	 * A marker class. It's inherited from GenericBean because of Parent hierarchy!
	 * 
	 * @author Ren Jahn
	 */
	public static final class ImplicitLifeCycleObject extends GenericBean
	{
	}
	
}	// DefaultObjectProvider
