/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 * 
 * 21.11.2008 - [JR] - creation
 * 17.02.2009 - [JR] - set/getCursor implemented
 * 20.02.2009 - [JR] - extend IMessageConstants
 * 15.04.2009 - [JR] - log methods implemented
 * 22.04.2009 - [JR] - removed log methods
 * 28.05.2009 - [JR] - don't implement getContentPane -> possible a problem with return null
 *                     let subclasses implement the method because they know what's to do!
 * 06.08.2009 - [JR] - added simple showMessage calls                     
 * 12.06.2010 - [JR] - show... now returns IContent 
 *                   - show... generic opener
 * 21.03.2013 - [JR] - #649: IApplicationSetup introduced and used       
 * 24.06.2014 - [JR] - #1078: getControllerProperty overwritten
 * 05.05.2015 - [JR] - #1379: default implementation of getContent and showMessage  
 */
package javax.rad.application.genui;

import javax.rad.application.IApplication;
import javax.rad.application.IContent;
import javax.rad.ui.IContainer;
import javax.rad.ui.ICursor;

import com.sibvisions.util.Reflective;

/**
 * The <code>Application</code> is an abstract implementation of {@link IApplication}.
 * It only implements the methods for loading and creating classes.
 *  
 * @author Ren Jahn
 */
public abstract class Application extends ControllerContent 
								  implements IApplication
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the parameter name for the setup class. */
	public static final String PARAM_SETUP_CLASS = "Application.setup.classname";

	/** the launcher implementation. */
	private UILauncher launcher;
	
	/** the specific application setup. */
	private IApplicationSetup setup;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creates a new instance of <code>Application</code> with a desired 
	 * launcher.
	 * 
	 * @param pLauncher the launcher of this application
	 */
	public Application(UILauncher pLauncher)
	{
		setLauncher(pLauncher);
		
		try
		{
			String sClass = pLauncher.getParameter(PARAM_SETUP_CLASS);
			
			if (sClass != null)
			{
				setup = (IApplicationSetup)Reflective.construct(sClass);
				setup.apply(this);
				
				debug("Apply setup: ", sClass);
			}
		}
		catch (Throwable th)
		{
			//no specific setup
			
			debug(th);
		}
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	public UILauncher getLauncher()
	{
		return launcher;
	}

    /**
     * {@inheritDoc}
     */
    public <OP> IContent showMessage(OP pOpener, 
                                     int pIconType, 
                                     int pButtonType, 
                                     String pMessage, 
                                     String pOkAction, 
                                     String pCancelAction) throws Throwable
    {
        getLauncher().handleException(new Exception(pMessage));
        
        return null;
    }
	
	/**
	 * {@inheritDoc}
	 */
	public IContainer getApplicationPane()
	{
		return this;
	}

    /**
     * {@inheritDoc}
     */
	public IContainer getContentPane()
	{
	    return getApplicationPane();
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Sets the cursor of the launcher.
	 * 
	 * @param pCursor the cursor
	 */
	@Override
	public void setCursor(ICursor pCursor)
	{
		launcher.setCursor(pCursor);
	}
	
	/**
	 * Gets the cursor from the launcher.
	 * 
	 * @return the cursor of the launcher
	 */
	@Override
	public ICursor getCursor()
	{
		return launcher.getCursor();
	}
	
	/**
	 * Cancels all pending launcher threads.
	 */
	@Override
	public void notifyDestroy()
	{
		launcher.cancelPendingThreads();
	}
	
	/**
	 * Gets the current value of the given controller property. If the property was not set,
	 * the launcher parameter will be checked for a name with the prefix <code>Controller.</code>, e.g.
	 * Controller.name.
	 * 
	 * @param pName the name of the property
	 * @return the value or <code>null</code> if the property was not found
	 */
	@Override
    public Object getControllerProperty(String pName)
    {
	    Object oValue = super.getControllerProperty(pName);
	    
	    if (oValue == null)
	    {
	        return launcher.getParameter("Controller." + pName);
	    }
	    
	    return oValue;
    }	
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Shows a message with the {@link javax.rad.application.IMessageConstants#MESSAGE_ICON_INFO} icon and the
	 * {@link javax.rad.application.IMessageConstants#MESSAGE_BUTTON_OK} button.
	 * 
	 * @param <OP> the opener type
	 * @param pOpener the opener of the information
	 * @param pMessage the message/information to show
	 * @return the message content or <code>null</code> if the message has no content
	 * @throws Throwable if the message could not be initialized 
	 */
	public <OP> IContent showInformation(OP pOpener, String pMessage) throws Throwable
	{
		return showMessage(pOpener, MESSAGE_ICON_INFO, MESSAGE_BUTTON_OK, pMessage, null, null);
	}

	/**
	 * Shows a message with the {@link javax.rad.application.IMessageConstants#MESSAGE_ICON_INFO} icon and the
	 * {@link javax.rad.application.IMessageConstants#MESSAGE_BUTTON_OK} button.
	 * 
	 * @param <OP> the opener type
	 * @param pOpener the opener of the information
	 * @param pMessage the message/information to show
	 * @param pOkAction the action to call when OK was pressed
	 * @return the message content or <code>null</code> if the message has no content
	 * @throws Throwable if the message could not be initialized 
	 */
	public <OP> IContent showInformation(OP pOpener, String pMessage, String pOkAction) throws Throwable
	{
		return showMessage(pOpener, MESSAGE_ICON_INFO, MESSAGE_BUTTON_OK, pMessage, pOkAction, null);
	}

	/**
	 * Shows a message with the {@link javax.rad.application.IMessageConstants#MESSAGE_ICON_ERROR} icon and the
	 * {@link javax.rad.application.IMessageConstants#MESSAGE_BUTTON_OK} button.
	 * 
	 * @param <OP> the opener type
	 * @param pOpener the opener of the information
	 * @param pMessage the message/error to show
	 * @return the message content or <code>null</code> if the message has no content
	 * @throws Throwable if the message could not be initialized 
	 */
	public <OP> IContent showError(OP pOpener, String pMessage) throws Throwable
	{
		return showMessage(pOpener, MESSAGE_ICON_ERROR, MESSAGE_BUTTON_OK, pMessage, null, null);
	}
	
	/**
	 * Shows a message with the {@link javax.rad.application.IMessageConstants#MESSAGE_ICON_ERROR} icon and the
	 * {@link javax.rad.application.IMessageConstants#MESSAGE_BUTTON_OK} button.
	 * 
	 * @param <OP> the opener type
	 * @param pOpener the opener of the information
	 * @param pMessage the message/error to show
	 * @param pOkAction the action to call when OK was pressed
	 * @return the message content or <code>null</code> if the message has no content
	 * @throws Throwable if the message could not be initialized 
	 */
	public <OP> IContent showError(OP pOpener, String pMessage, String pOkAction) throws Throwable
	{
		return showMessage(pOpener, MESSAGE_ICON_ERROR, MESSAGE_BUTTON_OK, pMessage, pOkAction, null);
	}
	
	/**
	 * Shows a message with the {@link javax.rad.application.IMessageConstants#MESSAGE_ICON_QUESTION} icon and the
	 * {@link javax.rad.application.IMessageConstants#MESSAGE_BUTTON_YES_NO} buttons.
	 * 
	 * @param <OP> the opener type
	 * @param pOpener the opener of the information
	 * @param pMessage the message/question to show
	 * @param pOkAction the action to call when yex was pressed
	 * @return the message content or <code>null</code> if the message has no content
	 * @throws Throwable if the message could not be initialized 
	 */
	public <OP> IContent showQuestion(OP pOpener, String pMessage, String pOkAction) throws Throwable
	{
		return showMessage(pOpener, MESSAGE_ICON_QUESTION, MESSAGE_BUTTON_YES_NO, pMessage, pOkAction, null);
	}

	/**
	 * Shows a message with the {@link javax.rad.application.IMessageConstants#MESSAGE_ICON_QUESTION} icon and the
	 * {@link javax.rad.application.IMessageConstants#MESSAGE_BUTTON_YES_NO} buttons.
	 * 
	 * @param <OP> the opener type
	 * @param pOpener the opener of the information
	 * @param pMessage the message/question to show
	 * @param pOkAction the action to call when yex was pressed
	 * @param pCancelAction the action to call when no/x was pressed
	 * @return the message content or <code>null</code> if the message has no content
	 * @throws Throwable if the message could not be initialized 
	 */
	public <OP> IContent showQuestion(OP pOpener, String pMessage, String pOkAction, String pCancelAction) throws Throwable
	{
		return showMessage(pOpener, MESSAGE_ICON_QUESTION, MESSAGE_BUTTON_YES_NO, pMessage, pOkAction, pCancelAction);
	}

	/**
	 * Sets the launcher. Be careful with this method because it changes the behaviour of
	 * the application if you use the wrong launcher.
	 * 
	 * @param pLauncher the launcher
	 */
	protected final void setLauncher(UILauncher pLauncher)
	{
		launcher = pLauncher;
	}
	
	/**
	 * Gets the application setup, if used.
	 * 
	 * @return the application setup or <code>null</code> if no specific setup was used
	 */
	protected IApplicationSetup getSetup()
	{
		return setup;
	}
	
}	// Application
