/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 08.04.2009 - [JR] - creation
 * 28.04.2009 - [JR] - merge method removed
 * 29.07.2009 - [JR] - formatLabel copied from ColumnDefinition
 * 06.10.2009 - [JR] - getAlphaNumeric implemented
 * 26.11.2009 - [JR] - toString(Object) implemented
 * 13.02.2010 - [JR] - formatInitCap implemented
 * 14.02.2010 - [JR] - formatInitCap with remove spaces implemented
 * 26.03.2010 - [JR] - #92: removeQuotes implemented
 * 09.10.2010 - [JR] - #114: added removeCharacters
 * 29.11.2010 - [JR] - replace implemented
 * 04.03.2011 - [JR] - getCaseSensitiveType and getCharacterType implemented
 * 10.03.2011 - [JR] - getAlphaNumeric renamed to getText and used TextType
 *                   - added LettersDigitsWhitespace and LettersDigitsSpace
 * 07.06.2011 - [JR] - #383
 *                     * removed formatHumanReadable
 *                     * removed keep format parameter from formatInitCap
 *                     * introduced convertToMethodName, convertToMemberName
 * 21.06.2011 - [JR] - convertToName implemented  
 * 06.07.2011 - [JR] - #416: formatInitCap trim implemented      
 * 05.08.2011 - [JR] -  mapping was A not AE [BUGFIX]     
 * 10.08.2011 - [JR] - formatInitCap: '_' replacement is included in whitespaces
 * 30.10.2011 - [JR] - convertMethodNameToText implemented
 * 20.11.2011 - [JR] - convertMethodNameToText with replace parameter implemented 
 * 26.01.2013 - [JR] - convertMemberNameToText implemented
 * 21.02.2013 - [JR] - isEmpty implemented    
 * 12.04.2013 - [RH] - padLeft/padRight methods implemented
 * 18.04.2013 - [JR] - containsWhitespace implemented
 * 09.05.2013 - [JR] - countCharacter implemented
 * 14.05.2013 - [JR] - separate implemented
 * 18.01.2014 - [JR] - convertMethodNameToText, convertMemberNameToText now replaces '_' with ' '
 * 13.07.2014 - [JR] - firstCharLower implemented
 * 26.10.2014 - [JR] - getFirstWord implemented
 * 29.11.2014 - [JR] - toString with max array length support
 * 17.06.2015 - [JR] - deepToString now with IdentityHashMap instead of HashSet, to avoid
 *                     recursion with hashCode if same object were found 
 * 30.07.2015 - [JR] - getText with char... support                     
 */
package com.sibvisions.util.type;

import java.util.Arrays;
import java.util.Collection;
import java.util.Dictionary;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import com.sibvisions.util.ArrayUtil;

/**
 * The <code>StringUtil</code> contains string dependent utility methods.
 * 
 * @author Ren Jahn
 */
public final class StringUtil
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** case senstive types. */
	public static enum CaseSensitiveType
	{
		/** all letters are lower case. */
		LowerCase,
		/** all letters are upper case. */
		UpperCase,
		/** letters are upper and lower case. */
		MixedCase,
		/** no letter available. */
		NoLetter
	}
	
	/** the character types. */
	public static enum CharacterType
	{
		/** only letters. */
		Letters,
		/** only digits. */
		Digits,
		/** letters and digits. */
		LettersDigits,
		/** only whitespaces. */
		OnlyWhitespace,
		/** no letters and no digits. */
		OnlySpecial,
		/** letters and no digits. */
		LettersSpecial,
		/** digits and no letters. */
		DigitsSpecial,
		/** letters and digits and whitespace. */
		LettersDigitsWhitespace,
		/** letters and digits and space. */
		LettersDigitsSpace,
		/** all kind of characters. */
		All,
		/** null or empty. */
		None
	}
	
	/** the text types. */
	public static enum TextType
	{
		/** only letters. */
		Letters,
		/** only digits. */
		Digits,
		/** letters and digits. */
		LettersDigits,
		/** letters and digits and whitespace. */
		LettersDigitsWhitespace,
		/** letters and digits and space. */
		LettersDigitsSpace,
		/** only letters from A(a) to Z(z). */ 
		AZLetters,
		/** letters from A(a) to Z(z) and digits. */ 
		AZLettersDigits,
		/** all characters but without leading digits. */
		WithoutLeadingDigits,
		/** all characters but start from first letter. */
		FromFirstLetter,
		/** all characters that are uppercase. */
		UpperCase,
		/** all characters that are lowercase. */
		LowerCase
	}
	
	/** the character replacements. */
	private static HashMap<Character, String> hmpReplaceCharacters = new HashMap<Character, String>();
	
	/** the character replacements (reverse mode). */
	private static HashMap<String, Character> hmpReplaceReverseCharacters = new HashMap<String, Character>();
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Initializes the replacement mapping for name conversions.
	 */
	static
	{
		hmpReplaceReverseCharacters.put("ue", Character.valueOf((char)0x00FC));
		hmpReplaceReverseCharacters.put("Ue", Character.valueOf((char)0x00DC));
		hmpReplaceReverseCharacters.put("UE", Character.valueOf((char)0x00DC));
		hmpReplaceReverseCharacters.put("ae", Character.valueOf((char)0x00E4));
		hmpReplaceReverseCharacters.put("Ae", Character.valueOf((char)0x00C4));
		hmpReplaceReverseCharacters.put("AE", Character.valueOf((char)0x00C4));
		hmpReplaceReverseCharacters.put("oe", Character.valueOf((char)0x00F6));
		hmpReplaceReverseCharacters.put("Oe", Character.valueOf((char)0x00D6));
		hmpReplaceReverseCharacters.put("OE", Character.valueOf((char)0x00D6));
		hmpReplaceReverseCharacters.put("ss", Character.valueOf((char)0x00DF));
		
		//see: http://jrgraphix.net/r/Unicode/
		
		//Latin-1 Supplement
		hmpReplaceCharacters.put(Character.valueOf((char)0x00A2), "c");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00A5), "Y");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00A9), "c");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00AA), "a");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00AE), "R");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00C0), "A");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00C1), "A");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00C2), "A");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00C3), "A");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00C4), "AE");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00C5), "A");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00C6), "AE");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00C7), "C");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00C8), "E");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00C9), "E");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00CA), "E");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00CB), "E");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00CC), "I");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00CD), "I");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00CE), "I");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00CF), "I");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00D0), "D");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00D1), "N");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00D2), "O");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00D3), "O");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00D4), "O");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00D5), "O");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00D6), "OE");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00D7), "x");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00D8), "O");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00D9), "U");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00DA), "U");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00DB), "U");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00DC), "UE");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00DD), "Y");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00DF), "ss");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00E0), "a");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00E1), "a");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00E2), "a");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00E3), "a");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00E4), "ae");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00E5), "a");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00E6), "ae");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00E7), "c");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00E8), "e");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00E9), "e");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00EA), "e");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00EB), "ae");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00EC), "i");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00ED), "i");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00EE), "i");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00EF), "i");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00F0), "o");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00F1), "n");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00F2), "o");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00F3), "o");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00F4), "o");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00F5), "o");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00F6), "oe");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00F8), "o");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00F9), "u");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00FA), "u");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00FB), "u");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00FC), "ue");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00FD), "y");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00FE), "b");
		hmpReplaceCharacters.put(Character.valueOf((char)0x00FF), "y");
		
		//Latin Extended-A
		hmpReplaceCharacters.put(Character.valueOf((char)0x0100), "A");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0101), "a");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0102), "A");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0103), "a");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0104), "A");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0105), "a");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0106), "C");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0107), "c");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0108), "C");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0109), "c");
		hmpReplaceCharacters.put(Character.valueOf((char)0x010A), "C");
		hmpReplaceCharacters.put(Character.valueOf((char)0x010B), "c");
		hmpReplaceCharacters.put(Character.valueOf((char)0x010C), "C");
		hmpReplaceCharacters.put(Character.valueOf((char)0x010D), "c");
		hmpReplaceCharacters.put(Character.valueOf((char)0x010E), "D");
		hmpReplaceCharacters.put(Character.valueOf((char)0x010F), "d");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0110), "D");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0111), "d");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0112), "E");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0113), "e");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0114), "E");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0115), "e");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0116), "E");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0117), "e");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0118), "E");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0119), "e");
		hmpReplaceCharacters.put(Character.valueOf((char)0x011A), "E");
		hmpReplaceCharacters.put(Character.valueOf((char)0x011B), "e");
		hmpReplaceCharacters.put(Character.valueOf((char)0x011C), "G");
		hmpReplaceCharacters.put(Character.valueOf((char)0x011D), "g");
		hmpReplaceCharacters.put(Character.valueOf((char)0x011E), "G");
		hmpReplaceCharacters.put(Character.valueOf((char)0x011F), "g");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0120), "G");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0121), "g");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0122), "G");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0123), "g");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0124), "H");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0125), "h");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0126), "H");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0127), "h");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0128), "I");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0129), "i");
		hmpReplaceCharacters.put(Character.valueOf((char)0x012A), "I");
		hmpReplaceCharacters.put(Character.valueOf((char)0x012B), "i");
		hmpReplaceCharacters.put(Character.valueOf((char)0x012C), "I");
		hmpReplaceCharacters.put(Character.valueOf((char)0x012D), "i");
		hmpReplaceCharacters.put(Character.valueOf((char)0x012E), "I");
		hmpReplaceCharacters.put(Character.valueOf((char)0x012F), "i");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0130), "i");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0131), "I");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0132), "IJ");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0133), "ij");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0134), "J");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0135), "j");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0136), "K");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0137), "k");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0138), "K");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0139), "L");
		hmpReplaceCharacters.put(Character.valueOf((char)0x013A), "I");
		hmpReplaceCharacters.put(Character.valueOf((char)0x013B), "L");
		hmpReplaceCharacters.put(Character.valueOf((char)0x013C), "I");
		hmpReplaceCharacters.put(Character.valueOf((char)0x013D), "L");
		hmpReplaceCharacters.put(Character.valueOf((char)0x013E), "I");
		hmpReplaceCharacters.put(Character.valueOf((char)0x013F), "L");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0140), "I");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0141), "L");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0142), "I");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0143), "N");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0144), "n");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0145), "N");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0146), "n");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0147), "N");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0148), "n");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0149), "n");
		hmpReplaceCharacters.put(Character.valueOf((char)0x014A), "n");
		hmpReplaceCharacters.put(Character.valueOf((char)0x014B), "n");
		hmpReplaceCharacters.put(Character.valueOf((char)0x014C), "O");
		hmpReplaceCharacters.put(Character.valueOf((char)0x014D), "o");
		hmpReplaceCharacters.put(Character.valueOf((char)0x014E), "O");
		hmpReplaceCharacters.put(Character.valueOf((char)0x014F), "o");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0150), "OE");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0151), "oe");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0152), "OE");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0153), "oe");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0154), "R");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0155), "r");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0156), "R");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0157), "r");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0158), "R");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0159), "r");
		hmpReplaceCharacters.put(Character.valueOf((char)0x015A), "S");
		hmpReplaceCharacters.put(Character.valueOf((char)0x015B), "s");
		hmpReplaceCharacters.put(Character.valueOf((char)0x015C), "S");
		hmpReplaceCharacters.put(Character.valueOf((char)0x015D), "s");
		hmpReplaceCharacters.put(Character.valueOf((char)0x015E), "S");
		hmpReplaceCharacters.put(Character.valueOf((char)0x015F), "s");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0160), "S");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0161), "s");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0162), "T");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0163), "t");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0164), "T");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0165), "t");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0166), "F");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0167), "f");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0168), "U");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0169), "u");
		hmpReplaceCharacters.put(Character.valueOf((char)0x016A), "U");
		hmpReplaceCharacters.put(Character.valueOf((char)0x016B), "u");
		hmpReplaceCharacters.put(Character.valueOf((char)0x016C), "U");
		hmpReplaceCharacters.put(Character.valueOf((char)0x016D), "u");
		hmpReplaceCharacters.put(Character.valueOf((char)0x016E), "U");
		hmpReplaceCharacters.put(Character.valueOf((char)0x016F), "u");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0170), "UE");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0171), "ue");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0172), "U");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0173), "u");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0174), "W");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0175), "w");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0176), "Y");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0177), "y");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0178), "Y");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0179), "Z");
		hmpReplaceCharacters.put(Character.valueOf((char)0x017A), "z");
		hmpReplaceCharacters.put(Character.valueOf((char)0x017B), "Z");
		hmpReplaceCharacters.put(Character.valueOf((char)0x017C), "z");
		hmpReplaceCharacters.put(Character.valueOf((char)0x017D), "Z");
		hmpReplaceCharacters.put(Character.valueOf((char)0x017E), "z");
		
		//Latin Extended-B
		hmpReplaceCharacters.put(Character.valueOf((char)0x0180), "b");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0181), "B");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0182), "b");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0183), "b");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0184), "b");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0185), "b");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0186), "C");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0187), "C");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0188), "c");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0189), "D");
		hmpReplaceCharacters.put(Character.valueOf((char)0x018A), "D");
		hmpReplaceCharacters.put(Character.valueOf((char)0x018E), "E");
		hmpReplaceCharacters.put(Character.valueOf((char)0x018F), "e");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0191), "F");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0192), "f");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0193), "G");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0194), "Y");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0196), "I");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0197), "I");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0198), "K");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0199), "k");
		hmpReplaceCharacters.put(Character.valueOf((char)0x019A), "I");
		hmpReplaceCharacters.put(Character.valueOf((char)0x019D), "N");
		hmpReplaceCharacters.put(Character.valueOf((char)0x019E), "n");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01A0), "O");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01A1), "o");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01A4), "P");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01A6), "R");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01A7), "S");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01A8), "s");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01AA), "l");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01AB), "t");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01AC), "T");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01AD), "f");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01AE), "T");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01AF), "U");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01B0), "u");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01B2), "U");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01B3), "Y");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01B4), "y");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01B5), "Z");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01B6), "z");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01BC), "5");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01BF), "P");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01C0), "I");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01C4), "DZ");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01C5), "Dz");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01C6), "dz");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01C7), "LJ");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01C8), "Lj");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01C9), "lj");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01CA), "NJ");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01CB), "Nj");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01CC), "nj");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01CD), "A");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01CE), "a");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01CF), "I");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01D0), "I");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01D1), "O");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01D2), "o");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01D3), "U");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01D4), "u");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01D5), "U");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01D6), "u");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01D7), "U");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01D8), "u");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01D9), "U");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01DA), "u");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01DB), "U");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01DC), "u");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01DD), "e");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01DE), "A");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01DF), "a");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01E0), "A");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01E1), "a");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01E3), "AE");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01E4), "ae");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01E4), "G");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01E5), "g");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01E6), "G");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01E7), "g");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01E8), "K");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01E9), "k");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01EA), "O");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01EB), "o");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01EC), "O");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01ED), "o");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01F0), "J");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01F1), "DZ");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01F2), "Dz");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01F3), "dz");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01F4), "G");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01F5), "g");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01F6), "Hu");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01F7), "D");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01F8), "N");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01F9), "n");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01FA), "A");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01FB), "a");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01FC), "AE");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01FD), "ae");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01FE), "O");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01FF), "o");
		hmpReplaceCharacters.put(Character.valueOf((char)0x01F8), "N");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0200), "A");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0201), "a");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0202), "A");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0203), "a");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0204), "E");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0205), "e");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0206), "E");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0207), "e");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0208), "I");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0209), "i");
		hmpReplaceCharacters.put(Character.valueOf((char)0x020A), "I");
		hmpReplaceCharacters.put(Character.valueOf((char)0x020B), "i");
		hmpReplaceCharacters.put(Character.valueOf((char)0x020C), "OE");
		hmpReplaceCharacters.put(Character.valueOf((char)0x020D), "oe");
		hmpReplaceCharacters.put(Character.valueOf((char)0x020E), "O");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0210), "R");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0211), "r");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0212), "R");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0213), "r");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0214), "UE");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0215), "ue");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0216), "U");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0217), "u");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0218), "S");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0219), "s");
		hmpReplaceCharacters.put(Character.valueOf((char)0x021A), "T");
		hmpReplaceCharacters.put(Character.valueOf((char)0x021E), "H");
		hmpReplaceCharacters.put(Character.valueOf((char)0x021F), "h");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0220), "n");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0221), "d");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0224), "Z");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0225), "z");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0226), "A");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0227), "a");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0228), "E");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0229), "e");
		hmpReplaceCharacters.put(Character.valueOf((char)0x022A), "O");
		hmpReplaceCharacters.put(Character.valueOf((char)0x022B), "o");
		hmpReplaceCharacters.put(Character.valueOf((char)0x022C), "O");
		hmpReplaceCharacters.put(Character.valueOf((char)0x022D), "o");
		hmpReplaceCharacters.put(Character.valueOf((char)0x022E), "O");
		hmpReplaceCharacters.put(Character.valueOf((char)0x022F), "o");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0230), "O");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0231), "o");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0232), "Y");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0233), "y");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0234), "I");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0235), "n");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0236), "t");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0237), "J");
		hmpReplaceCharacters.put(Character.valueOf((char)0x023A), "A");
		hmpReplaceCharacters.put(Character.valueOf((char)0x023B), "C");
		hmpReplaceCharacters.put(Character.valueOf((char)0x023C), "c");
		hmpReplaceCharacters.put(Character.valueOf((char)0x023D), "t");
		hmpReplaceCharacters.put(Character.valueOf((char)0x023E), "T");
		hmpReplaceCharacters.put(Character.valueOf((char)0x023F), "s");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0240), "Z");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0243), "B");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0244), "U");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0246), "E");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0247), "e");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0248), "J");
		hmpReplaceCharacters.put(Character.valueOf((char)0x0249), "j");
		hmpReplaceCharacters.put(Character.valueOf((char)0x024A), "q");
		hmpReplaceCharacters.put(Character.valueOf((char)0x024B), "q");
		hmpReplaceCharacters.put(Character.valueOf((char)0x024C), "R");
		hmpReplaceCharacters.put(Character.valueOf((char)0x024D), "r");
		hmpReplaceCharacters.put(Character.valueOf((char)0x024E), "Y");
		hmpReplaceCharacters.put(Character.valueOf((char)0x024F), "y");
	}
	
	/**
	 * Invisible constructor because <code>StringUtil</code> is a utility
	 * class.
	 */
	private StringUtil()
	{
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Returns the boundaries of quoted strings.
	 * Examples.: 
	 * <pre>
	 * unitTest("Test", Application.class, " Application.class");
	 * unitTest("Test\", Application.class", Application.class");
	 * unitTest("Test", Application.class", Application.class);
	 * </pre>
	 * 
	 * @param sText the text with our without quotes
	 * @return the boundaries as array [start, stop, start, stop] or an empty array
	 */
	public static int[] getQuotedBoundaries(String sText)
	{
		int[] iBoundaries = new int[0];

		if (sText != null)
		{
			int iStart = -1;
			
			boolean bStart = false;
			
			for (int i = 0, length = sText.length(); i < length; i++)
			{
				if (sText.charAt(i) == '\"' && (i == 0 || sText.charAt(i - 1) != '\\'))
				{
					bStart = !bStart;
					
					if (bStart)
					{
						iStart = i;
					}
					else
					{
						iBoundaries = ArrayUtil.add(iBoundaries, iStart);
						iBoundaries = ArrayUtil.add(iBoundaries, i);
					}
				}
			}
		}
		
		return iBoundaries;
	}
	
	/**
	 * Gets all words from the given text with the given maximum length. If
	 * the given text is {@code null} or empty the given text is returned. If
	 * the given text does not contain any upper case characters, an empty
	 * string is returned.
	 * 
	 * Anything that is not a letter is returned verbatim and assumed to be
	 * a word separator.
	 * 
	 * Examples:
	 * <pre>
	 * SomeDataWorkScreen, 3		-&gt; SomDatWorScr
	 * ContractsEducation, 3		-&gt; ConEdu
	 * ContractsEducation, 5		-&gt; ContrEduca
	 * com.sibvisions.test.ClassName, 3	-&gt; com.sib.tes.ClaNam
	 * </pre>
	 * 
	 * @param pText the text to process.
	 * @param pMaxWordLength the maximum length of the words.
	 * @return all words from the given text with the given maximum length.
	 *         {@code null} if the given text was null, an empty string if
	 *         the given text was empty or the maximum word length was equal or
	 *         less than zero.
	 */
	public static String getShortenedWords(String pText, int pMaxWordLength)
	{
		if (isEmpty(pText))
		{
			return pText;
		}
		
		if (pMaxWordLength <= 0)
		{
			return "";
		}
		
		StringBuilder sbWords = new StringBuilder();
		
		char ch;
		
		for (int i = 0, iCurrentWordLength = 0, len = pText.length(); i < len; i++, iCurrentWordLength++)
		{
			ch = pText.charAt(i);
			
			if (Character.isUpperCase(ch))
			{
				sbWords.append(ch);
				iCurrentWordLength = 0;
			}
			else if (!Character.isLetter(ch))
			{
				sbWords.append(ch);
				iCurrentWordLength = -1;
			}
			else if (iCurrentWordLength < pMaxWordLength)
			{
				sbWords.append(ch);
			}
		}
		
		return sbWords.toString();
	}
	
	/**
	 * Gets the first word of a text. The end of the first word is defined as end of text or
	 * if a uppercase letter follows.
	 * 
	 * @param pText the text
	 * @return the first word
	 */
	public static String getFirstWord(String pText)
	{
	    if (isEmpty(pText))
        {
	        return pText;
	    }
	    
	    StringBuilder sbWord = new StringBuilder();
	    
	    char ch;
	    
	    for (int i = 0, len = pText.length(); i < len; i++)
	    {
	        ch = pText.charAt(i);
	        
	        if (i > 0 && Character.isUpperCase(ch))
	        {
	            return sbWord.toString();
	        }
	        else
	        {
	            sbWord.append(ch);
	        }
	    }
	    
	    return sbWord.toString();
	}
	
	/**
	 * Separates a string of values with a configurable delimiter.
	 * <p>
	 * @param pList string with values
	 * @param pDelimiter delimiter to separate
	 * @param pTrim true to trim the separated values
	 * @return list of separated values
	 */
	public static ArrayUtil<String> separateList(String pList, String pDelimiter, boolean pTrim)
	{
	    ArrayUtil<String> alTokens = new ArrayUtil<String>();
	
	    
	    if (pList != null)
	    {
	        if (!pList.endsWith(pDelimiter))
	    	{
	        	//don't ignore the last element 
	        	//TODO [HM] doubles memory usage and heap work.
	        	pList += pDelimiter;
	    	}
	    	
	        String sPart;
	        
	        int iStart = 0;
	        int iEnd = 0;
	
	        while ((iEnd = pList.indexOf(pDelimiter, iStart)) >= 0)
	        {
	        	sPart = pList.substring(iStart, iEnd);
	        	
	        	if (pTrim)
	        	{
	        		sPart = sPart.trim();
	        	}
	        	
	            alTokens.add(sPart);
	            
	            iStart = iEnd + pDelimiter.length();
	        }
	    }
	    
	    return alTokens;
	}

	/**
	 * Separates the given text in parts.
	 * 
	 * @param pText the text
	 * @param pStartDelimiter the start delimiter
	 * @param pEndDelimiter the end delimiter
	 * @param pIncludeDelimiter <code>true</code> to include the delimiters in the result for every found part
	 * @return all found parts
	 */
	public static ArrayUtil<String> separate(String pText, String pStartDelimiter, String pEndDelimiter, boolean pIncludeDelimiter)
	{
		ArrayUtil<String> auResult = new ArrayUtil<String>();
		
		if (pText != null)
		{
			int iStart = 0;
			int iEnd = 0;
			
			while ((iStart = pText.indexOf(pStartDelimiter, iStart)) >= 0)
			{
				iEnd = pText.indexOf(pEndDelimiter, iStart);
				
				if (iEnd >= 0)
				{
					if (pIncludeDelimiter)
					{
						auResult.add(pText.substring(iStart, iEnd + 1));
					}
					else
					{
						auResult.add(pText.substring(iStart + 1, iEnd));
					}
					
					iStart = iEnd + pEndDelimiter.length();
				}
			}		
		}
		
		return auResult;
	}
	
	/**
	 * Gets the int values from a string with delimiters.
	 * 
	 * @param pValues the string with numbers and delimiters
	 * @param pDelimiter the delimiter
	 * @return the int values or <code>null</code> if the values are <code>null</code> or empty
	 */
	public static int[] parseInteger(String pValues, String pDelimiter)
	{
		if (pValues == null || pValues.trim().length() == 0)
		{
			return null;
		}
		
		ArrayUtil<String> auList = separateList(pValues, pDelimiter, true);
		
		int[] iValues = new int[auList.size()];
		
		for (int i = 0, length = iValues.length; i < length; i++)
		{
			try
			{
				iValues[i] = Integer.parseInt(auList.get(i));
			}
			catch (NumberFormatException nfe)
			{
				iValues[i] = -1;
			}
		}
		
		return iValues;
	}

	/**
	 * Sub function of like. For performance reasons, the original Strings are not modified,
	 * The start and end defines the region to search.
	 * 
	 * @param pSource	the source string.
	 * @param pSrcStart	the start index of the source region.
	 * @param pSrcEnd	the end index of the source region.
	 * @param pSearch	the search string.
	 * @param pStart  	the start index of the search region.
	 * @param pEnd		the end index of the search region.
	 * @return true, if the like matches.
	 */
	private static boolean like(String pSource, int pSrcStart, int pSrcEnd, 
			                    String pSearch, int pStart,    int pEnd)
	{
		int pos = pSrcStart;
	  	for (int i = pStart; i < pEnd; i++) 
	  	{
	  		char ch = pSearch.charAt(i);
	  		if (ch == '*') 
	  		{
	  			if (i == pEnd - 1) 
	  			{
	  				return true;
	  			}
	  			int nStart = i + 1;
	  			while (pos < pSrcEnd) 
	  			{ 
	  				if (like(pSource, pos, pSrcEnd, pSearch, nStart, pEnd))
	  				{
	  					return true;
	  				}
	  				pos++;
	  			}
	  			return false;
	  		}
	  		else if (pos == pSrcEnd) 
	  		{
	  			return false;
	  		}
	  		else 
	  		{
	  			if (ch != '?' && ch != pSource.charAt(pos)) 
	  			{
	  				return false;
	  			}
	  			pos++;
	  		}
	  	}
	  	return pos == pSrcEnd;
	}

	/**
	 * Calculates the Damerau-Levenshtein distance between two strings.
	 * 
	 * The return value is basically the number of operations needed to turn one
	 * string into another. An operation in this context is a insertion, deletion, substitution or transposition.
	 * 
	 * @param pStringA the first string. Needs to be not {@code null}.
	 * @param pStringB the second string. Needs to be not {@code null}.
	 * @return the Damerau-Levenshtein distance.
	 * @see <a href="http://en.wikipedia.org/wiki/Damerau%E2%80%93Levenshtein_distance">http://en.wikipedia.org/wiki/Damerau%E2%80%93Levenshtein_distance</a>
	 */
	public static int levenshteinDistance(String pStringA, String pStringB)
	{
		int[][] distanceTable = new int[pStringA.length() + 1][pStringB.length() + 1];
		
		for (int indexA = 0; indexA <= pStringA.length(); indexA++)
		{
			distanceTable[indexA][0] = indexA;
		}
		for (int indexB = 0; indexB <= pStringB.length(); indexB++)
		{
			distanceTable[0][indexB] = indexB;
		}
		
		for (int indexA = 1; indexA <= pStringA.length(); indexA++)
		{
			for (int indexB = 1; indexB <= pStringB.length(); indexB++)
			{
				int cost = 0;
				
				if (pStringA.charAt(indexA - 1) != pStringB.charAt(indexB - 1))
				{
					cost = 1;
				}
				
				int deletion = distanceTable[indexA - 1][indexB] + 1;
				int insertion = distanceTable[indexA][indexB - 1] + 1;
				int substitution = distanceTable[indexA - 1][indexB - 1] + cost;
				
				distanceTable[indexA][indexB] = Math.min(Math.min(deletion, insertion), substitution);
				
				if (indexA > 1 && indexB > 1)
				{
					if (pStringA.charAt(indexA - 1) == pStringB.charAt(indexB - 2) && pStringA.charAt(indexA - 2) == pStringB.charAt(indexB - 1))
					{
						int transposition = distanceTable[indexA - 2][indexB - 2] + cost;
						distanceTable[indexA][indexB] = Math.min(distanceTable[indexA][indexB], transposition);
					}
				}
			}
		}
		
		return distanceTable[pStringA.length()][pStringB.length()];
	}
	
	/**
	 * Fast like search in Strings with wildcard(* and ?) support.
	 *  
	 * @param pSource any string
	 * @param pSearch search pattern with or without wildcards.
	 * @return <code>true</code> if, and only if, the string matches the pattern
	 */
	public static boolean like(String pSource, String pSearch) 
	{
	    if (pSource == null && pSearch == null) 
	    {
	    	return true;
	    }
	    else if (pSource == null || pSearch == null) 
	    {
	    	return false;
	    }
	    else 
	    {
	    	return like(pSource, 0, pSource.length(), pSearch, 0, pSearch.length());
	    }
	}

	/**
	 * Sets the first character in each word to uppercase and the rest to lowercase.
	 * 
	 * @param pName the unformatted text
	 * @return the formatted test
	 */
	public static String formatInitCap(String pName)
	{
		return formatInitCap(pName, false);
	}
	
	/**
	 * Sets the first character in each word to uppercase and the rest to lowercase.
	 * 
	 * @param pName the unformatted or formatted text
	 * @param pRemoveSpaces <code>true</code> to remove whitespace characters from the result
	 * @return the formatted test
	 */
	public static String formatInitCap(String pName, boolean pRemoveSpaces)
	{
		if (pName == null)
		{
			return null;
		}
		
		int     iLen   = pName.length();
	    boolean bStart = true;
	    boolean bWhite;
	    
	    int iStart = -1;
	    int iEnd = -1;

	    StringBuilder sbResult = new StringBuilder(iLen);	    
	    
	    for (int i = 0; i < iLen; i++) 
	    {
	    	char ch = pName.charAt(i);
	    	
	    	bWhite = Character.isWhitespace(ch);
	    	
	    	if (!bWhite && iStart == -1)
	    	{
	    		iStart = i;
	    	}
	    	
	    	if (iStart >= 0)
	    	{
		    	if (Character.isLetter(ch)) 
		    	{
		    		if (bStart) 
		    		{
		    			ch = Character.toUpperCase(ch);
		    			bStart = false;
		    		}
		    		else 
		    		{
		    			ch = Character.toLowerCase(ch);
		    		}
		    		
			    	sbResult.append(ch);
		    	}
		    	else if (bWhite || ch == '_')
		    	{
		    		bWhite = true;
		    		bStart = true;
		    	
		    		if (!pRemoveSpaces)
		    		{
		    	    	sbResult.append(' ');
		    		}
		    	}
		    	else
		    	{
		    		sbResult.append(ch);
		    	}
		    	
	    		if (!bWhite)
	    		{
	    			iEnd = sbResult.length();
	    		}
	    	}
	    }
	    
	    if (iStart >= 0 && iEnd >= 0)
	    {
	    	return sbResult.substring(0, iEnd);
	    }
	    
	    return sbResult.toString();
	}	
	
    /**
     * Formats a member name. A member starts always with a lower case letter.
     * 
     * @param pName the member name
     * @return the formatted member name
     */
    public static String formatMemberName(String pName)
    {
    	return formatMethodName(null, pName);
    }	

    /**
     * Formats a method name with a given property name and the method prefix.
     * 
     * @param pPrefix the method prefix, an empty string or null if the method name has no prefix
     * @param pName the unformatted method name
     * @return the formatted method name. The first character is upper case when a prefix is used,
     *         and the first character is lower case when no prefix is used. The other characters 
     *         are unchanged. 
     */
    public static String formatMethodName(String pPrefix, String pName)
    {
    	if (pName != null && pName.trim().length() > 0)
    	{
    		if (pPrefix == null)
    		{
    			return Character.toLowerCase(pName.charAt(0)) + pName.substring(1);
    		}
    		else
    		{
    			return pPrefix + Character.toUpperCase(pName.charAt(0)) + pName.substring(1);
    		}
    	}
    	
    	return null;
    }
    
    /**
     * Converts any text to a member name. If the text contains whitespaces or '_' they will be removed. If the text
     * contains special characters like German umlauts, the characters will be replaced by appropriate ASCII characters.
     * 
     * @param pName the text
     * @return the converted member name
     */
    public static String convertToMemberName(String pName)
    {
    	return convertToMethodName(null, pName);
    }
    
    /**
     * Converts any text to a method name. If the text contains whitespaces or '_' they will be removed. If the text
     * contains special characters like German umlauts, the characters will be replaced by appropriate ASCII characters.
     * If the name contains no text, null will be returned.
     * 
     * @param pPrefix the method prefix e.g. get, set, is, has, ...
     * @param pName the text
     * @return the converted member name
     */
    public static String convertToMethodName(String pPrefix, String pName)
    {
    	if (pName != null && pName.length() > 0)
    	{
    		int     iLen   = pName.length();
    	    boolean bStart = false;
    	    boolean bFirstLetter = false;

    	    boolean bLetter;
    	    
    	    String sReplace;
    	    
    	    StringBuilder sbResult = new StringBuilder();
    	    
    	    for (int i = 0; i < iLen; i++) 
    	    {
    	    	char ch = pName.charAt(i);

    	    	bLetter = Character.isLetter(ch);

	    		sReplace = hmpReplaceCharacters.get(Character.valueOf(ch));
    	    	
        	    if (!bFirstLetter)
        	    {
        	    	if (sReplace != null || bLetter)
        	    	{
        	    		bFirstLetter = true;
        	    	}
        	    }
        	    
        	    if (bFirstLetter)
        	    {
        	    	if (sReplace != null)
        	    	{
        	    		if (bStart) 
        	    		{
        	    			if (sReplace.length() > 0)
        	    			{
        	    				sReplace = Character.toUpperCase(sReplace.charAt(0)) + sReplace.substring(1).toLowerCase();
        	    			}
        	    			else
        	    			{
        	    				sReplace = sReplace.toUpperCase();
        	    			}
        	    			
        	    			bStart = false;
        	    		}
        	    		else
        	    		{
        	    			sReplace = sReplace.toLowerCase();
        	    		}

        	    		sbResult.append(sReplace);
        	    	}
        	    	else if (bLetter) 
        	    	{
        	    		if (bStart) 
        	    		{
        	    			ch = Character.toUpperCase(ch);
        	    			bStart = false;
        	    		}
        	    		else
        	    		{
        	    			ch = Character.toLowerCase(ch);
        	    		}
        	    		
        		    	sbResult.append(ch);
        	    	}
        	    	else if (Character.isWhitespace(ch) || ch == '_')
        	    	{
        	    		bStart = true;
        	    	}
        	    	else if (Character.isJavaIdentifierPart(ch))
        	    	{
        	    		sbResult.append(ch);
        	    	}
        	    }
    	    }
    		
    	    if (sbResult.length() > 0)
    	    {
	    		if (pPrefix == null)
	    		{
	    			return Character.toLowerCase(sbResult.charAt(0)) + sbResult.substring(1);
	    		}
	    		else
	    		{
	    			return pPrefix + Character.toUpperCase(sbResult.charAt(0)) + sbResult.substring(1);
	    		}
    	    }
    	}
    	
    	return null;
    }

    /**
     * Converts text to a vaild name. All whitespaces will be removed and non ASCII characters will
     * be replaced with ASCII characters, if this is possible.
     * 
     * @param pText any text
     * @return a valid name
     */
    public static String convertToName(String pText)
    {
    	if (pText != null && pText.length() > 0)
    	{
    	    String sReplace;
    	    
    	    StringBuilder sbResult = new StringBuilder();

    	    boolean bValidStart = false;
    	    
    	    for (int i = 0, length = pText.length(); i < length; i++) 
    	    {
    	    	char ch = pText.charAt(i);

    	    	sReplace = hmpReplaceCharacters.get(Character.valueOf(ch));

    	    	if (!bValidStart)
    	    	{
	    	    	if (sReplace != null)
	    	    	{
	    	    		bValidStart = Character.isLetter(sReplace.charAt(0)) || Character.isDigit(sReplace.charAt(0));
	    	    	}
	    	    	else
	    	    	{
	    	    		bValidStart = Character.isLetter(ch) || Character.isDigit(ch);
	    	    	}
    	    	}
    	    	
    	    	if (bValidStart)
    	    	{
	    	    	if (sReplace != null)
	    	    	{
	    	    		sbResult.append(sReplace);
	    	    	}
	    	    	else if (!Character.isWhitespace(ch) && Character.isJavaIdentifierPart(ch))
	    	    	{
	    	    		sbResult.append(ch);
	        	    }
    	    	}
    	    }
    	    
    	    return sbResult.toString();
    	}
    	
    	return pText;
    }

    /**
     * Converts a method name in a human readable format. The prefix is removed and spaces are inserted where the
     * character case is changed e.g. <code>getMyMethodName</code> is converted to <code>My Method Name</code>.
     * No special character sequences are replaced e.g. ue is not replaced with  (german umlaut).
     * 
     * @param pMethod the method name
     * @return the human readable name
     * @see #convertMethodNameToText(String, boolean)
     */
    public static String convertMethodNameToText(String pMethod)
    {
    	return convertMethodNameToText(pMethod, false);
    }    
    
    /**
     * Converts a method name in a human readable format. The prefix is removed and spaces are inserted where the
     * character case is changed e.g. <code>getMyMethodName</code> is converted to <code>My Method Name</code>.
     * 
     * @param pMethod the method name
     * @param pReplaceSpecialCharacterSequences <code>true</code> to replace e.g. ue to  (german umlaut) 
     * @return the human readable name
     */
    public static String convertMethodNameToText(String pMethod, boolean pReplaceSpecialCharacterSequences)
    {
    	if (pMethod != null && pMethod.length() > 0)
    	{
    		boolean bName = false;
    		
    		StringBuilder sbResult = new StringBuilder();
    		
    		Character chReplace;
    		
    		int iJumpBack;
    		
    	    for (int i = 0, length = pMethod.length(); i < length; i++) 
    	    {
    	    	char ch = pMethod.charAt(i);

    	    	iJumpBack = 1;
    	    	
    	    	if (pReplaceSpecialCharacterSequences)
    	    	{
		    		chReplace = hmpReplaceReverseCharacters.get(String.valueOf(ch));
		    		
		    		if (chReplace == null)
		    		{
		    			if (i < length - 1)
		    			{
		    				chReplace = hmpReplaceReverseCharacters.get(pMethod.substring(i, i + 2));
		    				
		    				if (chReplace != null)
		    				{
		    					i++;
		    					iJumpBack++;
		    				}
		    			}
		    		}
		    		
		    		if (chReplace != null)
		    		{
		    			ch = chReplace.charValue();
		    		}
    	    	}
    	    	else
    	    	{
    	    		chReplace = null;
    	    	}
    	    	
    	    	if (!bName)
    	    	{
    	    		bName = Character.isUpperCase(ch) || ch == '_';
    	    	}
    	    	
    	    	if (bName)
    	    	{
    	    		if (sbResult.length() > 0)
    	    		{
        	    		if (Character.isUpperCase(ch))
        	    		{
        	    			char chPrevious = pMethod.charAt(i - iJumpBack);

        	    			if (!Character.isUpperCase(chPrevious) && chPrevious != '_')
        	    			{
        	    				sbResult.append(" ");
        	    			}
    	    			}
    	    		}
    	    		
    	    		if (ch == '_')
    	    		{
    	    			if (sbResult.length() > 0)
    	    			{
    	    				sbResult.append(" ");
    	    			}
    	    		}
    	    		else
    	    		{
    	    			sbResult.append(ch);
    	    		}
    	    	}
    	    }
    	    
    	    return sbResult.toString();
    	}
    	
    	return pMethod;
    }

    /**
     * Converts a member name in a human readable format e.g. <code>myMethodName</code> is converted to <code>My Method Name</code>.
     * No special character sequences are replaced e.g. ue is not replaced with  (german umlaut).
     * 
     * @param pMember the member name
     * @return the human readable name
     * @see #convertMethodNameToText(String, boolean)
     */
    public static String convertMemberNameToText(String pMember)
    {
    	return convertMemberNameToText(pMember, false);
    }    
    
    /**
     * Converts a member name in a human readable format e.g. <code>myMethodName</code> is converted to <code>My Method Name</code>.
     * 
     * @param pMember the member name
     * @param pReplaceSpecialCharacterSequences <code>true</code> to replace e.g. ue to  (german umlaut) 
     * @return the human readable name
     */
    public static String convertMemberNameToText(String pMember, boolean pReplaceSpecialCharacterSequences)
    {
    	if (pMember != null && pMember.length() > 0)
    	{
    		StringBuilder sbResult = new StringBuilder();
    		
    		boolean bFirstChar = true;
    		
    		Character chReplace;
    		
    		int iJumpBack;
    		
    	    for (int i = 0, length = pMember.length(); i < length; i++) 
    	    {
    	    	char ch = pMember.charAt(i);

    	    	iJumpBack = 1;
    	    	
    	    	if (pReplaceSpecialCharacterSequences)
    	    	{
		    		chReplace = hmpReplaceReverseCharacters.get(String.valueOf(ch));
		    		
		    		if (chReplace == null)
		    		{
		    			if (i < length - 1)
		    			{
		    				chReplace = hmpReplaceReverseCharacters.get(pMember.substring(i, i + 2));
		    				
		    				if (chReplace != null)
		    				{
		    					i++;
		    					iJumpBack++;
		    				}
		    			}
		    		}
		    		
		    		if (chReplace != null)
		    		{
		    			ch = chReplace.charValue();
		    		}
    	    	}
    	    	
	    		if (sbResult.length() > 0)
	    		{
    	    		if (Character.isUpperCase(ch))
    	    		{
    	    			char chPrevious = pMember.charAt(i - iJumpBack);

    	    			if (!Character.isUpperCase(chPrevious) && chPrevious != '_')
    	    			{
    	    				sbResult.append(" ");
    	    			}
	    			}
	    		}
	    		
	    		if (bFirstChar)
	    		{
	    			if (Character.isLetter(ch))
	    			{
	    				sbResult.append(Character.toUpperCase(ch));
	    				
	    				bFirstChar = false;
	    			}
	    			else
	    			{
	    				sbResult.append(ch);
	    			}
	    		}
	    		else
	    		{
	    			if (ch == '_')
	    			{
	    				sbResult.append(" ");
	    			}
	    			else
	    			{
	    				sbResult.append(ch);
	    			}
	    		}
    	    }
    	    
    	    return sbResult.toString();
    	}
    	
    	return pMember;
    }
    
	/**
	 * Gets only specific characters from a string.
	 * 
	 * @param pText any text
	 * @param pType the character return type
	 * @return the alpha numeric characters
	 */
	public static String getText(String pText, TextType pType)
	{
		if (pText == null)
		{
			return null;
		}
		else
		{
			StringBuilder sbName = new StringBuilder();
			
			char ch;
			
			boolean bValidCharacter = false;
			
			for (int i = 0, length = pText.length(); i < length; i++)
			{
				ch = pText.charAt(i);
			
				switch (pType)
				{
					case Letters:
						if (Character.isLetter(ch))
						{
							sbName.append(ch);
						}
						break;
					case Digits:
						if (Character.isDigit(ch))
						{
							sbName.append(ch);
						}
						break;
					case LettersDigits:
						if (Character.isLetter(ch) || Character.isDigit(ch))
						{
							sbName.append(ch);
						}
						break;
					case LettersDigitsSpace:
						if (Character.isLetter(ch) || Character.isDigit(ch) || Character.isSpaceChar(ch))
						{
							sbName.append(ch);
						}
						break;
					case LettersDigitsWhitespace:
						if (Character.isLetter(ch) || Character.isDigit(ch) || Character.isWhitespace(ch))
						{
							sbName.append(ch);
						}
						break;
					case AZLetters:
						if ((ch >= 'A' && ch <= 'Z') 
							|| (ch >= 'a' && ch <= 'z'))
						{
							sbName.append(ch);
						}
						break;
					case AZLettersDigits:
						if ((ch >= 'A' && ch <= 'Z') 
							|| (ch >= 'a' && ch <= 'z')
							|| Character.isDigit(ch))
						{
							sbName.append(ch);
						}
						break;
					case WithoutLeadingDigits:
						if (!bValidCharacter)
						{
							bValidCharacter = !Character.isDigit(ch);
						}
						
						if (bValidCharacter)
						{
							sbName.append(ch);
						}
						break;
					case FromFirstLetter:
						if (!bValidCharacter)
						{
							bValidCharacter = Character.isLetter(ch);
						}
						
						if (bValidCharacter)
						{
							sbName.append(ch);
						}
						break;
					case UpperCase:
						if (Character.isUpperCase(ch))
						{
							sbName.append(ch);
						}
						break;
					case LowerCase:
						if (Character.isLowerCase(ch))
						{
							sbName.append(ch);
						}
						break;
					default:
						//none
				}
			}
			
			return sbName.toString();
		}
	}

    /**
     * Gets only specific characters from a string.
     * 
     * @param pText any text
     * @param pAllowed the allowed characters
     * @return the alpha numeric characters
     */
    public static String getText(String pText, char... pAllowed)
    {
        if (pText == null)
        {
            return null;
        }
        else
        {
            StringBuilder sbName = new StringBuilder();
            
            char ch;
            
            for (int i = 0, length = pText.length(); i < length; i++)
            {
                ch = pText.charAt(i);

                if (ArrayUtil.indexOf(pAllowed, ch) >= 0)
                {
                    sbName.append(ch);
                }                    
            }
            
            return sbName.toString();
        }
    }
	
	/**
	 * Removes specific characters from a string.
	 * 
	 * @param pText a string
	 * @param pRemove the characters which should be removed
	 * @return the string without specified characters
	 */
	public static String removeCharacters(String pText, char[] pRemove)
	{
		if (pText == null)
		{
			return null;
		}
		
		if (pRemove == null || pRemove.length == 0)
		{
			return pText;
		}
		
		StringBuilder sbName = new StringBuilder();
		
		char ch;
		
		boolean bFound;
		
		for (int i = 0, length = pText.length(); i < length; i++)
		{
			ch = pText.charAt(i);
		
			bFound = false;
			
			for (int j = 0, anzj = pRemove.length; j < anzj && !bFound; j++)
			{
				bFound = (ch == pRemove[j]);
			}

			if (!bFound)
			{
				sbName.append(ch);
			}
		}
		
		return sbName.toString();
	}

    /**
     * Sanitizes the given input so that it can be used as ID (for example in
     * a HTML document).
     * 
     * The definition of "sanitize" in this case is taken from the HTML 4 spec:
     * <blockquote>
     * ID and NAME tokens must begin with a letter ([A-Za-z]) and may be followed
     * by any number of letters, digits ([0-9]), hyphens ("-"), underscores ("_"),
     * colons (":"), and periods (".").
     * </blockquote>
     * 
     * @param pId the id to sanitize.
     * @return the sanitized id. {@code null} if the given id was null.
     */
    public static String sanitizeId(String pId)
    {
    	// Not using StringUtil.isEmty(String) here because we do want
    	// to process strings that only contain whitespaces.
    	if (pId == null || pId.length() == 0)
    	{
    		return pId;
    	}
    	
    	StringBuilder sanitizedId = new StringBuilder();
    	
    	boolean startsWithLetter = false;
    	
    	for (char piece : pId.toCharArray())
    	{
    		// RegEx representation: [a-zA-Z0-9\-:._]
    		if ((piece >= 'A' && piece <= 'Z')
    				|| (piece >= 'a' && piece <= 'z')
    				|| (
    						(Character.isDigit(piece) || piece == '-' || piece == ':' || piece == '.' || piece == '_')
    						&& startsWithLetter))
    		{
    			startsWithLetter = true;
    			sanitizedId.append(piece);
    		}
    	}
    	
    	return sanitizedId.toString();
    }

    /**
     * Returns a string representation of the "deep contents" of the specified
     * object. If the object contains other objects as elements, the string
     * representation contains their contents and so on. This method is
     * designed for converting multidimensional arrays, Collections and Maps 
     * to strings.
     *
     * <p>The string representation consists of a list of the object's
     * elements, enclosed in brackets (<tt>"[]"</tt> or <tt>"{}"</tt>). Adjacent
     * elements are separated by the characters <tt>", "</tt> (a comma
     * followed by a space).
     *
     * <p>To avoid infinite recursion, if the specified object contains itself
     * as an element, or contains an indirect reference to itself through one
     * or more levels of arrays, the self-reference is converted to the string
     * <tt>"#REF#"</tt>. For example, an array containing only a reference
     * to itself would be rendered as <tt>"[#REF#]"</tt>.
     *
     * <p>This method returns <tt>"null"</tt> if the specified object
     * is <tt>null</tt>.
     *
     * @param pObject the object whose string representation to return
     * @return a string representation of <tt>pObject</tt>
     */
    public static String toString(Object pObject) 
    {
        return toString(pObject, -1);
    }

    /**
     * Returns a string representation of the "deep contents" of the specified
     * object. If the object contains other objects as elements, the string
     * representation contains their contents and so on. This method is
     * designed for converting multidimensional arrays, Collections and Maps 
     * to strings.
     *
     * <p>The string representation consists of a list of the object's
     * elements, enclosed in brackets (<tt>"[]"</tt> or <tt>"{}"</tt>). Adjacent
     * elements are separated by the characters <tt>", "</tt> (a comma
     * followed by a space).
     *
     * <p>To avoid infinite recursion, if the specified object contains itself
     * as an element, or contains an indirect reference to itself through one
     * or more levels of arrays, the self-reference is converted to the string
     * <tt>"#REF#"</tt>. For example, an array containing only a reference
     * to itself would be rendered as <tt>"[#REF#]"</tt>.
     *
     * <p>This method returns <tt>"null"</tt> if the specified object
     * is <tt>null</tt>.
     *
     * @param pObject the object whose string representation to return
     * @param pMaxArrayLength the maximum length of arrays. If the length of an array is exceeded, the elements won't be printed.
     *                        Instead a placeholder like byte[n] will be printed.
     * @return a string representation of <tt>pObject</tt>
     */    
    public static String toString(Object pObject, int pMaxArrayLength) 
    {
        if (pObject == null)
        {
            return "null";
        }
        
        StringBuilder buf = new StringBuilder();

        deepToString(pObject, buf, new IdentityHashMap<Object, Object>(), pMaxArrayLength);
        
        return buf.toString();
    }
    
    /**
     * Returns a string representation of the "deep contents" of the specified
     * object. If the object contains other objects as elements, the string
     * representation contains their contents and so on. This method is
     * designed for converting multidimensional arrays, Collections and Maps 
     * to strings.
     *
     * <p>The string representation consists of a list of the object's
     * elements, enclosed in brackets (<tt>"[]"</tt> or <tt>"{}"</tt>). Adjacent
     * elements are separated by the characters <tt>", "</tt> (a comma
     * followed by a space).
     *
     * <p>To avoid infinite recursion, if the specified object contains itself
     * as an element, or contains an indirect reference to itself through one
     * or more levels of arrays, the self-reference is converted to the string
     * <tt>"#REF#"</tt>. For example, an array containing only a reference
     * to itself would be rendered as <tt>"[#REF#]"</tt>.
     *
     * <p>This method returns <tt>"null"</tt> if the specified object
     * is <tt>null</tt>.
     *
     * @param pObject the object whose string representation to return
     * @param pBuffer the output buffer for recursive calls instead of a return value
     * @param pSelfCache the object cache to avoid recursion
     * @param pMaxArrayLength the maximum length of arrays. If the length of an array is exceeded, the elements won't be printed.
     *                        Instead a placeholder like #byte[n]# will be printed.
     */
    private static void deepToString(Object pObject, StringBuilder pBuffer, IdentityHashMap<Object, Object> pSelfCache, int pMaxArrayLength)
    {
        if (pObject == null) 
        {
            pBuffer.append("null");
            return;
        }

    	if (pSelfCache.containsKey(pObject))
        {
            pBuffer.append("#REF#");
            return;
        }
        
        pSelfCache.put(pObject, Boolean.TRUE);
        
        if (pObject instanceof Collection<?>)
        {
        	Collection<?> coll = (Collection<?>)pObject;
        	
            pBuffer.append('{');
            
            int i = 0;
        	
            for (Iterator it = coll.iterator(); it.hasNext(); i++)
        	{
                if (i != 0)
                {
                    pBuffer.append(", ");
                }
            	
        		deepToString(it.next(), pBuffer, pSelfCache, pMaxArrayLength);
        	}

        	pBuffer.append('}');        	
        }
        else if (pObject instanceof Map<?, ?>)
        {
        	Map<?, ?> map = (Map<?, ?>)pObject;
        	
            pBuffer.append('{');
            
            int i = 0;
            
        	for (Map.Entry<?, ?> entry : map.entrySet())
        	{
                if (i != 0)
                {
                    pBuffer.append(", ");
                }
                
                pBuffer.append('[');
        		deepToString(entry.getKey(), pBuffer, pSelfCache, pMaxArrayLength);
        		pBuffer.append(", ");
        		deepToString(entry.getValue(), pBuffer, pSelfCache, pMaxArrayLength);
                pBuffer.append(']');
        		
                i++;
        	}
            
            pBuffer.append('}');
        }
        else if (pObject instanceof Dictionary<?, ?>)
        {
        	Dictionary<?, ?> dict = (Dictionary<?, ?>)pObject;
        	
            pBuffer.append('{');
            
            int i = 0;
            
            Object oKey;
            
        	for (Enumeration<?> enKeys = dict.keys(); enKeys.hasMoreElements();)
        	{
        		oKey = enKeys.nextElement();
        		
                if (i != 0)
                {
                    pBuffer.append(", ");
                }
                
                pBuffer.append('[');
        		deepToString(oKey, pBuffer, pSelfCache, pMaxArrayLength);
        		pBuffer.append(", ");
        		deepToString(dict.get(oKey), pBuffer, pSelfCache, pMaxArrayLength);
                pBuffer.append(']');
        		
                i++;
        	}
            
            pBuffer.append('}');
        }
        else
        {
            Class eClass = pObject.getClass();

            if (eClass.isArray()) 
            {
                pBuffer.append('[');

                if (eClass == byte[].class)
                {
                    if (pMaxArrayLength >= 0 && ((byte[])pObject).length > pMaxArrayLength)
                    {
                        pBuffer.append("#byte[");
                        pBuffer.append(((byte[])pObject).length);
                        pBuffer.append("]#");
                    }
                    else
                    {
                        pBuffer.append(Arrays.toString((byte[])pObject));
                    }
                }
                else if (eClass == short[].class)
                {
                    if (pMaxArrayLength >= 0 && ((short[])pObject).length > pMaxArrayLength)
                    {
                        pBuffer.append("#short[");
                        pBuffer.append(((short[])pObject).length);
                        pBuffer.append("]#");
                    }
                    else
                    {
                        pBuffer.append(Arrays.toString((short[])pObject));
                    }
                }
                else if (eClass == int[].class)
                {
                    if (pMaxArrayLength >= 0 && ((int[])pObject).length > pMaxArrayLength)
                    {
                        pBuffer.append("#int[");
                        pBuffer.append(((int[])pObject).length);
                        pBuffer.append("]#");
                    }
                    else
                    {
                        pBuffer.append(Arrays.toString((int[])pObject));
                    }
                }
                else if (eClass == long[].class)
                {
                    if (pMaxArrayLength >= 0 && ((long[])pObject).length > pMaxArrayLength)
                    {
                        pBuffer.append("#long[");
                        pBuffer.append(((long[])pObject).length);
                        pBuffer.append("]#");
                    }
                    else
                    {
                        pBuffer.append(Arrays.toString((long[])pObject));
                    }
                }
                else if (eClass == char[].class)
                {
                    if (pMaxArrayLength >= 0 && ((char[])pObject).length > pMaxArrayLength)
                    {
                        pBuffer.append("#char[");
                        pBuffer.append(((char[])pObject).length);
                        pBuffer.append("]#");
                    }
                    else
                    {
                        pBuffer.append(Arrays.toString((char[])pObject));
                    }
                }
                else if (eClass == float[].class)
                {
                    if (pMaxArrayLength >= 0 && ((float[])pObject).length > pMaxArrayLength)
                    {
                        pBuffer.append("#float[");
                        pBuffer.append(((float[])pObject).length);
                        pBuffer.append("]#");
                    }
                    else
                    {
                        pBuffer.append(Arrays.toString((float[])pObject));
                    }
                }
                else if (eClass == double[].class)
                {
                    if (pMaxArrayLength >= 0 && ((short[])pObject).length > pMaxArrayLength)
                    {
                        pBuffer.append("#double[");
                        pBuffer.append(((double[])pObject).length);
                        pBuffer.append("]#");
                    }
                    else
                    {
                        pBuffer.append(Arrays.toString((double[])pObject));
                    }
                }
                else if (eClass == boolean[].class)
                {
                    if (pMaxArrayLength >= 0 && ((boolean[])pObject).length > pMaxArrayLength)
                    {
                        pBuffer.append("#boolean[");
                        pBuffer.append(((boolean[])pObject).length);
                        pBuffer.append("]#");
                    }
                    else
                    {
                        pBuffer.append(Arrays.toString((boolean[])pObject));
                    }
                }
                else 
                { 
                	// element is an array of object references
                	Object[] array = (Object[])pObject;
                	
                	if (pMaxArrayLength >= 0 && array.length > pMaxArrayLength)
                	{
                        pBuffer.append("#Object[");
                        pBuffer.append(array.length);
                        pBuffer.append("]#");
                	}
                	else
                	{
                    	for (int i = 0; i < array.length; i++)
                    	{
                            if (i != 0)
                            {
                                pBuffer.append(", ");
                            }
    
                            deepToString(array[i], pBuffer, pSelfCache, pMaxArrayLength);
                    	}
                	}
                }
                
                pBuffer.append(']');
            }
            else
            {
            	pBuffer.append(pObject.toString());
            }
        }
        
        pSelfCache.remove(pObject);
    }
    
    /**
     * Removes the begin and end quote of strings, e.g. <code>'text'</code> will be translated to <code>text</code>
     * 
     * @param pText the quoted text
     * @param pQuote the quote character e.g. <code>'</code> or <code>"</code>
     * @return the <code>pText</code> without begin and end quote
     */
    public static String removeQuotes(String pText, String pQuote)
    {
    	return removeQuotes(pText, pQuote, pQuote);
    }
    
    /**
     * Removes the begin and end quote of strings, e.g. <code>'text'</code> will be translated to <code>text</code>
     * 
     * @param pText the quoted text
     * @param pStartQuote the start quote character e.g. <code>'</code> or <code>"</code> or <code>(</code>
     * @param pEndQuote the end quote character e.g. <code>'</code> or <code>"</code> or <code>)</code>
     * @return the <code>pText</code> without begin and end quote
     */
    public static String removeQuotes(String pText, String pStartQuote, String pEndQuote)
    {
    	if (pText == null)
    	{
    		return null;
    	}
    	
		int iFirst;
		int iLast;
		
		iFirst = pText.indexOf(pStartQuote);
		iLast = pText.lastIndexOf(pEndQuote);

		if (iFirst >= 0 && iLast > iFirst)
		{
			return pText.substring(iFirst + pStartQuote.length(), iLast);
		}

		return pText;
    }    
	
	/**
	 * Adds a quote character to the begin and end of text. If the text contains the quote character
	 * then an additional quote character will be added e.g. "value is ""0"""
	 * 
	 * @param pText the text to protect
	 * @param pQuote the quote character
	 * @return the protected text
	 */
	public static String quote(String pText, char pQuote)
	{
		StringBuilder sbBuffer = new StringBuilder();
		
		sbBuffer.append(pQuote);
		
		if (pText != null)
		{
			char ch;
			
			for (int i = 0, length = pText.length(); i < length; i++)
			{
				ch = pText.charAt(i);
				
				if (ch == pQuote)
				{
					sbBuffer.append(pQuote);
				}
				
				sbBuffer.append(ch);
			}
		}
		
		sbBuffer.append(pQuote);
		
		return sbBuffer.toString();
	}

	/**
	 * Returns a new string resulting from replacing all occurrences of <code>pOld</code> in this string with <code>pNew</code>.
	 *  
	 * @param pText the original text
	 * @param pOld the text to replace
	 * @param pNew the replacemenet
	 * @return the resulting string
	 */
	public static String replace(String pText, String pOld, String pNew)
	{
		if (pText == null)
		{
			return null;
		}
		
		if (pOld == null)
		{
			return pText;
		}
		
		if (pNew == null)
		{
			pNew = "";
		}
		
		int i = 0;

		StringBuilder strBuf = new StringBuilder(pText);

		while ((i = strBuf.indexOf(pOld, i)) >= 0)
		{
			strBuf.replace(i, i + pOld.length(), pNew);

			//move after the replacement to avoid endless loop
			i += pNew.length();
		}

		return strBuf.toString();
	}	
	
	/**
	 * Gets the case sensitive type of a text.
	 * 
	 * @param pText any text or <code>null</code>
	 * @return {@link CaseSensitiveType#NoLetter} if the text contains no letter or the text is <code>null</code>.
	 *         {@link CaseSensitiveType#LowerCase} if the text contains at least one letter and all available letters are
	 *         lower case.
	 *         {@link CaseSensitiveType#UpperCase} if the text contains at least one letter and all available letters are
	 *         upper case.
	 *         {@link CaseSensitiveType#MixedCase} if the text contains at least two letters and the text contains lower
	 *         and upper case letters.
	 */
	public static CaseSensitiveType getCaseSensitiveType(String pText)
	{
		if (pText == null)
		{
			return CaseSensitiveType.NoLetter;
		}
		
		char ch;
		
		boolean bUpper = false;
		boolean bLower = false;
		
		for (int i = 0, length = pText.length(); i < length; i++)
		{
			ch = pText.charAt(i);
		
			if (Character.isLetter(ch))
			{
				if (Character.isLowerCase(ch))
				{
					bLower = true;
				}
				else
				{
					bUpper = true;
				}
				
				if (bUpper && bLower)
				{
					return CaseSensitiveType.MixedCase;
				}
			}
		}
		
		if (bLower)
		{
			return CaseSensitiveType.LowerCase;
		}
		else if (bUpper)
		{
			return CaseSensitiveType.UpperCase;
		}
		else
		{
			return CaseSensitiveType.NoLetter;
		}
	}
	
	/**
	 * Gets the character type of a text.
	 * 
	 * @param pText any text or <code>null</code>
	 * @return {@link CharacterType#None} if the text is empty or <code>null</code>.
	 *         {@link CharacterType#Letters} if the text only contains letters.
	 *         {@link CharacterType#Digits} if the text only contains digits.
	 *         {@link CharacterType#LettersDigits} if the text only contains letters and digits.
	 *         {@link CharacterType#LettersSpecial} if the text contains letters and other characters but no digits.
	 *         {@link CharacterType#DigitsSpecial} if the text contains digits and other characters but no letters.
	 *         {@link CharacterType#LettersDigitsWhitespace} if the text only contains letters, digits and whitespaces.
	 *         {@link CharacterType#LettersDigitsSpace} if the text only contains letters, digits and spaces.
	 *         {@link CharacterType#OnlyWhitespace} if the text only contains whitespaces.
	 *         {@link CharacterType#OnlySpecial} if the text contains no letters and no digits.
	 *         {@link CharacterType#All} if the text contains letters, digits and other characters.
	 */
	public static CharacterType getCharacterType(String pText)
	{
		if (pText == null)
		{
			return CharacterType.None;
		}
		
		char ch;
		
		boolean bLetter   = false;
		boolean bDigit    = false;
		boolean bWhite    = false;
		boolean bSpace    = false;
		boolean bNotSpace = false;
		boolean bSpecial  = false;
		boolean bOther    = false; 
		
		for (int i = 0, length = pText.length(); i < length; i++)
		{
			ch = pText.charAt(i);

			if (Character.isLetter(ch))
			{
				bLetter = true;
			}
			else if (Character.isDigit(ch))
			{
				bDigit = true;
			}
			else if (Character.isWhitespace(ch))
			{
				bWhite = true;
				bSpecial = true;
				
				if (Character.isSpaceChar(ch))
				{
					bSpace = true;
				}
				else
				{
					bNotSpace = true;
				}
			}
			else
			{
				bSpecial = true;
				bOther = true;
			}
			
			if (bLetter && bDigit && bWhite && bOther)
			{
				return CharacterType.All;
			}
		}

		if (bLetter && bDigit && bWhite && !bOther)
		{
			if (bSpace && !bNotSpace)
			{
				return CharacterType.LettersDigitsSpace;
			}
			
			return CharacterType.LettersDigitsWhitespace;
		}
		
		if (bLetter && bDigit && !bWhite && !bOther)
		{
			return CharacterType.LettersDigits;
		}

		if (bLetter && !bDigit && !bWhite && !bOther)
		{
			return CharacterType.Letters;
		}
		
		if (bLetter && !bDigit && (bWhite || bOther))
		{
			return CharacterType.LettersSpecial;
		}
		
		if (bDigit && !bLetter && !bWhite && !bOther)
		{
			return CharacterType.Digits;
		}
		
		if (bDigit && !bLetter && (bWhite || bOther))
		{
			return CharacterType.DigitsSpecial;
		}
		
		if (bWhite && !bOther)
		{
			return CharacterType.OnlyWhitespace;
		}
		
		if (bSpecial)
		{
			return CharacterType.OnlySpecial;
		}
		
		return CharacterType.None;
	}

	/**
	 * Gets whether the given text contains no characters.
	 * 
	 * @param pText a text
	 * @return <code>true</code> if <code>pText</code> is null or has 0 characters (whitespaces will be ignored)
	 */
	public static boolean isEmpty(String pText)
	{
		return pText == null || pText.trim().length() == 0;
	}
	
	/**
	 * Concatenates the given {@link List} of {@link Object}s with the given
	 * {@link String delimiter}. {@link Object#toString()} is used to convert
	 * the {@link Object} to a {@link String}, if the {@link Object} is
	 * {@code null}, the {@link String} {@code "null"} will be appended.
	 * 
	 * @param pDelimiter the {@link String delimiter} to use.
	 * @param pObjects the {@link Object}s to concatenate.
	 * @return the concatenated {@link String}. An empty {@link String} if
	 *         the {@link List} is either {@code null} or empty.
	 * @see #concat(String, Object...)
	 * @see #concat(String, String...)
	 * @see Object#toString()
	 */
	public static String concat(String pDelimiter, List<? extends Object> pObjects)
	{
		if (pObjects == null || pObjects.isEmpty())
		{
			return "";
		}
		
		StringBuilder result = new StringBuilder();
		
		for (Object object : pObjects)
		{
			result.append(object);
			result.append(pDelimiter);
		}
		
		// Remove the last delimiter.
		result.delete(result.length() - pDelimiter.length(), result.length());
		
		return result.toString();
	}
	
	/**
	 * Concatenates the {@link Object}s with the given {@link String delimiter}.
	 * {@link Object#toString()} is used to convert the the {@link Object} to
	 * a {@link String}, if the {@link Object} is {@code null}, the
	 * {@link String} {@code "null"} will be appended.
	 * 
	 * @param pDelimiter the {@link String delimiter} to use.
	 * @param pObjects the {@link Object}s to concatenate.
	 * @return the concatenated {@link String}. An empty {@link String} if
	 *         the {@link List} is either {@code null} or empty.
	 * @see #concat(String, List)
	 * @see #concat(String, String...)
	 * @see Object#toString()
	 */
	public static String concat(String pDelimiter, Object... pObjects)
	{
		return concat(pDelimiter, Arrays.asList(pObjects));
	}
	
	/**
	 * Concats the elements with pDelimiter as separator.
	 * 
	 * @param pDelimiter the delimiter.
	 * @param pElements the elements to concat.
	 * @return the concatenated string.
	 * @see #concat(String, List)
	 * @see #concat(String, Object...)
	 */
	public static String concat(String pDelimiter, String... pElements)
	{
		// I did some performance testing and neither the creation of the List
		// nor that toString() is called on a String (StringBuilder.append)
		// had a mentionable performance hit.
		return concat(pDelimiter, Arrays.asList(pElements));
	}
	
	/**
	 * It formats a string with right Padding over pWidth characters.
	 * 
	 * @param pText		the text to format.
	 * @param pWidth	the width in characters to pad right.
	 * @return the right padded String.
	 */
	public static String padRight(Object pText, int pWidth) 
	{
		if (pText == null)
		{
			return null;
		}
	    return String.format("%1$-" + pWidth + "s", StringUtil.toString(pText));  
	}

	/**
	 * It formats a string with left Padding over pWidth characters.
	 * 
	 * @param pText		the text to format.
	 * @param pWidth	the width in characters to pad left.
	 * @return the left padded String.
	 */
	public static String padLeft(Object pText, int pWidth) 
	{
		if (pText == null)
		{
			return null;
		}
	    return String.format("%1$" + pWidth + "s", StringUtil.toString(pText));  
	}

	/**
	 * Gets whether a text contains at least one whitespace character.
	 * 
	 * @param pText the text
	 * @return <code>true</code> if at least one whitespace character was found, <code>false</code> otherwise
	 */
	public static boolean containsWhitespace(String pText)
	{
		if (pText == null)
		{
			return false;
		}
		
		for (int i = 0, length = pText.length(); i < length; i++)
		{
			if (Character.isWhitespace(pText.charAt(i)))
			{
				return true;
			}
		}
		
		return false;
	}

	/**
	 * Gets a String without any white space. If the text is null an empty string is returned.
	 * 
	 * @param pText the text.
	 * @return the text without white spaces.
	 */
	public static String removeWhiteSpaces(String pText)
	{
		// TODO The name of this function should be changed to removeWhitespace or removeWhitespaces.
		
		if (pText == null || pText.length() == 0)
		{
			return "";
		}
		
		StringBuilder result = new StringBuilder(pText.length());
		
		for (int i = 0, length = pText.length(); i < length; i++)
		{
			char ch = pText.charAt(i);
			if (!Character.isWhitespace(ch))
			{
				result.append(ch);
			}
		}
		
		return result.toString();
	}
	
	/**
	 * Strips any tags from the given text.
	 * 
	 * This does only remove the tags from the string, it does not perform
	 * any sort of parsing or similar, so it does not check for any sort
	 * of malformed syntax and simply removes everything that is surrounded
	 * by &lt;&gt;.
	 * 
	 * Examples:
	 * 
	 * <pre>
	 * &lt;html&gt;Something&lt;/html&gt;					Something
	 * &lt;b&gt;Some&lt;/b&gt;thing&lt;/b&gt;					Something
	 * &lt;html&gt;This is&lt;br&gt;a&lt;br&gt;&lt;br&gt;new line test.&lt;/html&gt;		This isanew line test.
	 * </pre>
	 * 
	 * @param pText the test from which to strip all HTML tags.
	 * @return the given text without HTML tags.
	 */
	public static String stripTags(String pText)
	{
		if (pText == null)
		{
			return "";
		}
		
		StringBuilder result = new StringBuilder();
		
		boolean inTag = false;
		int lastTagStartIndex = 0;
		
		for (int idx = 0, length = pText.length(); idx < length; idx++)
		{
			char ch = pText.charAt(idx);
			
			if (ch == '<')
			{
				if (!inTag)
				{
					inTag = true;
				}
				else
				{
					idx = lastTagStartIndex;
					result.append('<');
					inTag = false;
				}
				
				lastTagStartIndex = idx;
			}
			else if (ch == '>' && inTag)
			{
				inTag = false;
			}
			else if (!inTag)
			{
				result.append(ch);
			}
		}
		
		return result.toString();
	}
	
	/**
	 * Counts the number of a specific characters in a text.
	 * 
	 * @param pText the text
	 * @param pChar the character to search
	 * @return the number of occurences of <code>pChar</code> in <code>pText</code>
	 */
	public static int countCharacter(String pText, char pChar)
	{
		if (pText == null)
		{
			return 0;
		}

		char ch;
		
		int iCount = 0;
		
		for (int i = 0, length = pText.length(); i < length; i++)
		{
			ch = pText.charAt(i);
			
			if (ch == pChar)
			{
				iCount++;
			}
		}
		
		return iCount;
	}

	/**
	 * Uppercase the first character.
	 * 
	 * @param pText the text
	 * @return the text but the first character is guaranteed uppercase.
	 */
	public static String firstCharUpper(String pText)
	{
		if (pText == null || pText.length() == 0)
		{
			return pText;
		}
		
		return Character.toUpperCase(pText.charAt(0)) + pText.substring(1);
	}

    /**
     * Lowercase the first character.
     * 
     * @param pText the text
     * @return the text but the first character is guaranteed lowercase.
     */
    public static String firstCharLower(String pText)
    {
        if (pText == null || pText.length() == 0)
        {
            return pText;
        }
        
        return Character.toLowerCase(pText.charAt(0)) + pText.substring(1);
    }
	
    /**
     * Replaces all placeholders with the values from the defined property. Use following syntax:
     * ${sys:propertyname} (replacement with the system property), ${env:parametername} (replacement with
     * the environment parameter) or ${name} (replacement with the system property or the environment
     * parameter if system property wasn't found). 
     * 
     * @param pValue the value with or without placeholders
     * @return the value with replaced placeholders
     */
    public static String replacePlaceholder(String pValue)
    {
        if (pValue != null)
        {
            int iStart = pValue.indexOf("${");

            if (iStart < 0)
            {
                return pValue;
            }

            StringBuilder sbNewValue = new StringBuilder(pValue);
            
            String sFoundValue;
            String sFoundParam;
            String sFoundParamPrefix;
            String sFoundParamPostfix;
            
            int iPrefixStart;
            int iEnd = 0;
                    
            while (iStart >= 0
                   && iEnd != -1)
            {
                iEnd = sbNewValue.indexOf("}", iStart + 2);
                
                if (iEnd > iStart)
                {                   
                    sFoundParam = sbNewValue.substring(iStart + 2, iEnd);

                    if (sFoundParam.length() > 0)
                    {
                        sFoundParamPrefix = null;
                        sFoundParamPostfix = null;
                        sFoundValue = null;
                        
                        iPrefixStart = sFoundParam.indexOf(':');
                        
                        if (iPrefixStart >= 0)
                        {
                            sFoundParamPrefix = sFoundParam.substring(0, iPrefixStart);
                            sFoundParamPostfix = sFoundParam.substring(iPrefixStart + 1);
                        }
                        
                        if ("sys".equalsIgnoreCase(sFoundParamPrefix))
                        {
                            sFoundValue = System.getProperty(sFoundParamPostfix);
                        }
                        else if ("env".equalsIgnoreCase(sFoundParamPrefix))
                        {
                            sFoundValue = System.getenv(sFoundParamPostfix);
                        }
                        else 
                        {
                            sFoundValue = System.getProperty(sFoundParam);
                            
                            if (sFoundValue == null)
                            {
                                sFoundValue = System.getenv(sFoundParam);
                            }
                        }
                        
                        if (sFoundValue != null)
                        {
                            sbNewValue = sbNewValue.replace(iStart, iEnd + 1, sFoundValue);
                            
                            iEnd = iStart + sFoundValue.length() - 1;
                        }
                    }
                }
                
                iStart = sbNewValue.indexOf("${", iEnd + 1);
            }
            
            return sbNewValue.toString();
        }
        
        return pValue;
    }

    /**
     * Removes all whitespaces before the first non whitespace character.
     * 
     * @param pText the text
     * @return the trimmed text
     */
    public static String ltrim(String pText)
    {
        int i = 0;
        
        int iLength = pText.length();
        
        while (i < iLength 
               && Character.isWhitespace(pText.charAt(i)))
        {
            i++;
        }
        
        return pText.substring(i);
    }
    
    /**
     * Removes all whitespaces after the last non whitespace character.
     * 
     * @param pText the text
     * @return the trimmed text
     */
    public static String rtrim(String pText)
    {
        int i = pText.length() - 1;
        
        while (i >= 0 
               && Character.isWhitespace(pText.charAt(i)))
        {
            i--;
        }
        return pText.substring(0, i + 1);
    }
    
}	// StringUtil
