/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 01.10.2008 - [RH] - creation
 * 13.11.2008 - [RH] - unnecessary throw ModelExcpetions removed
 * 07.04.2009 - [RH] - Interface review 
 *                     get/setDataRow removed
 *                     add/removeDataBook added
 *                     clone to createRowDefinition renamed
 *                     removeColumn removed 
 *                     add/getColumn renamed to add/getColumnDefintion and ModelException added
 * 31.03.2011 - [JR] - #318: add/removeControl, getControls added
 */
package javax.rad.model;

import javax.rad.model.ui.IControl;

/**
 * An <code>IRowDefinition</code> contains all <code>ColumnDefinition</code>'s of a 
 * <code>IDataRow</code>.
 * 
 * @see javax.rad.model.ColumnDefinition
 * 
 * @author Roland Hrmann
 */
public interface IRowDefinition extends Cloneable
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** A constant for using all columns. */
	public static final String[] ALL_COLUMNS = null;
	
		
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Method definitions
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * It register and adds a IDataBook to this IRowDefinition, which uses this IRowDefintion.
	 * The IDataBook implementations has to remove itself in the close() method.
	 * 
	 * @param pDataBook	the {@link IDataBook} to add
	 * @throws ModelException if databaook is already added or adding is not possible  
	 */
	public void addDataBook(IDataBook pDataBook) throws ModelException;
	
	/**
	 * It removes the DataBook reference to this IRowDefinition, which used this IRowDefintion before.
	 * The IDataBook implementations has to remove itself in the close() method.
     *
	 * @param pDataBook the {@link IDataBook} to remove
	 */
	public void removeDataBook(IDataBook pDataBook);

    /**
     * It returns the list of IDataBooks corresponding to this IRowDefintion.
     * @return An IDataBook array of currently using this IRowDefinition.
     */
    public IDataBook[] getDataBooks();

	/**
	 * Returns a cloned <code>IRowDefinition</code> with only a subset of 
	 * <code>ColumnDefinitions</code>, specified by a column name <code>String[]</code>.
	 * 
	 * @param pColumnNames
	 *            column names to use or null for all columns (getRowDefinition().getColumnNames()).
	 * @return a cloned <code>IRowDefinition</code> with only a subset column names
	 *         <code>String[]</code>.
	 * @throws ModelException
	 *             if the <code>IRowDefinition</code> couldn't constructed
	 */
	public IRowDefinition createRowDefinition(String[] pColumnNames) throws ModelException;
	
	/**
	 * Adds an <code>ColumnDefinition</code> to this <code>IRowDefinition</code>.<br>
	 * Implementation need to check, that this <code>ColumnDefinition</code> is only
	 * added in one <code>IRowDefinition</code>. After an IDataBook is registered (at open()) 
	 * to this <code>IRowDefinition</code>, it is not allowed to add columns.
	 * 
	 * @param pColumnDefinition
	 *            the new <code>ColumnDefinition</code>
	 * @throws ModelException 
	 * 			  if the <code>ColumnDefinition</code> name already exists or it is already 
	 *            added to another <code>IRowDefintion</code> or if an IDataBook is registered 
	 *            (at open()) to the <code>IRowDefinition</code>.
	 */
	public void addColumnDefinition(ColumnDefinition pColumnDefinition) throws ModelException;

	/**
	 * Returns the <code>ColumnDefinition</code> by name.
	 * 
	 * @param pColumnName
	 *            name of the <code>ColumnDefinition</code>
	 * @return the <code>ColumnDefinition</code> by name.
	 * @throws ModelException 
	 * 			  if the <code>ColumnDefinition</code> name not exists.
	 */
	public ColumnDefinition getColumnDefinition(String pColumnName) throws ModelException;

	/**
	 * Returns the <code>ColumnDefinition</code> by index.
	 * 
	 * @param pColumnIndex
	 *            index of the <code>ColumnDefinition</code>
	 * @return the <code>ColumnDefinition</code> by index.
	 * @throws IndexOutOfBoundsException 
	 *            if the index is out of bounds
	 */
	public ColumnDefinition getColumnDefinition(int pColumnIndex);
	
	/**
	 * Returns the index of the <code>ColumnDefinition</code> name of this 
	 * <code>IRowDefinition</code>.
	 * 
	 * @param pColumnName
	 *            name of the <code>ColumnDefinition</code>
	 * @return the index of the <code>ColumnDefinition</code> name of this 
	 *         <code>IRowDefinition</code>, or -1 if it does not exist.
	 */
	public int getColumnDefinitionIndex(String pColumnName);

	/**
	 * Returns the indexes of the <code>ColumnDefinition</code> names of this 
	 * <code>IRowDefinition</code>.
	 * 
	 * @param pColumnNames
	 *            names of the <code>ColumnDefinition</code>
	 * @return the indexes of the <code>ColumnDefinition</code> names of this 
	 *         <code>IRowDefinition</code>, or -1 if it does not exist.
	 */
	public int[] getColumnDefinitionIndexes(String[] pColumnNames);

	/**
	 * Returns the column count of this <code>IRowDefintion</code>.
	 * 
	 * @return the column count of this <code>IRowDefintion</code>.
	 */
	public int getColumnCount();

	/**
	 * Returns the String[] of all column names in this <code>IRowDefintion</code>.
	 * 
	 * @return the String[] of all column names in this <code>IRowDefintion</code>.
	 */
	public String[] getColumnNames();
	
	/**
	 * Returns the String[] of all column names in the primary key.
	 * 
	 * @return the String[] of all column names in the primary key.
	 */
	public String[] getPrimaryKeyColumnNames();

	/**
	 * Sets all primary key column names.<br>
	 * It isn't allowed to change the primary key after open() the corresponding 
	 * <code>DataBook</code> or if the <code>DataRow</code> is in use.
	 * 
	 * @param pColumnNames
	 *            the new String[] with the column names to set the primary key
	 * @throws ModelException 
	 *            if an <code>IDataBook</code> already registered (at open()) to this 
	 *            <code>IRowdDefinition</code>.
	 */
	public void setPrimaryKeyColumnNames(String[] pColumnNames) throws ModelException;

	/**	
	 * Gets all classes, for which column views are set.
	 * @return all classes, for which column views are set.
	 */
	public Class<? extends IControl>[] getColumnViewClasses();

	/**
	 * Gets the <code>ColumnView</code> to use for GUI controls.<br>
	 * The target control class has to be instance of <code>IControl</code>
	 * (eg. <code>ITableControl</code>, <code>ITreeControl</code>, <code>IEditorControl</code>)
	 * or null for default.<br>
	 * The <code>ColumnView</code> is stored to the base interface or class that is instance of IControl.
	 * So it does not matter if you use 
	 * <code>getColumnView(ITableControl)</code> or <code>getColumnView(ITable)</code>, 
	 * <code>getColumnView(UITable)</code> because they are all base on <code>ITableControl</code>.<br>
	 * The default is used as long as no ColumnView is set for a specific type of IControl.<br>
	 * 
	 * eg:<br>
	 * if you set:
	 * <code>rowDef.setColumnView(null, new ColumnView("NAME"))</code>
	 * then
	 * <code>rowDef.getColumnView(null)</code> and 
	 * <code>rowDef.getColumnView(ITableControl)</code> and 
	 * <code>rowDef.getColumnView(ITreeControl)</code>...
	 * are returning the same instance set before.<br>
	 * if you set:
	 * <code>rowDef.setColumnView(UITree, new ColumnView("NAME"))</code>
	 * then
	 * <code>rowDef.getColumnView(null)</code> and 
	 * <code>rowDef.getColumnView(ITableControl)</code>
	 * are returning the default.<br>
	 * <code>rowDef.getColumnView(ITreeControl)</code> and 
	 * <code>rowDef.getColumnView(UITree)</code>
	 * are returning the same instance set with UITree.<br>
	 *  
	 * @param pTargetControl the type of target class.
	 * @return the column view to use for GUI controls.
	 */
	public ColumnView getColumnView(Class<? extends IControl> pTargetControl);
	
	/**
	 * Sets the column view to use for GUI controls.
	 * The target control class has to be instance of <code>IControl</code>
	 * (eg. <code>ITableControl</code>, <code>ITreeControl</code>, <code>IEditorControl</code>)
	 * or null for default.<br>
	 * The <code>ColumnView</code> is stored to the base interface or class that is instance of IControl.
	 * So it does not matter if you use 
	 * <code>getColumnView(ITableControl)</code> or <code>getColumnView(ITable)</code>, 
	 * <code>getColumnView(UITable)</code> because they are all base on <code>ITableControl</code>.<br>
	 * The default is used as long as no ColumnView is set for a specific type of IControl.<br>
	 * 
	 * eg:<br>
	 * if you set:
	 * <code>rowDef.setColumnView(null, new ColumnView("NAME"))</code>
	 * then
	 * <code>rowDef.getColumnView(null)</code> and 
	 * <code>rowDef.getColumnView(ITableControl)</code> and 
	 * <code>rowDef.getColumnView(ITreeControl)</code>...
	 * are returning the same instance set before.<br>
	 * if you set:
	 * <code>rowDef.setColumnView(UITree, new ColumnView("NAME"))</code>
	 * then
	 * <code>rowDef.getColumnView(null)</code> and 
	 * <code>rowDef.getColumnView(ITableControl)</code>
	 * are returning the default.<br>
	 * <code>rowDef.getColumnView(ITreeControl)</code> and 
	 * <code>rowDef.getColumnView(UITree)</code>
	 * are returning the same instance set with UITree.<br>
	 * 
	 * @param pTargetControl the type of target class.
	 * @param pColumnView the column view to use for GUI controls.
	 * @throws ModelException	if the xxxColumnsName (eg.xxx=Table) are not existing in the IRowDefinition.  
	 */
	public void setColumnView(Class<? extends IControl> pTargetControl, ColumnView pColumnView) throws ModelException;
	
	/**
	 * Sets the specified column names as read only true. 
	 * 
	 * @param pColumnNames		the column names to use. if null or empty, then all columns are used.
	 * @throws ModelException	if one column name doesn't exist.
	 */
	public void setReadOnly(String[] pColumnNames) throws ModelException;

	/**
	 * Returns all read only (true) column names. 
	 * 
	 * @return all read only (true) column names. 
	 */
	public String[] getReadOnly();
	
	/**
	 * Adds a <code>IControl</code> to the <code>IRowDefinition</code>.<br>
	 * The registered IControl methods will be called if <code>IRowDefinition</code> has changed.
	 * 
	 * @param pControl the <code>IControl</code>
	 * @see javax.rad.model.ui.IControl
	 */
	public void addControl(IControl pControl);

	/**
	 * Removes the <code>IControl</code> from the <code>IRowDefinition</code>.
	 * 
	 * @param pControl the <code>IControl</code>
	 * @see javax.rad.model.ui.IControl
	 */
	public void removeControl(IControl pControl);

	/**
	 * Returns all <code>IControl</code>'s from the <code>IRowDefinition</code>.
	 * 
	 * @return all <code>IControl</code>'s from the <code>IRowDefinition</code>.
	 */
	public IControl[] getControls();
	
} 	// IRowDefinition
