/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 26.10.2009 - [HM] - creation
 */
package com.sibvisions.util.type;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.NumberFormat;
import java.text.ParseException;
import java.text.ParsePosition;
import java.util.Locale;
import java.util.WeakHashMap;

/**
 * The <code>NumberUtil</code> is a utility class for number conversion and for formatting numbers
 * as string.
 *  
 * @author Martin Handsteiner
 */
public class NumberUtil
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** The default NUMBER_FORMAT. */
	private static final String DEFAULT_NUMBER_FORMAT = "#.######################################"; // (DecimalFormat)NumberFormat.getNumberInstance();
	
	/** Cached decimal format symbols. */
	private static WeakHashMap<Locale, DecimalFormatSymbols> decimalFormatSymbolsCache = new WeakHashMap<Locale, DecimalFormatSymbols>();
	
	/** The locale base for creation. */
	private Locale creationLocale = null;
	/** The pattern base for creation. */
	private String creationPattern = null;
	
	/** The number format. */
	private DecimalFormat numberFormat;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** 
	 * Constructs a new instance of <code>NumberUtil</code> with default number format.
	 */
	public NumberUtil()
	{
		setNumberFormat(null);
	}

	/** 
	 * Constructs a new instance of <code>NumberUtil</code> that supports empty Strings and null values.
	 * 
	 * @param pNumberFormat the formatter that should support empty Strings and null values
	 */
	public NumberUtil(NumberFormat pNumberFormat)
	{
		setNumberFormat(pNumberFormat);
	}

	/** 
	 * Constructs a new instance of <code>NumberUtil</code> that supports empty Strings and null values.
	 * 
	 * @param pNumberPattern the pattern that should support empty Strings and null values
	 */
	public NumberUtil(String pNumberPattern)
	{
		setNumberPattern(pNumberPattern);
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Parses the number from text.
	 * 
	 * @param pText the text.
	 * @return the parsed number.
     * @throws ParseException if there is an error in the conversion
	 */
	public Number parse(String pText) throws ParseException
	{
		if (pText == null || pText.length() == 0 || "-".equals(pText))
		{
			return null;
		}
		else
		{   
			if (creationLocale != null)
			{
				setNumberFormatIntern(LocaleUtil.getDefault(), creationPattern);
			}
			
			numberFormat.setParseBigDecimal(true); // Ensure creation of BigDecimal
			
			ParsePosition pos = new ParsePosition(0);
			Number value = numberFormat.parse(pText, pos);
			if (pos.getIndex() < pText.length())
			{
				try
				{
					value = new BigDecimal(pText.replace(numberFormat.getDecimalFormatSymbols().getDecimalSeparator(), '.'));
				}
				catch (Exception ex)
				{
					throw new ParseException(pText, pos.getIndex());
				}
			}
			return value;
		}
	}

	/**
	 * Formats the number to text.
	 * 
	 * @param pNumber the number.
	 * @return the formatted text.
	 */
	public String format(Number pNumber)
	{
		if (pNumber == null)
		{   
			return null;
		}
		else
		{
			if (creationLocale != null)
			{
				setNumberFormatIntern(LocaleUtil.getDefault(), creationPattern);
			}
			
			return numberFormat.format(pNumber);
		}
	}

	/**
	 * Gets the number format.
	 * 
	 * @return the number format.
	 */
	public NumberFormat getNumberFormat()
	{
		return numberFormat;
	}

	/**
	 * Gets the decimal format symbols for te given locale.
	 * @param pLocale the locale
	 * @return the decimal format symbols
	 */
	private DecimalFormatSymbols getDecimalFormatSymbols(Locale pLocale)
	{
		DecimalFormatSymbols decimalFormatSymbols = decimalFormatSymbolsCache.get(pLocale);
		if (decimalFormatSymbols == null)
		{
			decimalFormatSymbols = new DecimalFormatSymbols(pLocale);
			decimalFormatSymbolsCache.put(pLocale, decimalFormatSymbols);
		}
		return decimalFormatSymbols;
	}
	
	/**
	 * Sets the new number format, if something was changed.
	 * @param pLocale the locale
	 * @param pNumberPattern the pattern
	 */
	private void setNumberFormatIntern(Locale pLocale, String pNumberPattern)
	{
		if (pLocale != creationLocale || !pNumberPattern.equals(creationPattern))
		{
			numberFormat = new DecimalFormat(pNumberPattern, getDecimalFormatSymbols(pLocale));
			
			creationLocale = pLocale;
			creationPattern = pNumberPattern;
		}
	}
	
	/**
	 * Gets the number format.
	 * 
	 * @param pNumberFormat the number format.
	 */
	public void setNumberFormat(NumberFormat pNumberFormat)
	{
		if (pNumberFormat == null)
		{
			setNumberFormatIntern(LocaleUtil.getDefault(), DEFAULT_NUMBER_FORMAT);
		}
		else if (pNumberFormat instanceof DecimalFormat)
		{
			numberFormat = (DecimalFormat)pNumberFormat;
			
			creationLocale = null;
			creationPattern = null;
		}
		else
		{
			throw new IllegalArgumentException("Only DecimalFormat is supported!");
		}
	}

	/**
	 * Gets the number format pattern.
	 * 
	 * @return the number format pattern.
	 */
	public String getNumberPattern()
	{
		return numberFormat.toPattern();
	}

	/**
	 * Gets the number format pattern.
	 * 
	 * @param pNumberPattern the number format pattern.
	 */
	public void setNumberPattern(String pNumberPattern)
	{
		if (pNumberPattern == null)
		{
			setNumberFormat(null);
		}
		else
		{
			setNumberFormatIntern(LocaleUtil.getDefault(), pNumberPattern);
		}
	}

	/**
	 * Formats a number.
	 * 
	 * @param pNumber the number
	 * @param pNumberPattern the format 
	 * @return the formatted number string
	 * @see DecimalFormat
	 */
	public static String format(Number pNumber, String pNumberPattern)
	{
		DecimalFormat df = new DecimalFormat(pNumberPattern);
		
		return df.format(pNumber);
	}
	
}	// NumberUtil
