/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 01.10.2008 - [RH] - creation
 * 10.04.2009 - [RH] - interface review - size to getRowCount renamed
 *                                        getDataRow uses an ChangeableDataRow
 *                                        getMasterRow(), getChangedDataRows() added
 *                                        change state management is moved to ChangeableDataRow and MemDataBook     
 */
package javax.rad.model;

import javax.rad.model.condition.ICondition;

/**
 * A <code>IDataPage</code> is the interface for one data page of a <code>IDataBook</code>.<br>
 * A master <code>IDataBook</code> has one <code>IDataPage</code> for itself. If the 
 * <code>IDataBook</code> is (also) a detail <code>IDataBook</code> 
 * it stores all <code>IDataPage</code>'s for each loaded master row (parent master). <br>
 * 
 * @see javax.rad.model.IDataBook
 * @see javax.rad.model.IChangeableDataRow
 * 
 * @author Roland Hrmann
 */
public interface IDataPage
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Method definitions
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Returns the <code>IDataBook</code> of the <code>IDataPage</code>.
	 * 
	 * @return the <code>IDataBook</code> of the <code>IDataPage</code>.
	 */
	public IDataBook getDataBook();

	/**
	 * Returns the corresponding master row to this IDataPage in the master DataBook.
	 * 
	 * @return the corresponding master row to this IDataPage in the master DataBook.
	 * @throws ModelException 
	 * 			  if a exception occurs during synchronize.
	 */
	public IDataRow getMasterDataRow() throws ModelException;
	
	/**
	 * Returns a copy of the <code>IDataRow</code> of the specified index. <br>
	 * If the requested row not in memory it will be fetched from the storage.
	 * If the row is also not existing on the storage, it returns null.
	 * 
	 * @param pDataRowIndex
	 *            the index of the requested row
	 * @return a copy of the <code>IDataRow</code> of the specified index.
	 * @throws ModelException 
	 * 			  if a exception occur during the fetch.
	 */
	public IChangeableDataRow getDataRow(int pDataRowIndex) throws ModelException;
	
	/**
	 * Returns the size.
	 * 
	 * @return the size.
	 * @throws ModelException 
	 * 			  if a exception occurs during synchronize.
	 */
	public int getRowCount() throws ModelException;
	
	/**
	 * It fetches all <code>IDataRow</code>'s from the storage.
	 * 
	 * @throws ModelException
	 *             if an <code>DataSourceException</code> happens during get all rows
	 */
	public void fetchAll() throws ModelException;

	/**
	 * Returns true if all rows are loaded.
	 * 
	 * @return true if all rows are loaded.
	 * @throws ModelException 
	 * 				if the <code>DataBook</code> isn't open
	 */
	public boolean isAllFetched() throws ModelException;

    /**
     * Returns an int array with all changed rows. 
     * 
     * @return an int array with all changed rows.
	 * @throws ModelException 
	 * 			  if a exception occurs during synchronize.
     */
    public int[] getChangedDataRows() throws ModelException;

	/**
	 * Searches the given condition. It returns -1 if the condition can not be found.
	 * 
	 * @param pCondition the condition.
	 * @return the row number or -1 if the condition can not be found.
	 * @throws ModelException if a ModelException occurs.
	 */
	public int searchNext(ICondition pCondition) throws ModelException;

	/**
	 * Searches the given condition starting at the given position. It returns -1 if the condition can not be found.
	 * 
	 * @param pCondition the condition.
	 * @param pRowNum the starting row.
	 * @return the row number or -1 if the condition can not be found.
	 * @throws ModelException if a ModelException occurs.
	 */
	public int searchNext(ICondition pCondition, int pRowNum) throws ModelException;
	
	/**
	 * Searches the given condition. It returns -1 if the condition can not be found.
	 * 
	 * @param pCondition the condition.
	 * @return the row number or -1 if the condition can not be found.
	 * @throws ModelException if a ModelException occurs.
	 */
	public int searchPrevious(ICondition pCondition) throws ModelException;

	/**
	 * Searches the given condition starting at the given position. It returns -1 if the condition can not be found.
	 * 
	 * @param pCondition the condition.
	 * @param pRowNum the starting row.
	 * @return the row number or -1 if the condition can not be found.
	 * @throws ModelException if a ModelException occurs.
	 */
	public int searchPrevious(ICondition pCondition, int pRowNum) throws ModelException;
	
} 	// IDataPage
