/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 22.04.2008 - [HM] - creation
 * 17.07.2009 - [JR] - getLength, getInputStream, getGZIPContent: used fileName property as option
 *                   - getLength: (zippedBytes[zippedBytes.length - 2] & 0xff) << 24) replaced with 
 *                                (zippedBytes[zippedBytes.length - 1] & 0xff) << 24) [BUGFIX]
 * 05.10.2010 - [JR] - #169: remove directories from filenames
 */
package javax.rad.io;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import com.sibvisions.util.type.FileUtil;

/**
 * FileHandle handles data in memory or in the file system.
 * The Bean standard guarantees with the properties "fileName" and "gZIPContent" that it
 * can be transported with a connection. 
 * 
 * For compatibility reasons is the FileHandle also Serializable.
 * 
 * @author Martin Handsteiner
 */
public class FileHandle implements IFileHandle, 
                                   Serializable
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** The content. */ 
	private String fileName;
	
	/** The content in Memory. */ 
	private transient ByteArrayOutputStream gZIPContent;
	
	/** The content in File. */ 
	private transient File file;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Constructs a new FileHandle.
	 */
	public FileHandle()
	{
		fileName = null;
		gZIPContent = null;
		file = null;
	}
	
	/**
	 * Constructs a new FileHandle.
	 * @param pAbsolutePath the absolute path of the file.
	 */
	public FileHandle(String pAbsolutePath)
	{
		fileName = FileUtil.getName(pAbsolutePath);
		
		gZIPContent = null;
		file = new File(pAbsolutePath);
	}
	
	/**
	 * Constructs a new memory file handle.
	 * @param pFileName the file name.
	 * @param pContent the content.
	 * @throws IOException if an IOException occurs.
	 */
	public FileHandle(String pFileName, byte[] pContent) throws IOException
	{
		fileName = FileUtil.getName(pFileName);
		setContent(pContent);
	}
	
	/**
	 * Constructs a new memory file handle.
	 * @param pFileName the file name.
	 * @param pContent the content.
	 * @throws IOException if an IOException occurs.
	 */
	public FileHandle(String pFileName, InputStream pContent) throws IOException
	{
		fileName = FileUtil.getName(pFileName);
		setContent(pContent);
	}
	
	/**
	 * Constructs a new FileHandle.
	 * @param pContent the content.
	 * @throws IOException if an IOException occurs.
	 */
	public FileHandle(File pContent) throws IOException
	{
		fileName = null;
		setContent(pContent);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	public String getFileName()
	{
		return fileName;
	}

	/**
	 * {@inheritDoc}
	 */
	public InputStream getInputStream() throws IOException
	{
		if (gZIPContent != null)
		{
			return new GZIPInputStream(new ByteArrayInputStream(gZIPContent.toByteArray()));
		}
		else if (file != null)
		{
			return new FileInputStream(file);
		}
		else if (fileName != null)
		{
			return new FileInputStream(fileName);
		}
		else
		{
			return null;
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	public long getLength() throws IOException
	{
		if (gZIPContent != null)
		{
			byte[] zippedBytes = gZIPContent.toByteArray();
			return ((zippedBytes[zippedBytes.length - 1] & 0xff) << 24) 
		       | ((zippedBytes[zippedBytes.length - 2] & 0xff) << 16) 
		       | ((zippedBytes[zippedBytes.length - 3] & 0xff) << 8) 
		       | (zippedBytes[zippedBytes.length - 4] & 0xff);
		}
		else if (file != null)
		{
			return file.length();
		}
		else if (fileName != null)
		{
			return new File(fileName).length();
		}
		else
		{
			return -1;
		}
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Sets the file name of this file handle.
	 * @param pFileName the file name of this file handle.
	 */
	public void setFileName(String pFileName)
	{
		fileName = FileUtil.getName(pFileName);
	}

	/**
	 * Gets the file that was set as content.
	 * @return the file that was set as content.
	 */
	public File getFile()
	{
		return file;
	}

	/**
	 * Gets the gzipped content. 
	 * @return the gzipped content. 
	 * @throws IOException if an IOException occurs.
	 */
	public byte[] getGZIPContent() throws IOException
	{
		if (gZIPContent != null)
		{
			return gZIPContent.toByteArray();
		}
		else if (file != null || fileName != null)
		{
			ByteArrayOutputStream result = new ByteArrayOutputStream();
			
			FileInputStream fis;
			
			if (file != null)
			{
				fis = new FileInputStream(file);
			}
			else
			{
				fis = new FileInputStream(fileName);
			}
			
			try
			{
				FileUtil.copy(fis, true, new GZIPOutputStream(result), true);
			}
			finally
			{
				try
				{
					fis.close();
				}
				catch (IOException ioe)
				{
					//nothing to be done
				}
			}
			return result.toByteArray();
		}
		else
		{
			return null;
		}
	}
	
	/**
	 * Sets the gzipped content.
	 * @param pGZIPContent the gzipped content.
	 * @throws IOException if an IOException occurs.
	 */
	public void setGZIPContent(byte[] pGZIPContent) throws IOException
	{
		if (pGZIPContent == null)
		{
			file = null;
			gZIPContent = null;
		}
		else
		{
			file = null;
			
			gZIPContent = new ByteArrayOutputStream(pGZIPContent.length);
			gZIPContent.write(pGZIPContent);
			gZIPContent.close();
		}
	}

	/**
	 * Resets the content to zero, and gets an OutputStresm to write the content. 
	 * @return an OutputStresm to write the content.
	 * @throws IOException if an IOException occurs.
	 */
	public OutputStream getOutputStream() throws IOException
	{
		file = null;
		gZIPContent = new ByteArrayOutputStream();
		return new GZIPOutputStream(gZIPContent);
	}
	
	/**
	 * Sets a new content with a byte array.
	 * @param pContent the new content.
	 * @throws IOException if an IOException occurs.
	 */
	public void setContent(byte[] pContent) throws IOException
	{
		if (pContent == null)
		{
			gZIPContent = null;
			file = null;
		}
		else
		{
			OutputStream outStream = getOutputStream();
			outStream.write(pContent);
			outStream.close();
		}
	}
	
	/**
	 * Sets a new content with a InputStream.
	 * @param pContent the new content.
	 * @throws IOException if an IOException occurs.
	 */
	public void setContent(InputStream pContent) throws IOException
	{
		if (pContent == null)
		{
			gZIPContent = null;
			file = null;
		}
		else
		{
			FileUtil.copy(pContent, true, getOutputStream(), true);
		}
	}
	
	/**
	 * Sets a new content with a File.
	 * @param pContent the new content.
	 * @throws IOException if an IOException occurs.
	 */
	public void setContent(File pContent) throws IOException
	{
		gZIPContent = null;
		if (pContent != null && fileName == null)
		{
			fileName = pContent.getName();
		}
		file = pContent;
	}

	/**
	 * Writes this Object to the ObjectOutputStream.
	 * @param pObjectOutputStream the ObjectOutputStream.
	 * @throws IOException if an IOException occurs.
	 */
	private void writeObject(ObjectOutputStream pObjectOutputStream) throws IOException
	{
		pObjectOutputStream.defaultWriteObject();
		byte[] data = getGZIPContent();
		if (data == null)
		{
			pObjectOutputStream.writeInt(-1);
		}
		else
		{
			pObjectOutputStream.writeInt(data.length);
			pObjectOutputStream.write(data);
		}
	}
	
	/**
	 * Reads this Object to the pObjectInputStream.
	 * @param pObjectInputStream the pObjectInputStream.
	 * @throws IOException if an IOException occurs.
	 * @throws ClassNotFoundException if a class does not exist.
	 */
	private void readObject(ObjectInputStream pObjectInputStream) throws IOException, ClassNotFoundException
	{
		pObjectInputStream.defaultReadObject();
		file = null;
		
		int len = pObjectInputStream.readInt();
		if (len < 0)
		{
			gZIPContent = null;
		}
		else
		{
			byte[] data = new byte[len];
			pObjectInputStream.readFully(data);
			setGZIPContent(data);
		}
	}
	
}	// FileHandle
