/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 01.10.2008 - [JR] - creation
 * 04.08.2009 - [JR] - initSsl: short exception log
 * 10.08.2009 - [JR] - used logger
 * 20.01.2010 - [JR] - Properties constructor forwarded to parent constructor
 * 15.10.2010 - [JR] - #185: SSL handling now gets the servlet URL
 */
package com.sibvisions.rad.remote.http;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.Provider;
import java.security.SecureRandom;
import java.security.Security;
import java.util.Hashtable;
import java.util.Map;
import java.util.Properties;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.rad.remote.ConnectionInfo;
import javax.rad.remote.IConnectionConstants;

import com.sibvisions.rad.remote.AbstractSerializedConnection;
import com.sibvisions.rad.remote.ISerializer;
import com.sibvisions.util.log.ILogger;
import com.sibvisions.util.log.LoggerFactory;

/**
 * The <code>HttpConnection</code> communicates with the remote server via http
 * protocol.
 * 
 * @author Ren Jahn
 */
public class HttpConnection extends AbstractSerializedConnection
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the prefix for request properties. */
	public static final String PREFIX_HTTP = IConnectionConstants.PREFIX_CLIENT + "http.";
	
	/** the property name for the servlet url. */
	public static final String PROP_SERVICE = "service";
	
	
	/** the SSL provider. */
	private static Provider provider = null;
	
	/** the TrustManager for SSL connections. */
    private static TrustManager[] tmSsl = null;
    
    /** the HostnameVerifier for SSL connections. */
    private static HostnameVerifier hvSsl = null;

    /** URL to connect to the remote server. */
	private URL urlServlet = null;

	/** URL of the remote servlet. */
	private String sServletURL = null;
	
	/** Current connection to the server. */
	private URLConnection ucServer = null;
	
	/** the logger. */
	private ILogger log = LoggerFactory.getInstance(getClass());

	/** the connection timeout. */
	private int iConTimeout = -1;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Creates a new instance of <code>HttpConnection</code> with 
	 * properties instead of many parameters. The supported property
	 * keys are:
	 * <ul>
	 *   <li>HttpConnection.PROP_SERIALIZER</li>
	 *   <li>HttpConnection.PROP_SERVICE</li>
	 * </ul>
	 * 
	 * @param pProperties the properties for the connection
	 * @throws MalformedURLException if the servlet URL is not valid
	 * @throws NoSuchAlgorithmException if the specified protocol is not 
	 *                                  available in the default provider package 
	 *                                  or any of the other provider packages that 
	 *                                  were searched.
	 * @throws ClassNotFoundException if the serializer is defined and could not be created                                  
	 * @throws KeyManagementException if ssl operation fails
	 */
	public HttpConnection(Properties pProperties) throws MalformedURLException,
	 											  		 NoSuchAlgorithmException,
	 											  		 KeyManagementException,
	 											  		 ClassNotFoundException
	{
		this(createSerializer(pProperties.getProperty(PROP_SERIALIZER)), pProperties.getProperty(PROP_SERVICE)); 
	}
	
	/**
	 * Creates a new instance of <code>HttpConnection</code> with the default
	 * serializer.
	 * 
	 * @param pServletURL URL to the remote server
	 * @throws MalformedURLException if the servlet URL is not valid
	 * @throws NoSuchAlgorithmException if the specified protocol is not 
	 *                                  available in the default provider package 
	 *                                  or any of the other provider packages that 
	 *                                  were searched.
	 * @throws KeyManagementException if ssl operation fails
	 * @see AbstractSerializedConnection#AbstractSerializedConnection(ISerializer)
	 */
	public HttpConnection(String pServletURL) throws MalformedURLException,
													 NoSuchAlgorithmException,
													 KeyManagementException
	{
		this(null, pServletURL);
	}
	
	/**
	 * Creates a new instance of <code>HttpConnection</code>.
	 * 
	 * @param pSerializer the serializer for the communication between client and server
	 * @param pServletURL URL to the remote server
	 * @throws MalformedURLException if the servlet URL is not valid
	 * @throws NoSuchAlgorithmException if the specified protocol is not 
	 *                                  available in the default provider package 
	 *                                  or any of the other provider packages that 
	 *                                  were searched.
	 * @throws KeyManagementException if ssl operation fails
	 * @see AbstractSerializedConnection#AbstractSerializedConnection(ISerializer)
	 */
	public HttpConnection(ISerializer pSerializer, String pServletURL) throws MalformedURLException,
																			  NoSuchAlgorithmException,
																			  KeyManagementException
	{
		super(pSerializer);
		
		this.sServletURL = pServletURL;
	
		urlServlet = new URL(pServletURL);
		
		log.info("Server: ", pServletURL, ", Serializer: ", pSerializer != null ? pSerializer.getClass() : null);
		
		initSsl();
	}
	
	/**
	 * Initializes the ssl context if necessary.
	 * 
	 * @throws KeyManagementException if ssl operation fails
	 * @throws NoSuchAlgorithmException if the specified protocol is not 
	 *                                  available in the default provider package 
	 *                                  or any of the other provider packages that 
	 *                                  were searched.
	 */
	private void initSsl() throws NoSuchAlgorithmException,
	  					   	      KeyManagementException
	{
		if ("https".equals(urlServlet.getProtocol().toLowerCase()))
		{
			if (provider == null)
			{
				try
				{
			        try
			        {
			        	provider = (Provider)Class.forName("com.sun.net.ssl.internal.ssl.Provider").newInstance();
			        }
			        catch (Exception e)
			        {
			        	log.error(e);
			        	
			        	//maybe android
			        	try
			        	{
			        		provider = (Provider)Class.forName("org.apache.harmony.xnet.provider.jsse.JSSEProvider").newInstance();
			        	}
			        	catch (Exception ex)
			        	{
			        		log.error(ex);
			        	}
			        }
			        
			        if (provider != null)
			        {
				        //Trust manager for certicifate validation
			    		tmSsl = new TrustManager[] { new HttpsTrustManager() };
				        hvSsl = new HttpsHostnameVerifier(); 
				
				    	Security.addProvider(provider);
				
				        //Install Trust Manager
				    	SSLContext sc = SSLContext.getInstance("SSL");
				        
				        sc.init(null, tmSsl, new SecureRandom());
				        
				        HttpsURLConnection.setDefaultSSLSocketFactory(sc.getSocketFactory());
				    	HttpsURLConnection.setDefaultHostnameVerifier(hvSsl);
			        }
			        else
			        {
			        	log.error("SSL Security provider not found!");
			        }
				}
				catch (SecurityException se)
				{
					//not allowed in applets
					log.error(se);
					
					tmSsl = null;
					hvSsl = null;
					provider = null;
				}
			}
			
			//add the server URL as trusted
			if (tmSsl != null && hvSsl != null)
			{
				((HttpsTrustManager)tmSsl[0]).addUrl(urlServlet);
				((HttpsHostnameVerifier)hvSsl).addUrl(urlServlet);
			}
		}
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Abstract methods implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	@Override
	public OutputStream getOutputStream(ConnectionInfo pConnectionInfo) throws IOException
	{
		ucServer = urlServlet.openConnection();
		
		ucServer.setUseCaches(false);
		ucServer.setRequestProperty("Content-Type", "application/octet-stream");
		
		if (iConTimeout >= 0)
		{
			ucServer.setConnectTimeout(iConTimeout);
			ucServer.setReadTimeout(iConTimeout);
		}
		else
		{
		    //defaults
            ucServer.setConnectTimeout(15000);
            ucServer.setReadTimeout(15000);
		}
		
	    ucServer.setDoInput(true);
	    ucServer.setDoOutput(true);
	    
	    //put http properties to the request
	    Hashtable<String, Object> htProps = pConnectionInfo.getProperties();
	    Object oValue;
	    
	    String sKey;
	    
	    for (Map.Entry<String, Object> entry : htProps.entrySet())
	    {
	    	sKey = entry.getKey();
	    	
	    	if (sKey.startsWith(PREFIX_HTTP))
	    	{
	    		oValue = entry.getValue();
	    		
	    		if (oValue instanceof String)
	    		{
	    			ucServer.setRequestProperty(sKey.substring(PREFIX_HTTP.length()), (String)oValue);
	    		}
	    	}
	    }

	    return ucServer.getOutputStream();
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public InputStream getInputStream(ConnectionInfo pConnectionInfo) throws IOException
	{
		if (ucServer != null)
		{
			return ucServer.getInputStream();
		}
		else
		{
			throw new IOException("The connection is not open!");
		}
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Gets the servlet URL to access the remote server.
	 * 
	 * @return URL to the remote server
	 */
	public String getServletURL()
	{
		return sServletURL;
	}

	/**
	 * Sets the timeout for the connection establishment.
	 * 
	 * @param pTimeout the timeout in millis or -1 for the default timeout
	 */
	public void setConnectionTimeout(int pTimeout)
	{
		iConTimeout = pTimeout;
	}
	
}	// HttpConnection
