/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 * 
 * 04.11.2014 - [TL] - creation.
 * 11.11.2014 - [TL] - Primary Keys are now get from the MetaData.
 */
package com.sibvisions.rad.persist.jdbc;

import java.math.BigDecimal;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;

import javax.rad.persist.DataSourceException;

import com.sibvisions.rad.persist.jdbc.ServerMetaData.PrimaryKeyType;
import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.GroupHashtable;
import com.sibvisions.util.log.ILogger.LogLevel;
import com.sibvisions.util.type.CommonUtil;

/**
 * The <code>HanaDBAccess</code> is the implementation for SAP HANA databases.<br>
 * 
 * @see com.sibvisions.rad.persist.jdbc.DBAccess
 * @author Thomas Lehner
 */
public class HanaDBAccess extends DBAccess
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Constants
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** Cache of <code>Sequences</code>'s to improve performance. */
	private static GroupHashtable<String, String, String>	ghtSequenceCache	= new GroupHashtable<String, String, String>();

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Constructs a new HanaDBAccess Object.
	 */
	public HanaDBAccess()
	{
		super();

		setDriver("com.sap.db.jdbc.Driver");
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object[] insert(String pWriteBackTable, ServerMetaData pServerMetaData, Object[] pNewDataRow) throws DataSourceException
	{
		if (!isOpen())
		{
			throw new DataSourceException("DBAccess is not open!");
		}

		if (pWriteBackTable == null)
		{
			throw new DataSourceException("Missing WriteBackTable!");
		}

		String columnId = null;
		String sequenceName = null;
		BigDecimal nextval = null;

		StringBuilder sInsertStatement = new StringBuilder("INSERT INTO ");
		sInsertStatement.append(pWriteBackTable);
		sInsertStatement.append(" (");

		// add column names to insert
		int iColumnCount = 0;
		String sDummyColumn = null;

		String[] saAutoIncrement;

		PreparedStatement psColumnId = getPreparedStatement("select column_id from table_columns where table_name = ? and column_name = ?",
															false);
		PreparedStatement psSequenceName = getPreparedStatement("select sequence_name from sequences where sequence_name like ?", false);
		PreparedStatement psGetNextVal = null;
		
		ResultSet rsColumnId = null;
		ResultSet rsSequenceName = null;
		ResultSet rsGetNextVal = null;
		
		try
		{
			TableInfo tinfo = getTableInfoIntern(pWriteBackTable);

			saAutoIncrement = getAutoIncrementColumns(tinfo.getSchema(), pWriteBackTable);
			
			for (String columnName : saAutoIncrement)
			{
				int index = pServerMetaData.getServerColumnMetaDataIndex(columnName);

				if (pNewDataRow[index] == null)
				{
					String dbAccessIdentifier = getIdentifier();
					String tableIdentifier = createIdentifier(tinfo.getSchema(), tinfo.getTable());

					sequenceName = ghtSequenceCache.get(dbAccessIdentifier, tableIdentifier);

					if (sequenceName == null)
					{
						psColumnId.setString(1, removeQuotes(pWriteBackTable));
						psColumnId.setString(2, columnName);
						rsColumnId = psColumnId.executeQuery();
						
						try
						{
							if (rsColumnId.next())
							{
								columnId = rsColumnId.getString(1);
							}
						}
						finally
						{
							CommonUtil.close(rsColumnId);
						}

						psSequenceName.setString(1, "%" + columnId + "%");
						rsSequenceName = psSequenceName.executeQuery();
						
						try
						{
							if (rsSequenceName.next())
							{
								sequenceName = rsSequenceName.getString(1);
							}
						}
						finally
						{
							CommonUtil.close(rsSequenceName);
						}
						
						if (sequenceName != null)
						{
							ghtSequenceCache.put(dbAccessIdentifier, tableIdentifier, sequenceName);
						}
					}
					
					psGetNextVal = getPreparedStatement("select " + sequenceName + ".nextval from dummy", false);
					
					try
					{
						rsGetNextVal = psGetNextVal.executeQuery();
						if (rsGetNextVal.next())
						{
							nextval = rsGetNextVal.getBigDecimal(1);
						}
						pNewDataRow[index] = nextval;
					}
					finally
					{
						CommonUtil.close(rsGetNextVal, psGetNextVal);
					}
				}
			}
		}
		catch (Exception e)
		{
			debug("Can't find sequence for ", pWriteBackTable, e);
		}
		finally
		{
			CommonUtil.close(psColumnId, psSequenceName);
		}

		ServerColumnMetaData[] cmdServerColumnMetaData = pServerMetaData.getServerColumnMetaData();
		int[] iaWriteables = pServerMetaData.getWritableColumnIndices();

		for (int i = 0; i < iaWriteables.length; i++)
		{
			if (pNewDataRow[iaWriteables[i]] != null)
			{
				if (iColumnCount > 0)
				{
					sInsertStatement.append(", ");
				}
				sInsertStatement.append(cmdServerColumnMetaData[iaWriteables[i]].getColumnName().getQuotedName());
				iColumnCount++;
			}
		}

		if (iColumnCount == 0)
		{
			// if no storable columns, put in a dummy one
			for (int i = 0; iColumnCount == 0 && i < iaWriteables.length; i++)
			{
				if (!cmdServerColumnMetaData[iaWriteables[i]].isAutoIncrement())
				{
					sDummyColumn = cmdServerColumnMetaData[iaWriteables[i]].getColumnName().getQuotedName();
					sInsertStatement.append(sDummyColumn);
					iColumnCount++;
				}
			}
		}

		// Add values '?' to insert
		sInsertStatement.append(") VALUES (");

		if (iColumnCount > 0)
		{
			sInsertStatement.append("?");

			for (int i = 1; i < iColumnCount; i++)
			{
				sInsertStatement.append(",?");
			}
		}
		sInsertStatement.append(")");

		setModified(true);

		pNewDataRow = insertDatabaseSpecific(pWriteBackTable, sInsertStatement.toString(), pServerMetaData, pNewDataRow, sDummyColumn);

		if (isLogEnabled(LogLevel.DEBUG))
		{
			debug(sInsertStatement.toString(), "[", pNewDataRow, "]");
		}

		// NO PK, and all columns are null -> Exception -> but shouldn't happen
		// if we want import
		// empty records via API calls
		if (pServerMetaData.getPrimaryKeyType() != PrimaryKeyType.AllColumns)
		{
			// check Empty PK and throw Exception
			boolean bPKEmpty = true;
			int[] iPKColsIndices = pServerMetaData.getPrimaryKeyColumnIndices();

			if (iPKColsIndices != null)
			{
				for (int i = 0; i < iPKColsIndices.length && bPKEmpty; i++)
				{
					if (pNewDataRow[iPKColsIndices[i]] != null)
					{
						bPKEmpty = false;
					}
				}
			}

			if (bPKEmpty)
			{
				throw new DataSourceException("Primary key column empty after insert!" + pWriteBackTable);
			}
		}
		
		return pNewDataRow;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object[] insertDatabaseSpecific(String pWriteBackTable, String pInsertStatement, ServerMetaData pServerMetaData,
			                               Object[] pNewDataRow, String pDummyColumn) throws DataSourceException
	{
		return insertHana(pWriteBackTable, pInsertStatement, pServerMetaData, pNewDataRow, pDummyColumn);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected TableInfo getTableInfoIntern(String pWriteBackTable) throws DataSourceException
	{
		TableInfo tableInfo = super.getTableInfoIntern(pWriteBackTable);

		PreparedStatement psSchema = getPreparedStatement("select CURRENT_SCHEMA from dummy", false);
		ResultSet rsSchema = null;

		String schema;
		
		try
		{
			schema = tableInfo.getSchema();
			
			if (schema == null)
			{
				try
				{
					rsSchema = psSchema.executeQuery();
					if (rsSchema.next())
					{
						schema = rsSchema.getString(1);
					}
				}
				catch (SQLException se)
				{
					throw new DataSourceException("Can't select the current Schema!", formatSQLException(se));
				}
				finally
				{
					CommonUtil.close(rsSchema);
				}
	
			}
		}
		finally
		{
			CommonUtil.close(psSchema);
		}
		
		String table = tableInfo.getTable();
		if (table != null && !table.startsWith(QUOTE) && !table.endsWith(QUOTE))
		{
			table = table.toUpperCase();
		}

		return new TableInfo(tableInfo.getCatalog(), schema, table);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
    protected String getColumnName(ResultSetMetaData pMetaData, int pColumn) throws SQLException
    {
    	return pMetaData.getColumnLabel(pColumn);
    }
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Returns the newly inserted row from a SAP Hana Database. <br>
	 * It gets the next value for all autogenerated Columns and sets them
	 * manually.
	 * 
	 * @param pWriteBackTable the table to use for the insert
	 * @param pInsertStatement the SQL Statement to use for the insert
	 * @param pServerMetaData the meta data to use.
	 * @param pNewDataRow the new row (Object[]) with the values to insert
	 * @param pDummyColumn true, if all writeable columns are null, but for a
	 *            correct INSERT it have
	 *            to be minimum one column to use in the syntax.
	 * @return the newly inserted row from a Ansi SQL Database.
	 * @throws DataSourceException
	 *             if an <code>Exception</code> occur during insert to the
	 *             storage
	 */
	protected Object[] insertHana(String pWriteBackTable, String pInsertStatement, ServerMetaData pServerMetaData,
			                      Object[] pNewDataRow, String pDummyColumn) throws DataSourceException
	{
		PreparedStatement psInsert = getPreparedStatement(pInsertStatement, false);

		try
		{
			ServerColumnMetaData[] cmdServerColumnMetaData = pServerMetaData.getServerColumnMetaData();
			int[] iaWriteables = pServerMetaData.getWritableColumnIndices();

			if (pDummyColumn == null)
			{
				setColumnsToStore(psInsert, cmdServerColumnMetaData, iaWriteables, pNewDataRow, null);
			}
			else
			{
				// set null value for it!
				try
				{
					for (int i = 0; i < cmdServerColumnMetaData.length; i++)
					{
						if (cmdServerColumnMetaData[i].getColumnName().getQuotedName().equals(pDummyColumn))
						{
							psInsert.setObject(1, null, cmdServerColumnMetaData[i].getSQLType());
							break;
						}
					}
				}
				catch (SQLException sqlException)
				{
					throw new DataSourceException("Insert failed! - " + pInsertStatement, formatSQLException(sqlException));
				}
			}
			if (executeUpdate(psInsert) == 1)
			{
				return pNewDataRow;
			}
			
			throw new DataSourceException("Insert failed! - Result row count != 1" + pInsertStatement);
		}
		finally
		{
			CommonUtil.close(psInsert);
		}
	}
	
	/**
	 * Returns all auto_incremenent columns for this table.
	 * 
	 * @param pSchema the schema to use.
	 * @param pTableName the table to use.
	 * @return all auto_incremenent columns for this table.
	 * @throws Exception if the auto increment columns couldn't determined.
	 */
	private String[] getAutoIncrementColumns(String pSchema, String pTableName) throws Exception
	{
		ArrayUtil<String> auAutoIncrement = new ArrayUtil<String>();

		PreparedStatement psAutoIncColumns = null;
		ResultSet rsAutoIncColumns = null;

		try
		{
			String sTableNameNoQuote = DBAccess.removeQuotes(pTableName);

			psAutoIncColumns = getPreparedStatement("select column_name from table_columns where generation_type = 'BY DEFAULT AS IDENTITY' and table_name = ? and schema_name = ?",
					                                false);
			psAutoIncColumns.setString(1, sTableNameNoQuote);
			psAutoIncColumns.setString(2, pSchema);

			rsAutoIncColumns = psAutoIncColumns.executeQuery();

			while (rsAutoIncColumns.next())
			{
				String incColumn = rsAutoIncColumns.getString(1);

				if (incColumn != null)
				{
					auAutoIncrement.add(incColumn);
				}
			}
		}
		finally
		{
			CommonUtil.close(rsAutoIncColumns, psAutoIncColumns);
		}

		String[] saAutoIncrement = new String[auAutoIncrement.size()];
		return auAutoIncrement.toArray(saAutoIncrement);
	}	
	
}	// HanaDBAccess
