/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 01.10.2008 - [RH] - creation
 * 11.10.2008 - [RH] - bug fixes, toString() optimized
 * 12.10.2008 - [RH] - bug fixes, store(), insert(), restore()
 * 02.11.2008 - [RH] - bug fixes in SetValue 
 *                     - copy original values corrected
 *                     - convertAndCheckToTypeClass add in setValue for correct DataType conversion 
 *                     SelectedDataRow support added
 * 11.11.2008 - [RH] - MasterRow set in MemDataPage
 * 17.11.2008 - [RH] - optimized; ArrayUtil to int[] changed; unnecessary intern removed    
 * 19.11.2008 - [RH] - fetchToRowInternal added for PK check, that no row get requested from AbstractStorage
 * 10.04.2009 - [RH] - interface review - size to getRowCount renamed
 *                                        getDataRow uses an ChangeableDataRow
 *                                        getMasterRow(), getChangedDataRows() added
 *                                        change state management is moved to ChangeableDataRow and MemDataBook
 * 18.04.2009 - [RH] - javadoc reviewed, code optimized                                                     
 * 12.06.2009 - [JR] - toString: used StringBuilder [PERFORMANCE]
 * 16.10.2009 - [RH] - fetchAll in one request implemented -> fecthToRow(-1)=fetchAll
 *                     filter StackOverflow Error fixed [BUGFIXED]
 * 18.10.2009 - [RH] - setFilter(xx), with isMemFilter == true, on an RemoteDataBook, didn't fetched all rows, before it filters in mem. [BUGFIXED]
 * 29.09.2010 - [RH] - countRows renamed to getEstimatedRowCount(),  getEstimatedRowCount() returns in MemDataPage getRowCount() 
 * 08.04.2011 - [RH] - #330 - restoreAllRows fails in DataSource Level.
 * 20.12.2012 - [RH] - Code Review - Changes management in insert, update, delete, store, restore, setDetailChanged is moved 
 *                                   from MemDataBook to the MemDataPage. 
 * 10.04.2013 - [RH] - #617 - saveAllDataBooks doesn't save all rows in DATASOURCE level - fixed
 * 10.04.2013 - [RH] - #617 - restoreAllRows fails with ArrayIndexOutOfBoundsException - fixed
 * 10.04.2013 - [RH] - #618 - restoreAllRows throws an Exception - fixed
 * 12.04.2013 - [RH] - #514 - JVx DataBook, DataPage, DataRow toString should be better formatted - fixed
 */
package com.sibvisions.rad.model.mem;

import javax.rad.model.IChangeableDataRow;
import javax.rad.model.IDataBook;
import javax.rad.model.IDataPage;
import javax.rad.model.IDataRow;
import javax.rad.model.IRowDefinition;
import javax.rad.model.ModelException;
import javax.rad.model.SortDefinition;
import javax.rad.model.condition.ICondition;
import javax.rad.model.datatype.BigDecimalDataType;
import javax.rad.model.datatype.DataType;
import javax.rad.model.datatype.IDataType;
import javax.rad.model.datatype.TimestampDataType;

import com.sibvisions.rad.model.DataBookSorter;
import com.sibvisions.util.ArrayUtil;
import com.sibvisions.util.type.CommonUtil;
import com.sibvisions.util.type.StringUtil;

/**
 * A <code>MemDataPage</code> is the memory implementation for a data page of an 
 * <code>IDataBook</code>.<br>
 * A master <code>IDataBook</code> has one <code>IDataPage</code> for itself. If the 
 * <code>IDataBook</code> is (also) a detail <code>IDataBook</code> 
 * it stores all <code>IDataPage</code>'s for each loaded master row (parent master). <br>
 * 
 * @see javax.rad.model.IDataBook
 * @see javax.rad.model.IDataPage
 * @see javax.rad.model.IChangeableDataRow
 * @author Roland Hrmann
 */
public class MemDataPage implements IDataPage
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/** The initial storage size for the DataRow/Object array. */
	private static final int	INIT_STORAGE_SIZE = 16;

	/** The IDataBook that uses this MemDataPage. */
	private MemDataBook			dbDataBook;
	/** The master row from the corresponding master DataBook. */
	private IDataRow			drMasterDataRow; 		
	/** The storage array with all DataRows of the DataPage. */
	private ArrayUtil<Object[]>	alStorage;
	
	/** The array with all changes rows. */
	private int[]			 	iaChangedRows = new int[0];
	
	/** 
	 * The boolean indicates if all rows are fetched from the under laying storage.
	 * In this MemDataPage its this always true.
	 */
	private boolean 			bAllFetched = true;
		
	/** Array with all rows that shows the MemDataBook outside. -> mem sort, filter. */
	private int[]				iaUsedRows;
		
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Construct a new MemDataPage for the specified IDataBook and the corresponding master row. 
	 * 
	 * @param pDataBook 		the IDataBook which uses this MemDataPage 
	 * @param pMasterDataRow	the corresponding master row of the master IDataBook of the 
	 *                          above specified IDataBook 
	 */
	public MemDataPage(MemDataBook pDataBook, IDataRow pMasterDataRow)
	{
		dbDataBook = pDataBook;
		drMasterDataRow = pMasterDataRow; 
		alStorage  = new ArrayUtil<Object[]>(INIT_STORAGE_SIZE);
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * {@inheritDoc}
	 */
	public IDataBook getDataBook()
	{
		return dbDataBook;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public IDataRow getMasterDataRow()
	{
		return drMasterDataRow;
	}
	
	/**
	 * {@inheritDoc}  
	 */
	public IChangeableDataRow getDataRow(int pDataRowIndex) throws ModelException
	{
		initFilterSort();
		
		if (!bAllFetched && pDataRowIndex >= alStorage.size())
		{
			fetchToRow(pDataRowIndex);
		}

		if (pDataRowIndex >= 0 && pDataRowIndex < getRowCountInternal())
		{
			return getDataRowWithoutSort(getInternalRowIndex(pDataRowIndex));
		}
		else
		{
			return null;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void fetchAll() throws ModelException
	{
		if (!bAllFetched)				
		{
			fetchToRow(-1);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean isAllFetched() throws ModelException
	{
		return bAllFetched;
	}

	/**
	 * {@inheritDoc}
	 */
	public int getRowCount() throws ModelException
	{
		initFilterSort();

		return getRowCountInternal();
	}		
	
	/**
	 * {@inheritDoc}
	 */
	public int[] getChangedDataRows()
	{
		if (iaChangedRows == null || iaChangedRows.length == 0 || iaUsedRows == null)
		{
			return iaChangedRows;
		}
		 
		// Convert internal indices to the current Filter/Sort external indices.
		// -> Remove it in the result, if they aren't in the filter result.
		
		int[] iaResultChanges = new int[iaChangedRows.length];
		int index = 0;
		for (int i = 0; i < iaChangedRows.length; i++)
		{		
			int iFound = ArrayUtil.indexOf(iaUsedRows, iaChangedRows[i]);
			if (iFound >= 0)
			{
				iaResultChanges[index] = iFound; 
				index++;
			}
		}
		
		return ArrayUtil.truncate(iaResultChanges, index);
	}

	/**
	 * {@inheritDoc}
	 */
	public int searchNext(ICondition pCondition) throws ModelException
	{
		return searchNext(pCondition, 0);
	}

	/**
	 * {@inheritDoc}
	 */
	public int searchNext(ICondition pCondition, int pRowNum) throws ModelException
	{
		if (pCondition != null && pRowNum >= 0)
		{
			initFilterSort();
			
			if (!bAllFetched && pRowNum >= alStorage.size())
			{
				fetchToRow(pRowNum);
			}
			
			while (pRowNum < getRowCountInternal())
			{
				if (pCondition.isFulfilled(getDataRowWithoutSort(getInternalRowIndex(pRowNum))))
				{
					return pRowNum;
				}
				pRowNum++;
				if (!bAllFetched && pRowNum >= alStorage.size())
				{
					fetchToRow(pRowNum);
				}
			}
		}
		return -1;
	}

	/**
	 * {@inheritDoc}
	 */
	public int searchPrevious(ICondition pCondition) throws ModelException
	{
		return searchPrevious(pCondition, -1);
	}

	/**
	 * {@inheritDoc}
	 */
	public int searchPrevious(ICondition pCondition, int pRowNum) throws ModelException
	{
		if (pCondition != null && pRowNum != 0)
		{
			fetchAll();
			initFilterSort();
			
			int rowCount = getRowCountInternal();

			if (pRowNum < 0 || pRowNum > rowCount)
			{
				pRowNum = rowCount - 1;
			}
			else
			{
				pRowNum--;
			}
			
			while (pRowNum >= 0)
			{
				if (pCondition.isFulfilled(getDataRowWithoutSort(getInternalRowIndex(pRowNum))))
				{
					return pRowNum;
				}
				pRowNum--;
			}
		}
		return -1;
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString()
	{
		return toString("");
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	
	
	/**
	 * Sets that the MemDataPage has all fetched. Should only used from derived classes.
	 * 
	 * @param pAllFetched	the boolean with the fetch state.
	 */
	protected void setAllFetched(boolean pAllFetched)
	{
		bAllFetched = pAllFetched;
	}

	/**
	 * It adds an new IDataRow to the DataPage in the object[] storage.
	 * 
	 * @param pValues			the values.
	 * @throws ModelException	if the IDataRow couldn't add to the storage.
	 */
	protected void addFetchedRow(Object[] pValues) throws ModelException
	{
		alStorage.add(pValues);
	}
	
	/**
	 * It inserts a new <code>IChangeableDataRow</code> in the MemDataPage at the specified index. 
	 * 
	 * @param pDataRowIndex		the row index to use.
	 * @throws ModelException	if the Filter/Sort couldn't initialized.
	 */
	void insert(int pDataRowIndex) throws ModelException
	{
		if (pDataRowIndex > alStorage.size())
		{
			throw new ModelException("DataRow index out of bounds");
		}
		initFilterSort();

		// create new data row, set Inserting and UID and set storage of the base DataRow to this 
		ChangeableDataRow cdr = new ChangeableDataRow(dbDataBook.getRowDefinition(),
													  null,
				 									  this,
				 									  pDataRowIndex);
		cdr.setDefaultValues();
		cdr.setInserting();
		cdr.setUID(Integer.valueOf(MemDataBook.getNextUID()));
				
	 	alStorage.add(pDataRowIndex, cdr.getStorage());

		if (iaUsedRows != null)
		{
			shiftIndicies(iaUsedRows, pDataRowIndex, 1);
			iaUsedRows = ArrayUtil.add(iaUsedRows, pDataRowIndex, pDataRowIndex);
		}

		shiftIndicies(iaChangedRows, pDataRowIndex, 1);
		addChange(pDataRowIndex);
	}
	
	/**
	 * Marks the specified row as Deleted with the given index in the MemDataPage. 
	 * if the row is inserting(), then the row is really removed from the MemDataPage.
	 * 
	 * @param pDataRowIndex	the row index to use.
	 * @return true if the row is removed from the memory or false if the row is only marked as deleting.
	 * @throws ModelException	if the delete operation fails.
	 */
	boolean delete(int pDataRowIndex) throws ModelException
	{		
		int iInternalIndex = getInternalRowIndex(pDataRowIndex);
		if (iInternalIndex >= alStorage.size())
		{
			throw new ModelException("Row index out of bounds");
		}	
		
		ChangeableDataRow cdr = getDataRowWithoutSort(iInternalIndex);
		
		if (cdr.isInserting())
		{
			// remove current row
			deleteRow(iInternalIndex);
			return true;
		}
		else
		{
	        // Otherwise mark current DataRow as deleting
			cdr.setDeleting(); 
			setDataRow(iInternalIndex, cdr);
			return false;
		}		
	}

	
	/**
	 * Sets the specified row as Updating with the given index in the MemDataPage. 
	 * 
	 * @param pDataRowIndex		the row index to use.
	 * 
	 * @throws ModelException 	if the updating operation fails.
	 */
	void update(int pDataRowIndex) throws ModelException
	{
		int iInternalIndex = getInternalRowIndex(pDataRowIndex);
		if (iInternalIndex >= alStorage.size())
		{
			throw new ModelException("DataRow index out of bounds");
		}		
		ChangeableDataRow cdr = getDataRowWithoutSort(iInternalIndex);
		cdr.setUpdating(); 
		
		setDataRow(iInternalIndex, cdr);				
	}

	/**
	 * Restores the specified row in the MemDataPage. 
	 * 
	 * @param pDataRowIndex		the row index to use.
	 * 
	 * @throws ModelException 	if the updating operation fails.
	 */
	void restore(int pDataRowIndex) throws ModelException
	{
		int iInternalIndex = getInternalRowIndex(pDataRowIndex);
		if (iInternalIndex >= alStorage.size())
		{
			throw new ModelException("DataRow index out of bounds");
		}		
		ChangeableDataRow cdr = getDataRowWithoutSort(iInternalIndex);
		if (cdr.isInserting())
		{
			deleteRow(iInternalIndex);
			return;
		}
		cdr.restore(); 
		
		setDataRow(iInternalIndex, cdr);
	}
	
	/**
	 * Stores the specified row in the MemDataPage. 
	 * 
	 * @param pDataRowIndex		the row index to use.
	 * @return true if the row is removed from the memory.
	 * @throws ModelException 	if the updating operation fails.
	 */
	protected boolean store(int pDataRowIndex) throws ModelException
	{
		int iInternalIndex = getInternalRowIndex(pDataRowIndex);
		if (iInternalIndex >= alStorage.size())
		{
			throw new ModelException("DataRow index out of bounds");
		}
		ChangeableDataRow cdr = getDataRowWithoutSort(iInternalIndex);
	
		if (cdr.isDeleting())
		{
			deleteRow(iInternalIndex);
			return true;
		}	
		cdr.store(); 
			
		setDataRow(iInternalIndex, cdr);
		return false;
	}	
	
	/**
	 * Set that the details to this row are changed. 
	 * 
	 * @param pDataRowIndex		the row index to use.
	 * @param bDetailChanged	true if the details changed, otherwise false.
	 *  
	 * @throws ModelException 	if the set detail changed operation fails.
	 */
	protected void setDetailChanged(int pDataRowIndex, boolean bDetailChanged) throws ModelException
	{
		int iInternalIndex = getInternalRowIndex(pDataRowIndex);
		if (iInternalIndex >= alStorage.size())
		{
			throw new ModelException("DataRow index out of bounds");
		}
		ChangeableDataRow cdr = getDataRowWithoutSort(iInternalIndex);
		cdr.setDetailChanged(bDetailChanged); 
			
		setDataRow(iInternalIndex, cdr);
		
		// manage list of changed Detail rows
		if (bDetailChanged)
		{
			addChange(iInternalIndex);
		}
		else
		{
			removeChange(iInternalIndex);
		}
	}

	/**
	 * Returns the internal storage for the specified DataRow Object[].
	 *  
	 * @param pDataRowIndex	the row index.
	 * @return the internal storage for the specified DataRow Object[].
	 * @throws ModelException	if the mem sort and/or filter  fails.
	 */
	protected Object[] getDataRowStorage(int pDataRowIndex) throws ModelException
	{
		initFilterSort();
		return alStorage.get(getInternalRowIndex(pDataRowIndex));
	}
	
	/**
	 * Sets a new Master DataRow, if it changes. Internal function!
	 * 
	 * @param pMasterDataRow	the new master DataRow
	 * @throws ModelException	if the column value couldn't converted 
	 */
	protected void setMasterDataRow(IDataRow pMasterDataRow) throws ModelException
	{
		drMasterDataRow = pMasterDataRow;

		// master row changed, copy master PK to Details FK Columns
		String[] saColumns = dbDataBook.getMasterReference().getColumnNames();
        int[] iaColumnIndicies = new int[saColumns.length];
        IRowDefinition rdRowDefinition = dbDataBook.getRowDefinition();
        
        // cache the column indicies
		for (int j = 0; j < iaColumnIndicies.length; j++)
		{
			iaColumnIndicies[j] = rdRowDefinition.getColumnDefinitionIndex(saColumns[j]);
		}
        
		// copy
		initFilterSort();
    	for (int i = 0; i < alStorage.size(); i++)
    	{
    		for (int j = 0; j < iaColumnIndicies.length; j++)
    		{
        		IDataType ctDataType = rdRowDefinition.getColumnDefinition(iaColumnIndicies[j]).getDataType();
        		alStorage.get(getInternalRowIndex(i))[iaColumnIndicies[j]] = ctDataType.convertAndCheckToTypeClass(drMasterDataRow.getValue(j));		    			
    		}
    	}		
	}
	
	/**
	 * Returns -1. Will/should be overridden in the derived Classes. 
	 * 
	 * @return -1.
	 * @throws ModelException see derived classes. 
	 */
	public int getEstimatedRowCount() throws ModelException
	{		
		return getRowCount();
	}	
	
	/**
	 * Will/should be overridden in the derived Classes to fetch data from the storage. 
	 * 
	 * @param pRowIndex	the row index to use.
	 * @throws ModelException see derived classes. 
	 */
	public void fetchToRow(int pRowIndex) throws ModelException
	{
	}
	
	/**
	 * Returns the row count of this MemDataPage. Internal use only.
	 * 
	 * @return the row count of this MemDataPage. Internal use only. 
	 */
	protected int getRowCountInternal()
	{
		if (iaUsedRows != null)
		{
			return iaUsedRows.length;
		}
		else
		{
			return alStorage.size();
		}
	}		
	
	/**
	 * Returns the specified DataRow without init the mem Filter/Sort and get data from
	 * storage.
	 * 
	 * @param pDataRowIndex		the row index to use.
	 * @return the specified DataRow without init the mem Filter/Sort and get data from
	 * 		   storage.
	 */
	private ChangeableDataRow getDataRowWithoutSort(int pDataRowIndex)
	{		
		return new ChangeableDataRow(dbDataBook.getRowDefinition(),
									 alStorage.get(pDataRowIndex),
							         this,
							         pDataRowIndex);
	}
		
	/**
	 * Returns the internal / real row index of the row in the MemDataPage.
	 * Its necessary because of memory sort and filter functionality. 
	 * 
	 * @param pRowIndex	the external used row index.
	 * @return the internal / real row index of the row in the MemDataPage.
	 */
	private int getInternalRowIndex(int pRowIndex)
	{
		if (iaUsedRows == null)
		{
			return pRowIndex;
		}
		return iaUsedRows[pRowIndex];
	}
	
	
	/**
	 * Sets the internal storage for the specified DataRow Object[].
	 * 
	 * @param pDataRowIndex		the row index.
	 * @param pRow				the ChangeableDataRow to use.
	 * @throws ModelException	if the mem sort and/or filter  fails.
	 */
	protected void setDataRow(int pDataRowIndex, ChangeableDataRow pRow) throws ModelException
	{
		alStorage.set(pDataRowIndex, pRow.getStorage());
		
		// set correct Change for new Storage Object[]
		if (pRow.isDeleting() || pRow.isInserting() || pRow.isUpdating())
		{
			addChange(pDataRowIndex);
		}
		else
		{
			removeChange(pDataRowIndex);
		}
	}	
	
	/**
	 * It shifts all items (row indices) by pShift (-1/+1).
	 *  
	 * @param pIntArray the int[] to shift.
	 * @param pRowIndex	all items with >= pRowIndex will be shifted. 
	 * @param pShift	the value to add/shift
	 */
	private void shiftIndicies(int[] pIntArray, int pRowIndex, int pShift)
	{
		for (int i = 0; i < pIntArray.length; i++)
		{
			int iCurrent = pIntArray[i];
			if (iCurrent >= pRowIndex)
			{
				iCurrent += pShift;
				pIntArray[i] = iCurrent;
			}
		}
	}
	
	/**
	 * Adds a new change in the internal list of changes.
	 * 
	 * @param iRowIndex	the row index where the change was made.
	 */
	private void addChange(int iRowIndex)
	{
		if (ArrayUtil.indexOf(iaChangedRows, iRowIndex) < 0)
		{
			iaChangedRows = ArrayUtil.add(iaChangedRows, iRowIndex);
		}
	}
	
	/**
	 * Remove an existing change by row index.
	 * 
	 * @param iRowIndex	the row index to use.
	 * @throws ModelException	if determining of the row state fails.
	 */
	private void removeChange(int iRowIndex) throws ModelException
	{
		int index = ArrayUtil.indexOf(iaChangedRows, iRowIndex);		
		if (index >= 0)
		{
			ChangeableDataRow cdr = null;
			if (iRowIndex < alStorage.size())
			{
				cdr = getDataRowWithoutSort(iRowIndex);
			}
			if (cdr == null 
					|| !(cdr.isDeleting() || cdr.isDetailChanged() || cdr.isInserting() || cdr.isUpdating()))
			{ 
				iaChangedRows = ArrayUtil.remove(iaChangedRows, index);
			}
		}
	}
	
	/**
	 * Remove an existing change by row index. 
	 * Also if this row is still isInserting(), isUpdating(),isDeleting(), isDetailChanged()!!!
	 * Its used for real delete of a row from the storage.
	 * 
	 * @param iRowIndex	the row index to use.
	 * @throws ModelException	if determining of the row state fails.
	 */
	private void removeChangeAlways(int iRowIndex) throws ModelException
	{
		int index = ArrayUtil.indexOf(iaChangedRows, iRowIndex);
		if (index >= 0)
		{
			iaChangedRows = ArrayUtil.remove(iaChangedRows, index);
		}
	}	
	
	/**
	 * It deletes the row with the given index in the MemDataPage. 
	 * 
	 * @param pDataRowIndex	the row index to use.
	 * @throws ModelException	if the removeChange fails.
	 */
	private void deleteRow(int pDataRowIndex) throws ModelException
	{		
		// remove current row
		alStorage.remove(pDataRowIndex);		
		
		if (iaUsedRows != null)
		{
			iaUsedRows = ArrayUtil.remove(iaUsedRows, ArrayUtil.indexOf(iaUsedRows, pDataRowIndex));
			shiftIndicies(iaUsedRows, pDataRowIndex, -1);
		}

		removeChangeAlways(pDataRowIndex);			
		shiftIndicies(iaChangedRows, pDataRowIndex, -1);					
	}	
	
	/**
	 * Initialize the mem filter and sort.
	 * 
	 * @throws ModelException	if the mem sort and/or filter fails.
	 */
	private void initFilterSort() throws ModelException
	{
		if (iaUsedRows == null && dbDataBook != null)
		{
			if (dbDataBook.isMemFilter())
			{
				iaUsedRows = filter();
			}
			if (dbDataBook.isMemSort())
			{
				iaUsedRows = sort();
			}
		}		
	}
	
	/**
	 * It clears the mem filter and sort in the MemDataPage.  
	 */
	protected void clear()
	{
		iaUsedRows = null;
	}
	
	/**
	 * It filters the MemDataPage and return an int[] with DataRow indices to use for 
	 * the filter result.
	 * 
	 * @return an int[] with DataRow indices to use for the filter result.
	 * @throws ModelException	if the isFulfilled fails.
	 */
	private int[] filter() throws ModelException
	{
		ICondition cFilter = dbDataBook.getFilter();
		
		if (cFilter != null)
		{
			fetchAll();

			int[] iaResult = new int[getRowCountInternal()];
			int iCount = 0;
			for (int i = 0; i < iaResult.length; i++)
			{
				if (cFilter.isFulfilled(getDataRowWithoutSort(i)))
				{
					iaResult[iCount] = i;
					iCount++;
				}
			}
			return ArrayUtil.truncate(iaResult, iCount);
		}
		return null;
	}
	
	/**
	 * It sorts the MemDataPage and return an int[] with DataRow indices to use for 
	 * the sorted result.
	 * 
	 * @return an int[] with DataRow indices to use for the sorted result.
	 * @throws ModelException	if the compareTo fails.
	 */
	private int[] sort() throws ModelException
	{
		SortDefinition sort = dbDataBook.getSort();
		if (sort == null)
		{
			return iaUsedRows;
		}

		fetchAll();

		int[] iaCurrentIndices = getInteralDataRowIndexes();
		IDataRow[] draCurrent = getDataRowsInternBy(iaCurrentIndices);
		
		int[] iaSortedIndices = new DataBookSorter(sort).sort(draCurrent);
		for (int i = 0; i < iaSortedIndices.length; i++)
		{
			iaSortedIndices[i] = iaCurrentIndices[iaSortedIndices[i]];
		}

		return iaSortedIndices;
	}

	/**
	 * Gets the current data row indexes including the filtered rows.
	 * @return the indexes.
	 */
	private int[] getInteralDataRowIndexes()
	{
		int[] iaResult = new int[getRowCountInternal()];
		for (int i = 0; i < iaResult.length; i++)
		{
			iaResult[i] = getInternalRowIndex(i);
		}
		return iaResult;
	}
	
	/**
	 * Gets the array of {@link IDataRow} by indexes. 
	 * @param pIndexes the indexes
	 * @return the array of {@link IDataRow}
	 */
	private IDataRow[] getDataRowsInternBy(int[] pIndexes) 
	{
		IDataRow[] dra = new IDataRow[getRowCountInternal()];
		for (int i = 0; i < dra.length; i++)
		{
			dra[i] = getDataRowWithoutSort(pIndexes[i]);
		}
		return dra;
	}
	
	/**
	 * Dumps this data page as string.
	 * 
	 * @param pPadding line padding
	 * @return the string
	 */
	public String toString(String pPadding)
	{
		StringBuilder sbResult = new StringBuilder();
		
		if (dbDataBook != null)
		{
			sbResult.append(pPadding);
			sbResult.append("DataBook = ");
			sbResult.append(dbDataBook.getName());
			sbResult.append("\n");
		}
		
		try
		{
			if (dbDataBook != null && dbDataBook.getRowDefinition() != null)
			{
				sbResult.append(pPadding);
		        sbResult.append("Changes  = [");
		        
		        for (int i = 0; i < iaChangedRows.length; i++)
		        {
		        	if (i > 0)
		        	{
		        		sbResult.append(", ");
		        	}
		        	sbResult.append(iaChangedRows[i]);
		        }
		        		        
		        sbResult.append("]");

		        // init padding infos
				IRowDefinition rowDef         = dbDataBook.getRowDefinition();
				String[]       sColumns       = rowDef.getColumnNames().clone();
				int[]          iaPadSize      = new int[sColumns.length];
				int[]          iaPadAlignment = new int[sColumns.length];
				
				int iMinusCount = 0;
				
				if (getRowCount() > 0)
				{
					sbResult.append("\n\n");
					sbResult.append(pPadding);
					sbResult.append("[    #][       UID][CHANGES] |");
					
					for (int i = 0, iSize = sColumns.length; i < iSize; i++)
					{
						IDataType dataType = rowDef.getColumnDefinition(sColumns[i]).getDataType();
						
						if (dataType instanceof DataType)
						{
							iaPadSize[i] = ((DataType)dataType).getSize();
							if (iaPadSize[i] > 50)
							{
								switch (dataType.getTypeIdentifier())
								{
									case BigDecimalDataType.TYPE_IDENTIFIER:
										iaPadSize[i] = 15;
										break;
									case TimestampDataType.TYPE_IDENTIFIER:
										iaPadSize[i] = 20;
										break;
									default:
										iaPadSize[i] = 50;
								}
							}
						}
						
						iaPadAlignment[i] = 0; // ==left
		
						if (dataType instanceof BigDecimalDataType)
						{
							iaPadAlignment[i] = 1; // ==right
						}
	
						if (sColumns[i] != null && sColumns[i].length() > iaPadSize[i])
						{
							sColumns[i] = sColumns[i].substring(0, iaPadSize[i]);
						}
						
						String sCol = StringUtil.padRight(sColumns[i], iaPadSize[i]);
						iMinusCount += sCol.length();
						sbResult.append(sCol);
						if (i + 1 < iSize)
						{
							sbResult.append("|");
							iMinusCount++;
						}
					}
					sbResult.append("\n");
					
					sbResult.append(pPadding);
					sbResult.append("-------------------------------");
					
					for (int i = 0; i < iMinusCount; i++)
					{
						sbResult.append('-');
					}
					
			        for (int i = 0, iLen = getRowCount(); i < iLen; i++)
					{
						sbResult.append("\n");
			        	sbResult.append(pPadding);
						sbResult.append("[");
						sbResult.append(StringUtil.padLeft("" + i, 5));
						sbResult.append("]");
			
						int               index = getInternalRowIndex(i);
						Object[]          oRow  = alStorage.get(index);
						ChangeableDataRow cdr   = getDataRowWithoutSort(index);

						sbResult.append("[");
						sbResult.append(StringUtil.padLeft((cdr.getUID() != null ? "" + cdr.getUID() : ""), 10));
						sbResult.append("]");
						
						
						sbResult.append("[");
						if (cdr.isInserting())
						{
							sbResult.append("I");
						}
						if (cdr.isUpdating())
						{
							sbResult.append("U");
						}
						if (cdr.isDeleting())
						{
							sbResult.append("D");
						}
						if (!cdr.isInserting() && !cdr.isUpdating() && !cdr.isDeleting())
						{
							sbResult.append(" ");
						}
						if (cdr.isDetailChanged())
						{
							sbResult.append("DC");
						}
						else
						{
							sbResult.append("  ");
						}
						sbResult.append("    ] |");
		
						for (int j = 0; j < sColumns.length; j++)
						{
							Object value = oRow[j]; 
						    
							if (value == null)
						    {
						    	value = "null";
						    }
							else
							{
								value = StringUtil.toString(value).replace("\n", "\\n");
								
								if (((String)value).length() > iaPadSize[j])
								{
									value = ((String)value).substring(0, iaPadSize[j]);
								}
							}
							
							if (iaPadAlignment[j] == 0)
							{
								sbResult.append(StringUtil.padRight(value, iaPadSize[j]));						
							}
							else
							{
								sbResult.append(StringUtil.padLeft(value, iaPadSize[j]));						
							}
							if (j + 1 < sColumns.length)
							{
								sbResult.append("|");
							}
						}
					}
				}
			}
		}
		catch (ModelException modelException)
		{
			sbResult.append("\n");
			sbResult.append(CommonUtil.dump(modelException, true));
			
	        return sbResult.toString();
		}
        
        return sbResult.toString();
	}	

} 	// MemDataPage
