/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 01.10.2008 - [HM] - creation
 * 28.02.2009 - [JR] - set modifiers from constructor parameter to member [BUGFIX]
 */
package javax.rad.ui.event;

import javax.rad.ui.IComponent;

/**
 * Platform and technology independent event definition.
 * It is designed for use with AWT, Swing, SWT, JSP, JSF, ... .
 * 
 * @author Martin Handsteiner
 * @see    java.awt.AWTEvent
 */
public abstract class UIEvent 
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    
    /**
     * The Shift key modifier constant.
     * It is recommended that SHIFT_DOWN_MASK be used instead.
     */
    public static final int SHIFT_MASK = 1 << 0;

    /**
     * The Control key modifier constant.
     * It is recommended that CTRL_DOWN_MASK be used instead.
     */
    public static final int CTRL_MASK = 1 << 1;

    /**
     * The Meta key modifier constant.
     * It is recommended that META_DOWN_MASK be used instead.
     */
    public static final int META_MASK = 1 << 2;

    /**
     * The Alt key modifier constant.
     * It is recommended that ALT_DOWN_MASK be used instead.
     */
    public static final int ALT_MASK = 1 << 3;

    /** The AltGraph key modifier constant. */
    public static final int ALT_GRAPH_MASK = 1 << 5;

    /**
     * The Mouse Button1 modifier constant.
     * It is recommended that BUTTON1_DOWN_MASK be used instead.
     */
    public static final int BUTTON1_MASK = 1 << 4;

    /**
     * The Mouse Button2 modifier constant.
     * It is recommended that BUTTON2_DOWN_MASK be used instead.
     * Note that BUTTON2_MASK has the same value as ALT_MASK.
     */
    public static final int BUTTON2_MASK = 1 << 3;

    /**
     * The Mouse Button3 modifier constant.
     * Note that BUTTON3_MASK has the same value as META_MASK.
     */
    public static final int BUTTON3_MASK = 1 << 2;


	/** The Source Component of this UIEvent. */
	private IComponent source;
	
	/** The event Id of this event. */
	private int id;
	
    /** The input event's Time stamp in UTC format. The time stamp indicates when the input event was created. */
	private long when;

    /** The state of the modifier mask at the time the input event was fired. */
    private int modifiers;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creates a new instance of <code>UIEvent</code>.
	 * 
	 * @param pSource the Source of this UIEvent.
	 * @param pId     the Id of this UIEvent.
     * @param pWhen   the time the event occurred
     * @param pModifiers represents the modifier keys and mouse buttons down while the event occurred
	 */
	protected UIEvent(IComponent pSource, int pId, long pWhen, int pModifiers)
	{
		checkId(pId);
		
		source    = pSource;
		id        = pId;
		when   	  = pWhen;
		modifiers = pModifiers;
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Checks if the current Instance of UIEvent allows the given id.
	 * 
	 * @param pId the Id of this UIEvent.
	 */
	protected void checkId(int pId)
	{
		throw new IllegalArgumentException("The given id is not allowed for a " + getClass().getName()); 
	}

	/**
     * The object on which the Event initially occurred.
     *
     * @return the object on which the Event initially occurred.
     */
    public IComponent getSource()
    {
    	return source;
    }

	/**
     * The id of this Event.
     *
     * @return the id of this Event.
     */
    public int getId()
    {
    	return id;
    }

    /**
     * Returns the timestamp of when this event occurred.
     *
     * @return the timestamp of when this event occurred.
     */
    public long getWhen() 
    {
        return when;
    }

    /**
     * Returns the modifier mask for this event.
     *
     * @return the modifier mask for this event.
     */
    public int getModifiers() 
    {
        return modifiers;
    }
    
    /**
     * True, if the given modifiers are set.
     * @param pModifier the modifiers to test.
     * @return true, if the given modifiers are set.
     */
    public boolean isModifier(int pModifier)
    {
    	return (modifiers & pModifier) == pModifier;
    }
    
}	// UIEvent
