/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 17.02.2009 - [HM] - creation
 * 03.03.2011 - [JR] - #300: beanType creation changed to be thread safe
 */
package javax.rad.type.bean;

import javax.rad.type.AbstractType;
import javax.rad.type.IType;

import com.sibvisions.util.ArrayUtil;

/**
 * The <code>AbstractBeanType</code> is a bean definition without a bean object/class. It holds
 * the information about property names and definitions
 * 
 * @param <T> the type.
 * 
 * @author Martin Handsteiner
 */
public abstract class AbstractBeanType<T> extends AbstractType<T> 
                                          implements IBeanType<T>
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/** The bean class. */
	protected String className;
	
	/** The cached property names. */
	protected String[] propertyNames;
	
	/** The property definitions. */
	protected transient PropertyDefinition[] propertyDefinitions;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Constructs a new <code>AbstractBeanType</code>
	 * .
	 */
	protected AbstractBeanType()
	{
	}
	
	/**
	 * Constructs a new <code>AbstractBeanType</code> with class name and a property list.
	 * 
	 * @param pClassName the class name.
	 * @param pPropertyNames the bean properties.
	 */
	protected AbstractBeanType(String pClassName, String[] pPropertyNames)
	{
		if (pClassName == null)
		{
			className = pClassName;
		}
		else
		{
			className = pClassName.intern();
		}
		
		if (pPropertyNames == null)
		{
			propertyNames = new String[0];
		}
		else
		{
			propertyNames = pPropertyNames;
			
			for (int i = 0; i < propertyNames.length; i++)
			{
				propertyNames[i] = propertyNames[i].intern();
			}
		}
		
		propertyDefinitions = null;
	}
	
	/**
	 * Constructs a new <code>AbstractBeanType</code> with a class name and {@link PropertyDefinition}s.
	 * 
	 * @param pClassName the class name.
	 * @param pPropertyDefinitions the property definitions.
	 */
	protected AbstractBeanType(String pClassName, PropertyDefinition[] pPropertyDefinitions)
	{
		if (pClassName == null)
		{
			className = pClassName;
		}
		else
		{
			className = pClassName.intern();
		}
		
		if (pPropertyDefinitions == null)
		{
			propertyNames = new String[0];
			
			propertyDefinitions = null;
		}
		else
		{
			propertyNames = new String[pPropertyDefinitions.length];
			
			for (int i = 0; i < propertyNames.length; i++)
			{
				propertyNames[i] = pPropertyDefinitions[i].name;
			}
		}
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	// IBeanType
	
	/**
	 * {@inheritDoc}
	 */
	public String[] getPropertyNames()
	{
		return propertyNames;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public int getPropertyIndex(String pPropertyName)
	{
		for (int i = 0; i < propertyNames.length; i++)
		{
			if (pPropertyName == propertyNames[i])
			{
				return i;
			}
		}
		String interned = pPropertyName.intern();
		if (interned != pPropertyName)
		{
			for (int i = 0; i < propertyNames.length; i++)
			{
				if (interned == propertyNames[i])
				{
					return i;
				}
			}
		}
		return -1;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public int getPropertyCount()
	{
		return propertyNames.length;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public PropertyDefinition getPropertyDefinition(int pIndex)
	{
		if (propertyDefinitions == null)
		{
			propertyDefinitions = createPropertyDefinitions(propertyNames.length);
		}
		
		return propertyDefinitions[pIndex];
	}

	/**
	 * {@inheritDoc}
	 */
	public PropertyDefinition getPropertyDefinition(String pPropertyName)
	{
		int index = getPropertyIndex(pPropertyName);
		
		if (index < 0)
		{
			throw new IllegalArgumentException("The property [" + pPropertyName + "] does not exist!");
		}
		else
		{
			if (propertyDefinitions == null)
			{
				propertyDefinitions = createPropertyDefinitions(propertyNames.length);
			}
			
			return propertyDefinitions[index];
		}
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int hashCode()
	{
		int hashCode = className == null ? 0 : className.hashCode();
		
     	for (int i = 0; i < propertyNames.length; i++)
     	{
		    String value = propertyNames[i];
		    hashCode = 31 * hashCode + value.hashCode();
		}
     	
     	return hashCode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
    public boolean equals(Object pObject)
    {
    	if (pObject == this)
    	{
    	    return true;
    	}
    	else if (pObject instanceof AbstractBeanType)
    	{
    		AbstractBeanType classDef = (AbstractBeanType)pObject;
    		
    		if (className != classDef.className || propertyNames.length != classDef.propertyNames.length)
    		{
    			return false;
    		}
    		else
    		{
				for (int i = 0; i < propertyNames.length; i++)
				{
					if (propertyNames[i] != classDef.propertyNames[i])
					{
						return false;
					}
				}
				return true;
    		}
    	}
    	else
    	{
    		return false;
    	}
    }
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Gets the class name or identifier of the bean class.
	 * This name has to be unique for every different bean class.
	 * 
	 * @return the class name.
	 */
	public String getClassName()
	{
		return className;
	}

	/**
	 * Creates the PropertyDefinitions.
	 * 
	 * @param pPropertyCount the property count
	 * @return the property definitions 
	 */
	protected PropertyDefinition[] createPropertyDefinitions(int pPropertyCount)
	{
		PropertyDefinition[] pdef = new PropertyDefinition[pPropertyCount];
		
		for (int i = 0; i < propertyNames.length; i++)
		{
			pdef[i] = createPropertyDefinition(propertyNames[i]);
		}
		
		return pdef;
	}
	
	/**
	 * Creates the PropertyDefinition.
	 * 
	 * @param pPropertyName the property name.
	 * @return the PropertyDefinition
	 */
	protected PropertyDefinition createPropertyDefinition(String pPropertyName)
	{
		return new PropertyDefinition(pPropertyName, IType.UNKNOWN_TYPE);
	}
	
	/**
	 * Adds a property definition.
	 * @param pPropertyDefinition the property definition.
	 */
	public void addPropertyDefinition(PropertyDefinition pPropertyDefinition)
	{
		if (className != null)
		{
			throw new UnsupportedOperationException("A Property can not be added manually if the IBeanType handles a bean class!");
		}
		
		if (ArrayUtil.indexOfReference(propertyNames, pPropertyDefinition.name) >= 0)
		{
    		throw new IllegalArgumentException("The property " + pPropertyDefinition.name + " is defined more than once!");
		}

		if (propertyDefinitions == null)
		{
			propertyDefinitions = createPropertyDefinitions(propertyNames.length + 1);

			propertyDefinitions[propertyNames.length] = pPropertyDefinition;
		}
		else
		{
			propertyDefinitions = ArrayUtil.add(propertyDefinitions, pPropertyDefinition);
		}

		propertyNames = ArrayUtil.add(propertyNames, pPropertyDefinition.name);
	}
	
	/**
	 * Adds a property definition.
	 * @param pPropertyName the property definition.
	 */
	public void addPropertyDefinition(String pPropertyName)
	{
		if (className != null)
		{
			throw new UnsupportedOperationException("The property " + pPropertyName + " does not exist!");
		}

		if (ArrayUtil.indexOfReference(propertyNames, pPropertyName) >= 0)
		{
    		throw new IllegalArgumentException("The property " + pPropertyName + " is defined more than once!");
		}
		
		pPropertyName = pPropertyName.intern();

		propertyNames = ArrayUtil.add(propertyNames, pPropertyName);
		
		if (propertyDefinitions != null)
		{
			propertyDefinitions = ArrayUtil.add(propertyDefinitions, createPropertyDefinition(pPropertyName));
		}
	}
	
	/**
	 * Removes all property definitions.
	 */
	public void removeAllPropertyDefinitions()
	{
	    propertyNames = new String[0];
	    propertyDefinitions = null;
	}
		
}	// AbstractBeanType
