/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 16.11.2008 - [HM] - creation
 * 08.12.2008 - [JR] - default button functionality implemented
 * 10.12.2008 - [JR] - moved accelerator methods from UIMenuItem
 * 24.10.2012 - [JR] - #604: added constructor
 * 22.09.2014 - [RZ] - #1103: the name is now created from the listeners or action command.
 */
package javax.rad.genui.component;

import javax.rad.genui.UIFactoryManager;
import javax.rad.ui.component.IButton;
import javax.rad.ui.event.IActionListener;

import com.sibvisions.util.type.StringUtil;

/**
 * Platform and technology independent button.
 * It is designed for use with AWT, Swing, SWT, JSP, JSF, ... .
 * 
 * @author Martin Handsteiner
 * 
 */
public class UIButton extends AbstractUIButton<IButton> 
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	  
    /**
     * Creates a new instance of <code>UIButton</code>.
     *
     * @see IButton
     */
	public UIButton()
	{
		super(UIFactoryManager.getFactory().createButton());
	}

    /**
     * Creates a new instance of <code>UIButton</code> with the given
     * button.
     *
     * @param pButton the button
     * @see IButton
     */
	protected UIButton(IButton pButton)
	{
		super(pButton);
	}	
	
    /**
     * Creates a new instance of <code>UIButton</code>.
     *
     * @param pText the label of the button.
     * @see IButton
     */
	public UIButton(String pText)
	{
		this();
		setText(pText);
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Overwritten methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String createComponentName()
	{
		String name = getNameFromActionListeners();
		
		if (!StringUtil.isEmpty(getActionCommand()))
		{
			if (!StringUtil.isEmpty(name))
			{
				name = name + "_" + getActionCommand();
			}
			else
			{
				name = getActionCommand();
			}
		}
		
		if (!StringUtil.isEmpty(name))
		{
			String rootPrefix = "";
			
			if (getRootName() != null)
			{
				rootPrefix = getRootName() + "_";
			}
			
			return incrementNameIfExists(rootPrefix + "B_" + name.toUpperCase(), getExistingNames(), false);
		}
		else
		{
			return super.createComponentName();
		}
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * Gets the name of the method from the registered action listeners. Returns
	 * {@code null} if there is no suitable listener registered.
	 * 
	 * @return the name of method. {@code null} if there is no suitable listener registered.
	 */
	private String getNameFromActionListeners()
	{
		for (IActionListener listener : eventAction().getListeners())
		{
			String listenerString = listener.toString();
			if (listenerString != null && !listenerString.contains("@"))
			{
				int arrowIndex = listenerString.indexOf(" -> ");
				
				if (arrowIndex >= 0)
				{
					listenerString = listenerString.substring(arrowIndex + 4);
				}
				
				// We are now working with something looking like this:
				// public void com.sibvisions.Object.doAction(UIActionEvent pEvent) throws Throwable
				
				// Split it at the paranthese to receive this part:
				// public void com.sibvisions.Object.doAction
				String[] parantheseSplitted = listenerString.split("\\(");
				
				// Split it at the dot and spaces, the last part is the name of the method.
				String[] dotSplitted = parantheseSplitted[0].split("[\\. ]");
				
				String name = dotSplitted[dotSplitted.length - 1];
				
				// Sanity check.
				if (!name.contains(" ") && !name.contains("."))
				{
					return name;
				}
			}
		}
		
		return null;
	}
	
}	// UIButton
