/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 14.11.2008 - [HM] - creation
 * 22.07.2009 - [JR] - getResource: check null [BUGFIX]
 * 24.10.2012 - [JR] - #604: check resource
 * 25.10.2012 - [JR] - removed resource check because it must be possible to wrap UI resources!
 */
package javax.rad.genui;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;

import javax.rad.ui.IResource;

/**
 * Platform and technology independent wrapper for IFactory Resource.
 * 
 * @author Martin Handsteiner
 * 
 * @param <UI> the corresponding UI Resource.
 */
public abstract class UIResource<UI extends IResource> implements IResource
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Class members
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** The IFactory Resource. */
	protected UI uiResource;
	
	/** Properties. */
	private HashMap<String, Object> hmObjects = null; 
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Initialization
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	  
	/**
	 * Constructs a new <code>UIResource</code>.
	 * 
	 * @param pUIResource the IFactory Resource.  
	 */
	protected UIResource(UI pUIResource)
	{
		uiResource = pUIResource;
	}
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// Interface implementation
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	  
	/**
	 * {@inheritDoc}
	 */
	public Object getResource()
	{
		if (uiResource == null)
		{
			return null;
		}
		else
		{
			return uiResource.getResource();
		}
	}

    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Overwritten methods
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean equals(Object pObject)
	{
		if (pObject instanceof IResource) 
		{
			return getResource().equals(((IResource)pObject).getResource());
		}
		else 
		{
		  return false;
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public int hashCode()
	{
		return getResource().hashCode();
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString()
	{
		return getClass().getName() + "[" + String.valueOf(getResource()) + "]";
	}	
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	// User-defined methods
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	  
	/**
	 * Gets the UI resource corresponding with this wrapper object.
	 * 
	 * @return the original resource
	 */
	public UI getUIResource()
	{
		return uiResource;
	}

	/**
	 * Get the specific Object.
	 * 
	 * @param pObjectName the property name.
	 * @return the property value.
	 * @see #getObjectNames()
	 * @see #putObject(String, Object)
	 */
	public Object getObject(String pObjectName)
	{
		if (hmObjects == null)
		{
			return null;
		}
		else
		{
			return hmObjects.get(pObjectName);
		}
	}
	
	/**
	 * Gets a {@link Collection} of all object names that are currently
	 * stored as property on this resource. Returns an empty {@link Collection}
	 * if there are no objects put. The returned {@link Collection} is a copy
	 * of the original collection of names.
	 * 
	 * @return the {@link Collection} of all object names. If there are no
	 *         objects put, returns an empty {@link Collection}.
	 * @see #getObject(String)
	 * @see #putObject(String, Object)
	 */
	public Collection<String> getObjectNames()
	{
		if (hmObjects == null)
		{
			return Collections.emptySet();
		}
		else
		{
			return new ArrayList(hmObjects.keySet());
		}
	}
	
	/**
	 * Puts the specific property.
	 * 
	 * @param pObjectName the property name.
	 * @param pObject the property values.
	 * @return the old pObject.
	 * @see #getObject(String)
	 * @see #getObjectNames()
	 */
	public Object putObject(String pObjectName, Object pObject)
	{
		if (pObject == null)
		{
			if (hmObjects == null)
			{
				return null;
			}
			else
			{
				Object value = hmObjects.remove(pObjectName);
				if (hmObjects.isEmpty())
				{
					hmObjects = null;
				}
				return value;
			}
		}
		else
		{
			if (hmObjects == null)
			{
				hmObjects = new HashMap<String, Object>();
				
			}
			return hmObjects.put(pObjectName, pObject);
		}
	}
	
}	// UIResource
