/*
 * Copyright 2009 SIB Visions GmbH
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 *
 * History
 *
 * 27.11.2008 - [HM] - creation
 * 19.07.2009 - [JR] - setText: repaint [BUGFIX]
 * 21.09.2011 - [JR] - #471: setText now checks empty text
 */
package com.sibvisions.rad.ui.swing.impl.container;

import java.awt.Color;
import java.awt.Font;

import javax.rad.ui.IAlignmentConstants;
import javax.rad.ui.IColor;
import javax.rad.ui.IFont;
import javax.rad.ui.container.IGroupPanel;
import javax.swing.BorderFactory;
import javax.swing.JPanel;
import javax.swing.border.TitledBorder;

import com.sibvisions.rad.ui.swing.impl.SwingComponent;

/**
 * The <code>SwingGroupPanel</code> is the <code>IGroupPanel</code>
 * implementation for swing.
 * 
 * @author Martin Handsteiner
 * @see	JPanel
 * @see IGroupPanel
 */
public class SwingGroupPanel extends SwingComponent<JPanel> 
							 implements IGroupPanel 
{
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Class members
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/** the title border. */
	private TitledBorder titledBorder;
	
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Initialization
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * Creates a new instance of <code>SwingGroupPanel</code>.
	 */
	public SwingGroupPanel()
	{
		super(new JPanel());

		titledBorder = BorderFactory.createTitledBorder("");
		setHorizontalAlignment(ALIGN_LEFT);
		setVerticalAlignment(ALIGN_TOP);
		
		resource.setBorder(titledBorder);
	}

	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Interface implementation
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

	/**
	 * {@inheritDoc}
	 */
	public String getText()
	{
		String text = titledBorder.getTitle();
		if (text == null || text.length() < 2)
		{
			return "";
		}
		else
		{
			return text.substring(1, text.length() - 1);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void setText(String pText)
	{
		if (pText == null || pText.length() == 0)
		{
			titledBorder.setTitle("");
		}
		else
		{
			titledBorder.setTitle(" " + pText + " ");
		}
		
		resource.repaint();
	}

	/**
	 * {@inheritDoc}
	 */
	public void setHorizontalAlignment(int pHorizontalAlignment)
	{
		super.setHorizontalAlignment(pHorizontalAlignment);
		int titleJustification;
		switch (pHorizontalAlignment) 
		{
			case IAlignmentConstants.ALIGN_LEFT:  
				titleJustification = TitledBorder.LEFT;
				break;
			case IAlignmentConstants.ALIGN_RIGHT: 
				titleJustification = TitledBorder.RIGHT;
				break;
			default:
				titleJustification = TitledBorder.CENTER;
				break;
		}

		titledBorder.setTitleJustification(titleJustification);
	}

	/**
	 * {@inheritDoc}
	 */
	public void setVerticalAlignment(int pVerticalAlignment)
	{
		super.setVerticalAlignment(pVerticalAlignment);
		int titlePosition;
		switch (pVerticalAlignment) 
		{
			case IAlignmentConstants.ALIGN_BOTTOM: 
				titlePosition = TitledBorder.BOTTOM;
				break;
			default:
				titlePosition = TitledBorder.TOP;
				break;
		}

		titledBorder.setTitlePosition(titlePosition);
	}

	/**
	 * {@inheritDoc}
	 */
	public void setFont(IFont pFont)
	{
		super.setFont(pFont);
		
		if (pFont == null) 
		{
			titledBorder.setTitleFont(null);
		}
		else
		{
			titledBorder.setTitleFont((Font)pFont.getResource());
		}
		
		resource.repaint();
	}

	/**
	 * {@inheritDoc}
	 */
	public void setForeground(IColor pColor)
	{
		super.setForeground(pColor);
		if (pColor == null)
		{
			titledBorder.setTitleColor(null);
		}
		else
		{
			titledBorder.setTitleColor((Color)pColor.getResource());
		}
		
		resource.repaint();
	}

}	// SwingGroupPanel
